// pstable.cpp 
//
// This program is free software. See the file COPYING for details.
// Author: Mattias Engdegrd, 1997-1999

#include <qpixmap.h>
#include <qpainter.h>

#include "pstable.h"
#include "proc.h"
#include <limits.h>

Pstable::Pstable(QWidget *parent)
	: HeadedTable(parent,
			HTBL_ROW_SELECTION
			| HTBL_ROW_DOUBLE_CLICK
			| HTBL_ROW_CONTEXT_MENU
			| HTBL_HEADING_TOOLTIPS
			| HTBL_HEADING_CONTEXT_MENU
			| HTBL_HEADING_CLICK
			| HTBL_REORDER_COLS),
	leftmostchanged(-1)
{
	connect(this, SIGNAL(selectionChanged(const Svec<int> *)),SLOT(selection_update(const Svec<int> *)));
	connect(this, SIGNAL(titleClicked(int)), SLOT(sortcol_change(int)));
	connect(this, SIGNAL(foldSubTree(int)), SLOT(subtree_folded(int)));
}

// who call this ? : from qps.cpp
void Pstable::setProcview(Procview *pv)
{
	procview = pv;
	set_sortcol();
}

QString Pstable::title(int col)
{
	return procview->cats[col]->name;
}

// ok 
QString Pstable::text(int row, int col)
{
	return procview->cats[col]->string(procview->procs[row]);
}

int Pstable::colWidth(int col)
{
	// this is -1 for variable width fields, htable keeps track of it
	return procview->cats[col]->width();
}

int Pstable::alignment(int col)
{
	//printf("debug:alignment()\n");
	Category *cat = procview->cats[col];
	return cat->alignment();
}

int Pstable::leftGap(int col)
{
	return procview->cats[col]->gap();
}

QString Pstable::tipText(int col)
{
	Category *cat = procview->cats[col];
	QString s(cat->help);
	
	if(cat == procview->sortcat)
		s.append(procview->reversed ? "\n(sorted backwards)" : "\n(sorted)");
	
	if(cat->index == F_STAT)
		s.append("\n(R =running, S =sleeping, T =stopped)");

	return s;
}

int Pstable::rowDepth(int row)
{
	return procview->procs[row]->level;
}

HeadedTable::NodeState Pstable::folded(int row)
{
	Procinfo *p = procview->procs[row];
	return (p->children && p->children->size() > 0)
		? (p->hidekids ? Closed : Open) : Leaf;
}

int Pstable::parentRow(int row)
{
	return procview->parent_rows[row];
}

bool Pstable::lastChild(int row)
{
	return procview->procs[row]->lastchild;
}

// Recompute table widths and determine the leftmost column whose width
// has changed
void Pstable::recompute_table_widths()
{
	//	printf("debug:recompute_table_widths()\n");
	leftmostchanged = INT_MAX;

	for(int col = 0; col < numCols(); col++)
		if(widthChanged(logCol(col)) && leftmostchanged > col)
			leftmostchanged = col;
	if(leftmostchanged < INT_MAX)
		updateTableSize();
	else
		leftmostchanged = -1;
}

// only update after update_button pressed!!
// repaint cells that have been changed from previous generation
void Pstable::repaint_changed()
{
	//printf("debug:repaint_changed()\n");
	int rows = numRows();
	int cols = numCols();
	int left = leftCell(), right = lastColVisible();
	int top = topCell(), bottom = lastRowVisible();
	if(right >= cols) right = cols - 1;
	if(bottom >= rows) bottom = rows - 1;
	int far_right = right;
	if(leftmostchanged != -1 && right >= leftmostchanged)
		right = leftmostchanged - 1;
	for(int c = right + 1; c <= far_right; c++)
		updateHeading(logCol(c));
	int oldrows = procview->old_procs.size();
	for(int r = top; r <= bottom; r++) {
		for(int c = left; c <= right; c++) {
			Category *cat = procview->cats[logCol(c)];
			Procinfo *p = procview->procs[r];
			if(r < oldrows) {
				Procinfo *oldp = procview->old_procs[r];
				if((!treeMode() || c != 0
							|| (!linesEnabled() && p->level == oldp->level))
						&& cat->compare(p, oldp) == 0
						&& p->selected == oldp->selected)
					continue;	// cell is unchanged
			}
			updateCell(r, c);
		}	
		// update all cells that have moved or changed width
		for(int c = right + 1; c <= far_right; c++)
			updateCell(r, c);
	}
	if(leftmostchanged != -1)
		clearRight();
	if(oldrows > rows && top == 0 && bottom == rows - 1)
		clearBelow();
}

// transfer selection from procview to pstable
// (no visible update is done)
void Pstable::transfer_selection()
{
	int rows = procview->procs.size();
	for(int i = 0; i < rows; i++)
		setSelected(i, procview->procs[i]->selected, FALSE);

}

void Pstable::setRows()
{
	setAutoUpdate(FALSE);
	setNumRows(procview->procs.size());
	setAutoUpdate(TRUE);
}

// slot: called when selection changes
void Pstable::selection_update(const Svec<int> *rows)
{
	//	printf("debug:selection_update()\n");
	for(int i = 0; i < rows->	size(); i++) {
		int row = (*rows)[i];
		procview->procs[row]->selected = isSelected(row);
	}
	emit selection_changed();
}

// slot: called when a title is clicked
void Pstable::sortcol_change(int col)
{
	invalidateCache();
	if(col == sortedCol()) {
		procview->reversed = !procview->reversed;
		if(!procview->treeview) {
			// just reverse the lines
			int n = procview->procs.size();
			for(int i = 0; i < n / 2; i++) {
				Procinfo *p = procview->procs[i];
				procview->procs[i] = procview->procs[n - 1 - i];
				procview->procs[n - 1 - i] = p;
			}
		} else
			procview->rebuild();
	} else {
		procview->reversed = FALSE;
		procview->sortcat = procview->cats[col];
		setSortedCol(col);
		procview->rebuild();
	}
	transfer_selection();
	topAndRepaint();
}

// set sorted column of table to procview->sortcol
void Pstable::set_sortcol()
{
	for(int i = 0; i < procview->cats.size(); i++)
		if(procview->cats[i] == procview->sortcat) {
			setSortedCol(i); return;
		}
	setSortedCol(-1);
}

// When a subtree is folded away, selections inside it disappear to prevent
// unexpected behaviour
static void clear_subtree_selections(Procinfo *p)
{
	for(int i = 0; i < p->children->size(); i++) {
		Procinfo *c = (*p->children)[i];
		c->selected = FALSE;
		if(c->children)
			clear_subtree_selections(c);
	}
}

// Slot: called when a subtree is opened or closed
// row = row number of sheet
void Pstable::subtree_folded(int row)
{
					
	Procinfo *p = procview->procs[row];
	//printf("subtree_folded(): row=%d\n",row);							  
	//printf("child_size()=%d\n",p->children->size());
	//printf("size()=%d\n",procview->procs.size()); // num process in table_body		
	p->hidekids = !p->hidekids;
	if(p->hidekids)
		clear_subtree_selections(p);
	Procinfo *nextp = (row < numRows() - 1) ? procview->procs[row + 1] : 0;

	resetWidths();
	invalidateCache();
	procview->rebuild();
	setRows();
	transfer_selection();
	if(!p->hidekids) {
		// Show as much as possible of the opened subtree
		int r = row + 1;
		while(r < numRows() && procview->procs[r] != nextp)
			r++;
		setAutoUpdate(FALSE);
		showRange(row, r - 1);
		setAutoUpdate(TRUE);
	}
	// This is a stopgap solution; it would be better to have htable
	// take care of the hiding of subtrees and repaint only the rows under
	// the line hidden
	repaintAll();

}

// slot: changes table mode
void Pstable::set_mode(bool treemode)
{
	invalidateCache();
	procview->treeview = treemode;
	procview->rebuild();
	setRows();
	transfer_selection();   //very bad solution ? (fasthyun@magicn.com) 
	setTreeMode(treemode);   
	repaintAll();
}

void Pstable::set_initial_mode(bool treemode)
{
	if(treemode != treeMode())
		setTreeMode(treemode);
}

void Pstable::update_customfield()
{
	int i;
	int place;
	// copy ...
	for(i=0;i<procview->cats.size();i++)
		procview->custom_fields[i]=procview->cats[logCol(i)]->index;

	procview->custom_fields[i]=F_END; 
}


// Description : FIELD movement by mouse drag (fasthyun@magicn.com) 
//				 col -> place
void Pstable::moveCol(int col, int place) // col==logcol !!
{
	int i;
//	printf("DEBUG: Pstable::moveCol()  col=%d(phyCol=%d) , place=%d \n",col,physCol(col),place);
	
	// this means "COMMAND" field should be the first field in TreeMode!
	if( treeMode()==true )  
	{
		if(procview->cats[col]->index==F_COMM) place=0; // *** important 
		else 
			if(place==0) place=col;// *** important 

	}

	// this means "COMMAND_LINE" field should be the last field 
	if(procview->cats[place]->index==F_CMDLINE)
		return;
	if(procview->cats[col]->index==F_CMDLINE)
		place=procview->cats.size() - 1;
	
	HeadedTable::moveCol(col, place);
	update_customfield();
}

// ** need new Design, tooo complex... should be  more easier !(by fasthyun@magicn.com) 
//
// Descrition :	draw the content of process table  FROM htable.cpp
// draw body cell: default implementation just draws text in the cell
// This is called after cell background has been painted
void Pstable::drawCellContents(int row, int col, int w, int h, QPainter *p)
{
	QString str;
	QString qstr;

	int 	x;
	Procinfo *pi = procview->procs[row];
	

	//drawCellContents1(row,col,w,h,p,&x, &str);
	if ( treemode ==true && col == 0 )
	{
		// *pi->residentrintf("%d\n",procview->procs.size());
		//
		//
//	if (pi->pcpu > 20) 
//		p->setPen(Qt::red);
		int n;
		if(pi->children)
		{
			//qstr= pi->comm + "(" + QString::number(pi->children->size()) + ")";
			n=pi->children->size();
			if (n>1) 
				qstr= " (" + QString::number(n) + ")";
			
		}
		str=qstr;
		//find PID
		//HeadedTable::drawCellContents(row, col, w, h, p);
		//p->drawText(x, 0, w - x, h, AlignVCenter ,str+ qstr );
	}
	else  	
		str="";        // crazy solution...(by fasthyun@magicn.com) 
	
	str=cachedText2(row, col) + str;
	
	unsigned long sizeM,sizeK;
	int idx=procview->cats[logCol(col)]->index;


	//*************** TESTING !!!!!!!!!!!!!!! (by fasthyun@magicn.com) 
#ifdef LINUX
	if( idx==F_RSS || idx==F_SIZE || idx==F_SHARE || idx==F_SWAP )
#else
	if( idx==F_RSS || idx==F_SIZE || idx==F_SWAP )
#endif
	{
		switch(idx)
		{
			case F_RSS:
				sizeK = pi->resident; 
				sizeM = pi->resident /1024; 
				break; //RSS
			case F_SIZE:			
				sizeK = pi->size; 
				sizeM = pi->size /1024; 
				break ;//SIZE 
#ifdef LINUX
			case F_SHARE: 
				sizeK = pi->share; 
				sizeM= pi->share /1024; break;//SHARE
#endif
			case F_SWAP: 
				sizeK = pi->size - pi->resident; //SWAP
				sizeM = sizeK /1024; break;
		}
		
		if ( sizeM > 0 ) 
			str= QString::number(sizeM) + "M" ;
		else
			str= QString::number(sizeK) + "K" ;

		if (sizeK == 0) 
			str = "0";

	}

	HeadedTable::drawCellContents2(row,col,w,h,p,&x, &str);
}

