/****************************************************************************
**
** Copyright (C) 1992-2005 Trolltech AS. All rights reserved.
**
** This file is part of the widgets module of the Qt Toolkit.
**
** This file may be distributed under the terms of the Q Public License
** as defined by Trolltech AS of Norway and appearing in the file
** LICENSE.QPL included in the packaging of this file.
**
** This file may be distributed and/or modified under the terms of the
** GNU General Public License version 2 as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL included in the
** packaging of this file.
**
** See http://www.trolltech.com/pricing.html or email sales@trolltech.com for
**   information about Qt Commercial License Agreements.
** See http://www.trolltech.com/qpl/ for QPL licensing information.
** See http://www.trolltech.com/gpl/ for GPL licensing information.
**
** Contact info@trolltech.com if any conditions of this licensing are
** not clear to you.
**
** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
**
****************************************************************************/

#include "qtoolbarhandle_p.h"

#include <qstyle.h>
#include <qstyleoption.h>
#include <qtoolbar.h>
#include <qapplication.h>
#include <qpainter.h>
#include <qevent.h>

#include <private/qmainwindowlayout_p.h>

static QStyleOption getStyleOption(const QToolBarHandle *tbh)
{
    QStyleOption opt;
    opt.init(tbh);
    if (tbh->orientation() == Qt::Horizontal)
	opt.state |= QStyle::State_Horizontal;
    return opt;
}

QToolBarHandle::QToolBarHandle(QToolBar *parent)
    : QWidget(parent), orient(parent->orientation()), state(0)
{
    setCursor(Qt::SizeAllCursor);
}

void QToolBarHandle::setOrientation(Qt::Orientation orientation)
{
    orient = orientation;

    if (orientation == Qt::Horizontal) {
        setSizePolicy(QSizePolicy::Minimum, QSizePolicy::Expanding);
    } else {
        setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Minimum);
    }

    // if we're dragging - swap the offset coords around as well
    if (state) {
	QPoint p = state->offset;
	state->offset = QPoint(p.y(), p.x());
    }

    update();
}

Qt::Orientation QToolBarHandle::orientation() const
{ return orient; }

QSize QToolBarHandle::sizeHint() const
{
    QStyleOption opt = getStyleOption(this);
    const int extent = style()->pixelMetric(QStyle::PM_ToolBarHandleExtent, &opt, this);
    return QSize(extent, extent);
}

void QToolBarHandle::paintEvent(QPaintEvent *)
{
    QPainter p(this);
    QStyleOption opt = getStyleOption(this);
    style()->drawPrimitive(QStyle::PE_IndicatorToolBarHandle, &opt, &p, this);
}

void QToolBarHandle::mousePressEvent(QMouseEvent *event)
{
    if (event->button() != Qt::LeftButton)
        return;

    Q_ASSERT(parentWidget());
    Q_ASSERT(!state);
    state = new DragState;
    state->offset = mapTo(parentWidget(), event->pos());
    if (orientation() == Qt::Horizontal) {
	state->offset = QStyle::visualPos(QApplication::layoutDirection(),
					  parentWidget()->rect(), state->offset);
    }
}

void QToolBarHandle::mouseReleaseEvent(QMouseEvent *event)
{
    if (event->button() != Qt::LeftButton)
        return;
    delete state;
    state = 0;
}

void QToolBarHandle::mouseMoveEvent(QMouseEvent *event)
{
    if (!state)
        return;

    QToolBar *toolBar = qobject_cast<QToolBar *>(parentWidget());
    Q_ASSERT_X(toolBar != 0, "QToolBar", "internal error");
    QMainWindow *mainWindow = qobject_cast<QMainWindow *>(toolBar->parentWidget());
    Q_ASSERT_X(mainWindow != 0, "QMainWindow", "internal error");
    QMainWindowLayout *layout = qobject_cast<QMainWindowLayout *>(mainWindow->layout());
    Q_ASSERT_X(layout != 0, "QMainWindow", "internal error");

    QPoint p = toolBar->mapFromGlobal(event->globalPos());
    if (orientation() == Qt::Horizontal)
	p = QStyle::visualPos(QApplication::layoutDirection(), toolBar->rect(), p);
    p -= state->offset;

    // offset is measured from the widget origin
    if (orientation() == Qt::Vertical)
        p.setX(state->offset.x() + p.x());
    else
        p.setY(state->offset.y() + p.y());

    // re-position toolbar
    layout->dropToolBar(toolBar, event->globalPos(), p);
}
