/****************************************************************************
**
** Copyright (C) 1992-2005 Trolltech AS. All rights reserved.
**
** This file is part of the linguist application of the Qt Toolkit.
**
** This file may be distributed under the terms of the Q Public License
** as defined by Trolltech AS of Norway and appearing in the file
** LICENSE.QPL included in the packaging of this file.
**
** This file may be distributed and/or modified under the terms of the
** GNU General Public License version 2 as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL included in the
** packaging of this file.
**
** See http://www.trolltech.com/pricing.html or email sales@trolltech.com for
**   information about Qt Commercial License Agreements.
** See http://www.trolltech.com/qpl/ for QPL licensing information.
** See http://www.trolltech.com/gpl/ for GPL licensing information.
**
** Contact info@trolltech.com if any conditions of this licensing are
** not clear to you.
**
** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
**
****************************************************************************/

#include "phrasemodel.h"
#include <QtAlgorithms>

static Qt::SortOrder sSortOrder = Qt::AscendingOrder;
static int sSortColumn = 1;

void PhraseModel::removePhrases()
{
    int r = plist.count();
    if (r > 0) {
        plist.clear();
        reset();
    }
}

Phrase PhraseModel::phrase(const QModelIndex &index) const
{
    return plist.at(index.row());
}

void PhraseModel::setPhrase(const QModelIndex &indx, Phrase ph)
{
    int r = indx.row();

    plist[r] = ph;

    // update item in view
    QModelIndex si = QAbstractTableModel::index(r, 0);
    QModelIndex ei = QAbstractTableModel::index(r, 2);
    emit dataChanged(si, ei);
}

QModelIndex PhraseModel::addPhrase(Phrase p)
{
    int r = plist.count();

    plist.append(p);

    // update phrases as we add them
    beginInsertRows(QModelIndex(), r, r);
    QModelIndex i = QAbstractTableModel::index(r, 0);
    endInsertRows();
    return i;
}

void PhraseModel::removePhrase(const QModelIndex &index)
{
    int r = index.row();
    beginRemoveRows(QModelIndex(), r, r);
    plist.removeAt(r);
    endRemoveRows();
}

bool PhraseModel::sortParameters(Qt::SortOrder &so, int &sc) const
{
    if (sortColumn == -1)
        return false;

    so = sortOrder;
    sc = sortColumn;

    return true;
}

void PhraseModel::resort()
{
    if (sortColumn == -1)
        return;

    sort(sortColumn, sortOrder);
}

QModelIndex PhraseModel::index(const Phrase phr) const
{
    int row;
    if ((row = plist.indexOf(phr)) == -1)
        return QModelIndex();

    return QAbstractTableModel::index(row,0);
}

int PhraseModel::rowCount(const QModelIndex &) const
{
    return plist.count();
}

int PhraseModel::columnCount(const QModelIndex &) const
{
    return 3;
}

QVariant PhraseModel::headerData(int section, Qt::Orientation orientation, int role) const
{
    if ((role == Qt::DisplayRole) && (orientation == Qt::Horizontal)) {
        switch(section) {
        case 0:
            return tr("Source phrase");
        case 1:
            return tr("Translation");
        case 2:
            return tr("Definition");
        }
    }

    return QVariant();
}

QVariant PhraseModel::data(const QModelIndex &index, int role) const
{
    int row = index.row();
    int column = index.column();

    if (row >= plist.count() || !index.isValid())
        return QVariant();

    Phrase phrase = plist.at(row);

    if (role == Qt::DisplayRole) {
        switch(column) {
        case 0: // source phrase
            return phrase.source().simplified();
        case 1: // translation
            return phrase.target().simplified();
        case 2: // definition
            return phrase.definition();
        }
    }

    return QVariant();
}

void PhraseModel::sort(int column, Qt::SortOrder order)
{
    if (plist.count() <= 0)
        return;

    sortOrder = sSortOrder = order;
    sortColumn = sSortColumn = column;

    qSort(plist.begin(), plist.end(), PhraseModel::compare);
    emit dataChanged(QAbstractTableModel::index(0,0),
        QAbstractTableModel::index(plist.count()-1, 2));
}

bool PhraseModel::compare(const Phrase &left, const Phrase &right)
{
    int res;
    switch (sSortColumn) {
    case 0:
        res = QString::localeAwareCompare(left.source().remove('&'),
            right.source().remove('&'));
        if ((sSortOrder == Qt::AscendingOrder) ? (res < 0) : !(res < 0))
            return true;
        break;
    case 1:
        res = QString::localeAwareCompare(left.target().remove('&'),
            right.target().remove('&'));
        if ((sSortOrder == Qt::AscendingOrder) ? (res < 0) : !(res < 0))
            return true;
        break;
    case 2:
        // handle the shortcuts when sorting
        if ((left.shortcut() != -1) && (right.shortcut() == -1))
            return (sSortOrder == Qt::AscendingOrder);
        else if ((left.shortcut() == -1) && (right.shortcut() != -1))
            return (sSortOrder != Qt::AscendingOrder);
        res = QString::localeAwareCompare(left.definition(), right.definition());
        if ((sSortOrder == Qt::AscendingOrder) ? (res < 0) : !(res < 0))
            return true;
        break;
    }

    return false;
}
