/****************************************************************************
**
** Copyright (C) 2012 Digia Plc and/or its subsidiary(-ies).
** Contact: http://www.qt-project.org/legal
**
** This file is part of the QtLocation module of the Qt Toolkit.
**
** $QT_BEGIN_LICENSE:LGPL$
** Commercial License Usage
** Licensees holding valid commercial Qt licenses may use this file in
** accordance with the commercial license agreement provided with the
** Software or, alternatively, in accordance with the terms contained in
** a written agreement between you and Digia.  For licensing terms and
** conditions see http://qt.digia.com/licensing.  For further information
** use the contact form at http://qt.digia.com/contact-us.
**
** GNU Lesser General Public License Usage
** Alternatively, this file may be used under the terms of the GNU Lesser
** General Public License version 2.1 as published by the Free Software
** Foundation and appearing in the file LICENSE.LGPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU Lesser General Public License version 2.1 requirements
** will be met: http://www.gnu.org/licenses/old-licenses/lgpl-2.1.html.
**
** In addition, as a special exception, Digia gives you certain additional
** rights.  These rights are described in the Digia Qt LGPL Exception
** version 1.1, included in the file LGPL_EXCEPTION.txt in this package.
**
** GNU General Public License Usage
** Alternatively, this file may be used under the terms of the GNU
** General Public License version 3.0 as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU General Public License version 3.0 requirements will be
** met: http://www.gnu.org/copyleft/gpl.html.
**
**
** $QT_END_LICENSE$
**
****************************************************************************/

#include "qgeorectangle.h"
#include "qgeorectangle_p.h"

#include "qgeocoordinate.h"
#include "qnumeric.h"

QT_BEGIN_NAMESPACE

/*!
    \class QGeoRectangle
    \inmodule QtLocation
    \ingroup QtLocation-positioning
    \since Qt Location 5.0

    \brief The QGeoRectangle class defines a rectangular geographic area.

    The rectangle is defined in terms of a QGeoCoordinate which specifies the
    top left coordinate of the rectangle and a QGeoCoordinate which specifies
    the bottom right coordinate of the rectangle.

    A geo rectangle is considered invalid if the top left or bottom right
    coordinates are invalid or if the top left coordinate is south of the
    bottom right coordinate.

    Geo rectangles can never cross the poles.

    Several methods behave as though the geo rectangle is defined in terms of a
    center coordinate, the width of the geo rectangle in degrees and the height
    of the geo rectangle in degrees.

    If the height or center of a geo rectangle is adjusted such that it would
    cross one of the poles the height is modified such that the geo rectangle
    touches but does not cross the pole and that the center coordinate is still
    in the center of the geo rectangle.
*/

#ifdef Q_NO_USING_KEYWORD
/*!
    \fn bool QGeoRectangle::operator==(const QGeoShape &other) const

    Returns true if \a other is equivalent to this geo rectangle; otherwise returns false.
*/

/*!
    \fn bool QGeoRectangle::operator!=(const QGeoShape &other) const

    Returns true if \a other is not equivalent to this geo rectangle; otherwise returns false.
*/

/*!
    \fn bool QGeoRectangle::contains(const QGeoCoordinate &coordinate) const

    Returns true if \a coordinate is within this geo rectangle; otherwise returns false.
*/
#endif

inline QGeoRectanglePrivate *QGeoRectangle::d_func()
{
    return static_cast<QGeoRectanglePrivate *>(d_ptr.data());
}

inline const QGeoRectanglePrivate *QGeoRectangle::d_func() const
{
    return static_cast<const QGeoRectanglePrivate *>(d_ptr.constData());
}

/*!
    Constructs a new, invalid geo rectangle.
*/
QGeoRectangle::QGeoRectangle()
:   QGeoShape(new QGeoRectanglePrivate)
{
}

/*!
    Constructs a new geo rectangle centered at \a center with a
    width in degrees of \a degreesWidth and a height in degrees of \a degreesHeight.

    If \a degreesHeight would take the geo rectangle beyond one of the poles,
    the height of the geo rectangle will be truncated such that the geo rectangle
    only extends up to the pole. The center of the geo rectangle will be
    unchanged, and the height will be adjusted such that the center point is at
    the center of the truncated geo rectangle.
*/
QGeoRectangle::QGeoRectangle(const QGeoCoordinate &center, double degreesWidth, double degreesHeight)
{
    d_ptr = new QGeoRectanglePrivate(center, center);
    setWidth(degreesWidth);
    setHeight(degreesHeight);
}

/*!
    Constructs a new geo rectangle with a top left coordinate \a topLeft and a bottom right
    coordinate \a bottomRight.
*/
QGeoRectangle::QGeoRectangle(const QGeoCoordinate &topLeft, const QGeoCoordinate &bottomRight)
{
    d_ptr = new QGeoRectanglePrivate(topLeft, bottomRight);
}

/*!
    Constructs a geo rectangle from the contents of \a other.
*/
QGeoRectangle::QGeoRectangle(const QGeoRectangle &other)
:   QGeoShape(other)
{
}

/*!
    Constructs a geo rectangle from the contents of \a other.
*/
QGeoRectangle::QGeoRectangle(const QGeoShape &other)
:   QGeoShape(other)
{
    if (type() != QGeoShape::RectangleType)
        d_ptr = new QGeoRectanglePrivate;
}

/*!
    Destroys this geo rectangle.
*/
QGeoRectangle::~QGeoRectangle()
{
}

/*!
    Assigns \a other to this geo rectangle and returns a reference to this geo rectangle.
*/
QGeoRectangle &QGeoRectangle::operator=(const QGeoRectangle &other)
{
    QGeoShape::operator=(other);
    return *this;
}

/*!
    Returns whether this geo rectangle is equal to \a other.
*/
bool QGeoRectangle::operator==(const QGeoRectangle &other) const
{
    Q_D(const QGeoRectangle);

    return *d == *other.d_func();
}

/*!
    Returns whether this geo rectangle is not equal to \a other.
*/
bool QGeoRectangle::operator!=(const QGeoRectangle &other) const
{
    Q_D(const QGeoRectangle);

    return !(*d == *other.d_func());
}

bool QGeoRectanglePrivate::isValid() const
{
    return topLeft.isValid() && bottomRight.isValid() &&
           topLeft.latitude() >= bottomRight.latitude();
}

bool QGeoRectanglePrivate::isEmpty() const
{
    if (!isValid())
        return true;

    return topLeft.latitude() == bottomRight.latitude() ||
           topLeft.longitude() == bottomRight.longitude();
}

/*!
    Sets the top left coordinate of this geo rectangle to \a topLeft.
*/
void QGeoRectangle::setTopLeft(const QGeoCoordinate &topLeft)
{
    Q_D(QGeoRectangle);

    d->topLeft = topLeft;
}

/*!
    Returns the top left coordinate of this geo rectangle.
*/
QGeoCoordinate QGeoRectangle::topLeft() const
{
    Q_D(const QGeoRectangle);

    return d->topLeft;
}

/*!
    Sets the top right coordinate of this geo rectangle to \a topRight.
*/
void QGeoRectangle::setTopRight(const QGeoCoordinate &topRight)
{
    Q_D(QGeoRectangle);

    d->topLeft.setLatitude(topRight.latitude());
    d->bottomRight.setLongitude(topRight.longitude());
}

/*!
    Returns the top right coordinate of this geo rectangle.
*/
QGeoCoordinate QGeoRectangle::topRight() const
{
    // TODO remove?
    if (!isValid())
        return QGeoCoordinate();

    Q_D(const QGeoRectangle);

    return QGeoCoordinate(d->topLeft.latitude(), d->bottomRight.longitude());
}

/*!
    Sets the bottom left coordinate of this geo rectangle to \a bottomLeft.
*/
void QGeoRectangle::setBottomLeft(const QGeoCoordinate &bottomLeft)
{
    Q_D(QGeoRectangle);

    d->bottomRight.setLatitude(bottomLeft.latitude());
    d->topLeft.setLongitude(bottomLeft.longitude());
}

/*!
    Returns the bottom left coordinate of this geo rectangle.
*/
QGeoCoordinate QGeoRectangle::bottomLeft() const
{
    // TODO remove?
    if (!isValid())
        return QGeoCoordinate();

    Q_D(const QGeoRectangle);

    return QGeoCoordinate(d->bottomRight.latitude(), d->topLeft.longitude());
}

/*!
    Sets the bottom right coordinate of this geo rectangle to \a bottomRight.
*/
void QGeoRectangle::setBottomRight(const QGeoCoordinate &bottomRight)
{
    Q_D(QGeoRectangle);

    d->bottomRight = bottomRight;
}

/*!
    Returns the bottom right coordinate of this geo rectangle.
*/
QGeoCoordinate QGeoRectangle::bottomRight() const
{
    Q_D(const QGeoRectangle);

    return d->bottomRight;
}

/*!
    Sets the center of this geo rectangle to \a center.

    If this causes the geo rectangle to cross on of the poles the height of the
    geo rectangle will be truncated such that the geo rectangle only extends up
    to the pole. The center of the geo rectangle will be unchanged, and the
    height will be adjusted such that the center point is at the center of the
    truncated geo rectangle.

*/
void QGeoRectangle::setCenter(const QGeoCoordinate &center)
{
    Q_D(QGeoRectangle);

    if (!isValid()) {
        d->topLeft = center;
        d->bottomRight = center;
        return;
    }
    double width = this->width();
    double height = this->height();

    double tlLat = center.latitude() + height / 2.0;
    double tlLon = center.longitude() - width / 2.0;
    double brLat = center.latitude() - height / 2.0;
    double brLon = center.longitude() + width / 2.0;

    if (tlLon < -180.0)
        tlLon += 360.0;
    if (tlLon > 180.0)
        tlLon -= 360.0;

    if (brLon < -180.0)
        brLon += 360.0;
    if (brLon > 180.0)
        brLon -= 360.0;

    if (tlLat > 90.0) {
        brLat = 2 * center.latitude() - 90.0;
        tlLat = 90.0;
    }

    if (tlLat < -90.0) {
        brLat = -90.0;
        tlLat = -90.0;
    }

    if (brLat > 90.0) {
        tlLat = 90.0;
        brLat = 90.0;
    }

    if (brLat < -90.0) {
        tlLat = 2 * center.latitude() + 90.0;
        brLat = -90.0;
    }

    if (width == 360.0) {
        tlLon = -180.0;
        brLon = 180.0;
    }

    d->topLeft = QGeoCoordinate(tlLat, tlLon);
    d->bottomRight = QGeoCoordinate(brLat, brLon);
}

/*!
    Returns the center of this geo rectangle.
*/
QGeoCoordinate QGeoRectangle::center() const
{
    if (!isValid())
        return QGeoCoordinate();

    Q_D(const QGeoRectangle);

    double cLat = (d->topLeft.latitude() + d->bottomRight.latitude()) / 2.0;

    double cLon = (d->bottomRight.longitude() + d->topLeft.longitude()) / 2.0;
    if (d->topLeft.longitude() > d->bottomRight.longitude()) {
        cLon = cLon - 180.0;
    }

    if (cLon < -180.0)
        cLon += 360.0;
    if (cLon > 180.0)
        cLon -= 360.0;

    return QGeoCoordinate(cLat, cLon);
}

/*!
    Sets the width of this geo rectangle in degrees to \a degreesWidth.

    If \a degreesWidth is less than 0.0 or if this geo rectangle is invalid this
    function does nothing.  To set up the values of an invalid
    geo rectangle based on the center, width and height you should use
    setCenter() first in order to make the geo rectangle valid.

    If \a degreesWidth is greater than 360.0 then 360.0 is used as the width,
    the leftmost longitude of the geo rectangle is set to -180.0 degrees and the
    rightmost longitude of the geo rectangle is set to 180.0 degrees.
*/
void QGeoRectangle::setWidth(double degreesWidth)
{
    if (!isValid())
        return;

    if (degreesWidth < 0.0)
        return;

    Q_D(QGeoRectangle);

    if (degreesWidth >= 360.0) {
        d->topLeft.setLongitude(-180.0);
        d->bottomRight.setLongitude(180.0);
        return;
    }

    double tlLat = d->topLeft.latitude();
    double brLat = d->bottomRight.latitude();

    QGeoCoordinate c = center();

    double tlLon = c.longitude() - degreesWidth / 2.0;

    if (tlLon < -180.0)
        tlLon += 360.0;
    if (tlLon > 180.0)
        tlLon -= 360.0;

    double brLon = c.longitude() + degreesWidth / 2.0;

    if (brLon < -180.0)
        brLon += 360.0;
    if (brLon > 180.0)
        brLon -= 360.0;

    d->topLeft = QGeoCoordinate(tlLat, tlLon);
    d->bottomRight = QGeoCoordinate(brLat, brLon);
}

/*!
    Returns the width of this geo rectangle in degrees.

    The return value is undefined if this geo rectangle is invalid.
*/
double QGeoRectangle::width() const
{
    if (!isValid())
        return qQNaN();

    Q_D(const QGeoRectangle);

    double result = d->bottomRight.longitude() - d->topLeft.longitude();
    if (result < 0.0)
        result += 360.0;
    if (result > 360.0)
        result -= 360.0;

    return result;
}

/*!
    Sets the height of this geo rectangle in degrees to \a degreesHeight.

    If \a degreesHeight is less than 0.0 or if this geo rectangle is invalid
    this function does nothing. To set up the values of an invalid
    geo rectangle based on the center, width and height you should use
    setCenter() first in order to make the geo rectangle valid.

    If the change in height would cause the geo rectangle to cross a pole
    the height is adjusted such that the geo rectangle only touches the pole.

    This change is done such that the center coordinate is still at the
    center of the geo rectangle, which may result in a geo rectangle with
    a smaller height than might otherwise be expected.

    If \a degreesHeight is greater than 180.0 then 180.0 is used as the height.
*/
void QGeoRectangle::setHeight(double degreesHeight)
{
    if (!isValid())
        return;

    if (degreesHeight < 0.0)
        return;

    if (degreesHeight >= 180.0) {
        degreesHeight = 180.0;
    }

    Q_D(QGeoRectangle);

    double tlLon = d->topLeft.longitude();
    double brLon = d->bottomRight.longitude();

    QGeoCoordinate c = center();

    double tlLat = c.latitude() + degreesHeight / 2.0;
    double brLat = c.latitude() - degreesHeight / 2.0;

    if (tlLat > 90.0) {
        brLat = 2* c.latitude() - 90.0;
        tlLat = 90.0;
    }

    if (tlLat < -90.0) {
        brLat = -90.0;
        tlLat = -90.0;
    }

    if (brLat > 90.0) {
        tlLat = 90.0;
        brLat = 90.0;
    }

    if (brLat < -90.0) {
        tlLat = 2 * c.latitude() + 90.0;
        brLat = -90.0;
    }

    d->topLeft = QGeoCoordinate(tlLat, tlLon);
    d->bottomRight = QGeoCoordinate(brLat, brLon);
}

/*!
    Returns the height of this geo rectangle in degrees.

    The return value is undefined if this geo rectangle is invalid.
*/
double QGeoRectangle::height() const
{
    if (!isValid())
        return qQNaN();

    Q_D(const QGeoRectangle);

    double result = d->topLeft.latitude() - d->bottomRight.latitude();
    if (result < 0.0)
        result = qQNaN();
    return result;
}

bool QGeoRectanglePrivate::contains(const QGeoCoordinate &coordinate) const
{
    if (!isValid() || !coordinate.isValid())
        return false;

    double left = topLeft.longitude();
    double right = bottomRight.longitude();
    double top = topLeft.latitude();
    double bottom = bottomRight.latitude();

    double lon = coordinate.longitude();
    double lat = coordinate.latitude();

    if (lat > top)
        return false;
    if (lat < bottom)
        return false;

    if ((lat == 90.0) && (top == 90.0))
        return true;

    if ((lat == -90.0) && (bottom == -90.0))
        return true;

    if (left <= right) {
        if ((lon < left) || (lon > right))
            return false;
    } else {
        if ((lon < left) && (lon > right))
            return false;
    }

    return true;
}

/*!
    Returns whether the geo rectangle \a rectangle is contained within this
    geo rectangle.
*/
bool QGeoRectangle::contains(const QGeoRectangle &rectangle) const
{
    Q_D(const QGeoRectangle);

    return (d->contains(rectangle.topLeft())
            && d->contains(rectangle.topRight())
            && d->contains(rectangle.bottomLeft())
            && d->contains(rectangle.bottomRight()));
}

/*!
    Returns whether the geo rectangle \a rectangle intersects this geo rectangle.

    If the top or bottom edges of both geo rectangles are at one of the poles
    the geo rectangles are considered to be intersecting, since the longitude
    is irrelevant when the edges are at the pole.
*/
bool QGeoRectangle::intersects(const QGeoRectangle &rectangle) const
{
    Q_D(const QGeoRectangle);

    double left1 = d->topLeft.longitude();
    double right1 = d->bottomRight.longitude();
    double top1 = d->topLeft.latitude();
    double bottom1 = d->bottomRight.latitude();

    double left2 = rectangle.d_func()->topLeft.longitude();
    double right2 = rectangle.d_func()->bottomRight.longitude();
    double top2 = rectangle.d_func()->topLeft.latitude();
    double bottom2 = rectangle.d_func()->bottomRight.latitude();

    if (top1 < bottom2)
        return false;

    if (bottom1 > top2)
        return false;

    if ((top1 == 90.0) && (top1 == top2))
        return true;

    if ((bottom1 == -90.0) && (bottom1 == bottom2))
        return true;

    if (left1 < right1) {
        if (left2 < right2) {
            if ((left1 > right2) || (right1 < left2))
                return false;
        } else {
            if ((left1 > right2) && (right1 < left2))
                return false;
        }
    } else {
        if (left2 < right2) {
            if ((left2 > right1) && (right2 < left1))
                return false;
        } else {
            // if both wrap then they have to intersect
        }
    }

    return true;
}

/*!
    Translates this geo rectangle by \a degreesLatitude northwards and \a
    degreesLongitude eastwards.

    Negative values of \a degreesLatitude and \a degreesLongitude correspond to
    southward and westward translation respectively.

    If the translation would have caused the geo rectangle to cross a pole the
    geo rectangle will be translated until the top or bottom edge of the geo rectangle
    touches the pole but not further.
*/
void QGeoRectangle::translate(double degreesLatitude, double degreesLongitude)
{
    // TODO handle dlat, dlon larger than 360 degrees

    Q_D(QGeoRectangle);

    double tlLat = d->topLeft.latitude();
    double tlLon = d->topLeft.longitude();
    double brLat = d->bottomRight.latitude();
    double brLon = d->bottomRight.longitude();

    if ((tlLat != 90.0) || (brLat != -90.0)) {
        tlLat += degreesLatitude;
        brLat += degreesLatitude;
    }

    if ( (tlLon != -180.0) || (brLon != 180.0) ) {
        tlLon += degreesLongitude;
        brLon += degreesLongitude;
    }

    if (tlLon < -180.0)
        tlLon += 360.0;
    if (tlLon > 180.0)
        tlLon -= 360.0;

    if (brLon < -180.0)
        brLon += 360.0;
    if (brLon > 180.0)
        brLon -= 360.0;

    if (tlLat > 90.0)
        tlLat = 90.0;

    if (tlLat < -90.0)
        tlLat = -90.0;

    if (brLat > 90.0)
        brLat = 90.0;

    if (brLat < -90.0)
        brLat = -90.0;

    d->topLeft = QGeoCoordinate(tlLat, tlLon);
    d->bottomRight = QGeoCoordinate(brLat, brLon);
}

/*!
    Returns a copy of this geo rectangle translated by \a degreesLatitude northwards and \a
    degreesLongitude eastwards.

    Negative values of \a degreesLatitude and \a degreesLongitude correspond to
    southward and westward translation respectively.

    \sa translate()
*/
QGeoRectangle QGeoRectangle::translated(double degreesLatitude, double degreesLongitude) const
{
    QGeoRectangle result(*this);
    result.translate(degreesLatitude, degreesLongitude);
    return result;
}

/*!
    Returns the smallest geo rectangle which contains both this geo rectangle and \a rectangle.

    If the centers of the two geo rectangles are separated by exactly 180.0 degrees then the
    width is set to 360.0 degrees with the leftmost longitude set to -180.0 degrees and the
    rightmost longitude set to 180.0 degrees.  This is done to ensure that the result is
    independent of the order of the operands.

*/
QGeoRectangle QGeoRectangle::united(const QGeoRectangle &rectangle) const
{
    QGeoRectangle result(*this);
    result |= rectangle;
    return result;
}

/*!
    \fn QGeoRectangle QGeoRectangle::operator|(const QGeoRectangle &rectangle) const

    Returns the smallest geo rectangle which contains both this geo rectangle and \a rectangle.

    If the centers of the two geo rectangles are separated by exactly 180.0 degrees then the
    width is set to 360.0 degrees with the leftmost longitude set to -180.0 degrees and the
    rightmost longitude set to 180.0 degrees.  This is done to ensure that the result is
    independent of the order of the operands.

*/

/*!
    Returns the smallest geo rectangle which contains both this geo rectangle and \a rectangle.

    If the centers of the two geo rectangles are separated by exactly 180.0 degrees then the
    width is set to 360.0 degrees with the leftmost longitude set to -180.0 degrees and the
    rightmost longitude set to 180.0 degrees.  This is done to ensure that the result is
    independent of the order of the operands.

*/
QGeoRectangle &QGeoRectangle::operator|=(const QGeoRectangle &rectangle)
{
    // If non-intersecting goes for most narrow box

    Q_D(QGeoRectangle);

    double left1 = d->topLeft.longitude();
    double right1 = d->bottomRight.longitude();
    double top1 = d->topLeft.latitude();
    double bottom1 = d->bottomRight.latitude();

    double left2 = rectangle.d_func()->topLeft.longitude();
    double right2 = rectangle.d_func()->bottomRight.longitude();
    double top2 = rectangle.d_func()->topLeft.latitude();
    double bottom2 = rectangle.d_func()->bottomRight.latitude();

    double top = qMax(top1, top2);
    double bottom = qMin(bottom1, bottom2);

    double left = 0.0;
    double right = 0.0;

    bool wrap1 = (left1 > right1);
    bool wrap2 = (left2 > right2);

    if ((wrap1 && wrap2) || (!wrap1 && !wrap2)) {

        double w = qAbs((left1 + right1 - left2 - right2) / 2.0);

        if (w < 180.0) {
            left = qMin(left1, left2);
            right = qMax(right1, right2);
        } else if (w > 180.0) {
            left = qMax(left1, left2);
            right = qMin(right1, right2);
        } else {
            left = -180.0;
            right = 180.0;
        }

    } else {
        double wrapLeft = 0.0;
        double wrapRight = 0.0;
        double nonWrapLeft = 0.0;
        double nonWrapRight = 0.0;

        if (wrap1) {
            wrapLeft = left1;
            wrapRight = right1;
            nonWrapLeft = left2;
            nonWrapRight = right2;
        } else {
            wrapLeft = left2;
            wrapRight = right2;
            nonWrapLeft = left1;
            nonWrapRight = right1;
        }

        bool joinWrapLeft = (nonWrapRight >= wrapLeft);
        bool joinWrapRight = (nonWrapLeft <= wrapRight);

        if (joinWrapLeft) {
            if (joinWrapRight) {
                left = -180.0;
                right = 180.0;
            } else {
                left = nonWrapLeft;
                right = wrapRight;
            }
        } else {
            if (joinWrapRight) {
                left = wrapLeft;
                right = nonWrapRight;
            } else {
                double wrapRightDistance = nonWrapLeft - wrapRight;
                double wrapLeftDistance = wrapLeft - nonWrapRight;

                if (wrapLeftDistance == wrapRightDistance) {
                    left = -180.0;
                    right = 180.0;
                } else if (wrapLeftDistance < wrapRightDistance) {
                    left = nonWrapLeft;
                    right = wrapRight;
                } else {
                    left = wrapLeft;
                    right = nonWrapRight;
                }
            }
        }
    }

    if (((left1 == -180) && (right1 == 180.0))
            || ((left2 == -180) && (right2 == 180.0))) {
        left = -180;
        right = 180;
    }

    d->topLeft = QGeoCoordinate(top, left);
    d->bottomRight = QGeoCoordinate(bottom, right);

    return *this;
}

/*******************************************************************************
*******************************************************************************/

QGeoRectanglePrivate::QGeoRectanglePrivate()
:   QGeoShapePrivate(QGeoShape::RectangleType)
{
}

QGeoRectanglePrivate::QGeoRectanglePrivate(const QGeoCoordinate &topLeft,
                                               const QGeoCoordinate &bottomRight)
:   QGeoShapePrivate(QGeoShape::RectangleType), topLeft(topLeft), bottomRight(bottomRight)
{
}

QGeoRectanglePrivate::QGeoRectanglePrivate(const QGeoRectanglePrivate &other)
:   QGeoShapePrivate(QGeoShape::RectangleType), topLeft(other.topLeft),
    bottomRight(other.bottomRight)
{
}

QGeoRectanglePrivate::~QGeoRectanglePrivate() {}

QGeoShapePrivate *QGeoRectanglePrivate::clone() const
{
    return new QGeoRectanglePrivate(*this);
}

bool QGeoRectanglePrivate::operator==(const QGeoShapePrivate &other) const
{
    if (!QGeoShapePrivate::operator==(other))
        return false;

    const QGeoRectanglePrivate &otherBox = static_cast<const QGeoRectanglePrivate &>(other);

    return topLeft == otherBox.topLeft && bottomRight == otherBox.bottomRight;
}

QT_END_NAMESPACE

