/* Copyright (C) 2006 P.L. Lucas
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, 
 * Boston, MA 02111-1307, USA. 
 */

#include "autocomplete.h"
#include <QString>
#include <QStringList>
#include <QFile>
#include <QTextStream>
#include <QEvent>
#include <QKeyEvent>
#include <QLineEdit>
#include <iostream>

using namespace std;

Autocomplete::Autocomplete(QWidget *parent): QComboBox(parent), search_string(""),
		tab_flag(false)
{
	current_match = first_match = word_list.end();
}

void Autocomplete::load_from_file(const char *file)
{
	QFile input(file);

	if(input.open(QIODevice::ReadOnly | QIODevice::Text))
	{
		QTextStream in_stream(&input);
		QString word;

		while(!in_stream.atEnd())
		{
			in_stream >> word;
			word_list.push_back(word);
		}

		word_list.sort();
	}else
		cerr << "No se ha podido cargar " << file << endl;
}

void Autocomplete::add(QString word)
{
	word_list.push_back(word);
}

void Autocomplete::remove(QString word)
{
	int i = word_list.indexOf(QRegExp(word, Qt::CaseInsensitive));
	word_list.removeAt(i);
}

void Autocomplete::clear()
{
	word_list.clear();
	current_match = first_match = word_list.end();
}

QString Autocomplete::search(QString start)
{
	QStringList::iterator i;
	QString word;

	search_string = start;

	for(i = word_list.begin(); i != word_list.end(); i++)
	{
		if(i->startsWith(start))
		{
			current_match = first_match = i;
			return *i;
		}
	}

	current_match = first_match = word_list.end();
	return QString("");
}

QString Autocomplete::get_next()
{
	if(first_match != word_list.end())
	{
		current_match++;

		if(!current_match->startsWith(search_string, Qt::CaseInsensitive))
			current_match = first_match;
		
		return *current_match;
	}else
		return QString("");
}

/* event function reimplemented */
bool Autocomplete::event(QEvent *event)
{
	if (event->type() == QEvent::KeyPress)
	{
		QKeyEvent *key_event = (QKeyEvent*)event;
		if(key_event->key() == Qt::Key_Tab)
		{
			// Autocomplete
			if(!tab_flag)
			{
				QString start, result;
				QString text = lineEdit()->text();
				int i;

				//Get the last word being writted
				for(i = text.size() - 1; i >= 0 && text[i].isLetterOrNumber(); i--);

				if(i == -1)
					start = text;
				else
					start = text.right(text.size() - i - 1);

				result = search(start);
				if(!result.isEmpty())
					lineEdit()->setText(text.left(i + 1) + search(start));

			}else if(current_match != word_list.end())
			{
				QString text = lineEdit()->text();
				int len = text.size() - current_match->size();

				lineEdit()->setText(text.left(len) + get_next());
			}

			tab_flag = true;
			// Event accepted
			key_event->accept();
			return TRUE;
		}else
			tab_flag = false;
	}

	return QWidget::event(event);
}

