/* Copyright (C) 2006 P.L. Lucas
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, 
 * Boston, MA 02111-1307, USA. 
 */

#include <QBoxLayout>
#include <QToolBar>
#include <QIcon>
#include <QMdiArea>
#include <QFileDialog>
#include <QMessageBox>
#include <QTextCursor>
#include <QMenuBar>
#include "editor.h"
#include "config.h"
#include "search_dialog.h"
#include "numberedtextview.h"
#include "navigator.h"


// Constructor
Editor::Editor(QWidget *parent): BaseWidget(parent)
{
	widget_type=EDITOR;
	
	currentNtv=NULL;
	search_dialog=NULL;
	// Title
	setWindowTitle(tr("Editor"));
	setWindowIcon(QIcon(QString(ICON_PATH) + "/xedit.png"));
	
	// Layout
	QVBoxLayout *vLayout = new QVBoxLayout;
	centralWidget()->setLayout(vLayout);
	vLayout->setSpacing (0);
	
	resize(540, 400);
	
	// Toolbar
	toolBar = addToolBar("Options");
	toolBar->setObjectName("Editor Options");
	//toolBar->setOrientation(Qt::Horizontal);
	//toolBar->setMinimumHeight(40);
	//toolBar->setMaximumHeight(40);
	connect(toolBar, SIGNAL(actionTriggered(QAction*)),
		this, SLOT(toolbar_action(QAction*)));
	//vLayout->addWidget(toolBar);
	//toolBar->show();
	
	// Toolbar buttons
	actionNew = toolBar->addAction(QIcon(QString(ICON_PATH) + "/filenew.png"), tr("New (Ctrl+N)"));
	actionNew->setShortcut(tr("Ctrl+N"));
	actionOpen = toolBar->addAction(QIcon(QString(ICON_PATH) + "/fileopen"), tr("Open (Ctrl+O)"));
	actionOpen->setShortcut(tr("Ctrl+O"));
	actionSave = toolBar->addAction(QIcon(QString(ICON_PATH) + "/filesave"), tr("Save (Ctrl+S)"));
	actionSave->setShortcut(tr("Ctrl+S"));
	actionSaveAs = toolBar->addAction(QIcon(QString(ICON_PATH) + "/filesaveas.png"), tr("Save as"));
	actionClose = toolBar->addAction(QIcon(QString(ICON_PATH) + "/fileclose.png"), tr("Close tab"));
	toolBar->addSeparator();
	
	actionRun = toolBar->addAction(QIcon(QString(ICON_PATH) + "/run.png"), tr("Run"));
	actionDebug = toolBar->addAction(QIcon(QString(ICON_PATH) + "/find.png"), tr("Debug"));
	actionStep = toolBar->addAction(QIcon(QString(ICON_PATH) + "/step.png"), tr("Next line"));
	actionStep->setEnabled(false);
	actionSendToOctave=toolBar->addAction(QIcon(QString(ICON_PATH) + "/konsole.png"), tr("Send to Octave") );
	toolBar->addSeparator();
	
	actionUndo = toolBar->addAction(QIcon(QString(ICON_PATH) + "/undo.png"), tr("Undo (Ctrl+Z)"));
	actionUndo->setShortcut(tr("Ctrl+Z"));
	actionCut = toolBar->addAction(QIcon(QString(ICON_PATH) + "/editcut"), tr("Cut (Ctrl+X)"));
	actionCut->setShortcut(tr("Ctrl+X"));
	actionCopy = toolBar->addAction(QIcon(QString(ICON_PATH) + "/editcopy"), tr("Copy (Ctrl+C)"));
	actionCopy->setShortcut(tr("Ctrl+C"));
	actionPaste = toolBar->addAction(QIcon(QString(ICON_PATH) + "/editpaste"),  tr("Paste (Ctrl+V)"));
	actionPaste->setShortcut(tr("Ctrl+V"));
	
	toolBar->addSeparator();
	
	actionSearch = toolBar->addAction(QIcon(QString(ICON_PATH) + "/search.png"),  tr("Search and replace"));
	
	//Create menus
	
	menuFile=menuBar()->addMenu("File");
	menuFile->addAction(actionNew);
	menuFile->addAction(actionOpen);
	menuFile->addAction(actionSave);
	menuFile->addAction(actionSaveAs);
	menuFile->addAction(actionClose);
	menuFile->addSeparator();
	QAction *actionCloseAll=menuFile->addAction("Close");
	connect(actionCloseAll, SIGNAL(triggered()),this, SLOT(close_editor()));
	//connect(menuFile, SIGNAL(triggered(QAction*)),this, SLOT(toolbar_action(QAction*)));
	
	menuEdit=menuBar()->addMenu("Edit");
	menuEdit->addAction(actionUndo);
	menuEdit->addAction(actionCut);
	menuEdit->addAction(actionCopy);
	menuEdit->addAction(actionPaste);
	menuEdit->addAction(actionSearch);
	//connect(menuEdit, SIGNAL(triggered(QAction*)),this, SLOT(toolbar_action(QAction*)));
	
	menuRun=menuBar()->addMenu("Run");
	menuRun->addAction(actionRun);
	menuRun->addAction(actionSendToOctave);
	menuRun->addAction(actionDebug);
	menuRun->addAction(actionStep);
	//connect(menuRun, SIGNAL(triggered(QAction*)),this, SLOT(toolbar_action(QAction*)));
	
	menuConfig=menuBar()->addMenu("Config");
	
	
	// TabWidget
	tabWidget = new QTabWidget(this);
	tabWidget->show();
	
	connect(tabWidget, SIGNAL(currentChanged(int)),
		this, SLOT(tabChanged(int)));
	
	vLayout->addWidget(tabWidget);
	
	//List of files
	QDockWidget *list_files_dock=new QDockWidget("File list",this);
	list_files_dock->setObjectName(list_files_dock->windowTitle());
	addDockWidget(Qt::LeftDockWidgetArea, list_files_dock);
	list_files_dock->show();
	
	list_files=new QListView(list_files_dock);
	list_files_dock->setWidget(list_files);
	list_files_dock->show();
	list_files->setModel(new ListModel());
	connect(list_files,SIGNAL(clicked (const QModelIndex &)), this, SLOT(file_selected(const QModelIndex &)) );
	
	//This menu is builded here because it show objects if they hav been created
	menuDocks=createPopupMenu();
	menuDocks->setTitle("Show/Hide Objects");
	menuConfig->addMenu(menuDocks);
	
	setAcceptDrops(true);
	
	project_name="Empty";
	
	toolbar_action(actionNew);
}

Editor::~Editor()
{
	saveProject();
	
	if(search_dialog!=NULL) delete search_dialog;
}

void Editor::saveProject()
{
	QStringList files;
	for(int i=0;i<tabWidget->count();i++)
	{
		QString path=((NumberedTextView*)tabWidget->widget(i) )->path();
		if(!path.isEmpty()) files.append( path );
	}
	if(project_name.isEmpty()) Projects::saveListFiles("Empty", files);
	else Projects::saveListFiles(project_name, files);
}


void Editor::dragEnterEvent(QDragEnterEvent *event)
{
	if (event->mimeData()->hasFormat("text/plain"))
		event->acceptProposedAction();
}


void Editor::dropEvent ( QDropEvent * event )
{
	QString path=event->mimeData()->text();
	
	openFile(path);

//       try{
// 	// Si la pestaña activa no contiene ningún texto ni es un archivo,
// 	// abrir en ella
// 	if(currentNtv->path().isEmpty() &&
// 	   currentNtv->textEdit()->document()->isEmpty())
// 	{
// 	  tabWidget->setTabText(tabWidget->currentIndex(), path.split("/").last());
// 	}
// 	else
// 	{
// 	  CodeEdit *codeEdit = new CodeEdit(NULL, QString(CONFIG_PATH) + "/octave.xml");
// 	  NumberedTextView *ntv = new NumberedTextView(NULL, codeEdit);
// 	  connect(ntv->textEdit(), SIGNAL(toggleBreakpoint(int)), this, SLOT(toggleBreakpoint(int)));
// 	  //connect(ntv, SIGNAL(textModified()), this, SLOT(textModified()));
// 	  connect(ntv->textEdit()->document(), SIGNAL(modificationChanged (bool)), this, SLOT(textModified(bool)));
// 
// 	  tabWidget->setCurrentIndex(tabWidget->addTab(ntv, path.split("/").last()));
// 	}  
// 
// 	disconnect(currentNtv->textEdit()->document(), SIGNAL(modificationChanged (bool)), this, SLOT(textModified(bool)));
// 	currentNtv->open(path);
// 	connect(currentNtv->textEdit()->document(), SIGNAL(modificationChanged (bool)), this, SLOT(textModified(bool)));
// 
//       }catch(...){
// 	QMessageBox::critical(NULL, tr("Error"), path + " can not be opened");
//       }
	
	event->acceptProposedAction();
}


void Editor::setOctaveConnection(OctaveConnection *oc)
{
  octave_connection = oc;
}


void Editor::toolbar_action(QAction *action)
{
  QStringList filters;

  filters << "Octave (*.m; *.M)"
	  << "Plain text (*.txt)"
	  << "All files (*)";

  
  if(action == actionNew){
    /** New **/
    CodeEdit *codeEdit = new CodeEdit(NULL, QString(CONFIG_PATH) + "/octave.xml");
    connect(codeEdit, SIGNAL(dynamic_help_required(const QString &)), this, SLOT(emit_dynamic_help_required(const QString &)));
    
    NumberedTextView *ntv = new NumberedTextView(NULL, codeEdit);
    connect(ntv->textEdit(), SIGNAL(toggleBreakpoint(int)), this, SLOT(toggleBreakpoint(int)));
    //connect(ntv, SIGNAL(textModified()), this, SLOT(textModified()));
    connect(ntv->textEdit()->document(), SIGNAL(modificationChanged (bool)), this, SLOT(textModified(bool)));
    
    currentNtv = ntv;

    tabWidget->setCurrentIndex( tabWidget->addTab(ntv, tr("New")) );

  }else if(action == actionOpen){
	/** Open **/
	openFile();
  }else if(action == actionSave && !currentNtv->path().isEmpty()){  
    /* Save */
    try{
      currentNtv->save();
      setTabText(tabWidget->currentIndex(), currentNtv->path().split("/").last());
    }catch(...){
      QMessageBox::critical(NULL, tr("Error"), "Can not be saved");
    }

  }else if(action == actionSaveAs || (action == actionSave && currentNtv->path().isEmpty())){
    /** Save as **/
    QString path;
    QFileDialog saveDialog(this, Qt::Dialog);

    saveDialog.setAcceptMode(QFileDialog::AcceptSave);
    saveDialog.setDefaultSuffix("m");
    saveDialog.setFilters(filters);
    
    //Use Navigator path if current path is empty
    if(currentNtv->path().isEmpty())
    {
	   QObject *obj= session->getFirstTool(NAVIGATOR);
	   if(obj!=NULL)
	   {
		   Navigator *nav=(Navigator*)obj;
		   saveDialog.setDirectory(nav->getNavigatorCurrentPath());
	   }
    }
    else
    {
	    QFileInfo current_file(currentNtv->path());
	    saveDialog.setDirectory(current_file.absolutePath());
	    saveDialog.selectFile(current_file.baseName());
    }

    if(saveDialog.exec() == QDialog::Accepted)
    {
      path = saveDialog.selectedFiles().first();

      try{
	currentNtv->save(path);
	setTabText(tabWidget->currentIndex(), path.split("/").last());
      }catch(...){
	QMessageBox::critical(NULL, tr("Error"), path + "can not be saved");
      }
    }
  }else if(action == actionRun){
	//if(currentNtv->path().isEmpty())
	//{
	//	QMessageBox::critical(NULL, tr("Error"), tr("You must save the file first"));
	//	return;
	//}
	if( currentNtv->modified() ) toolbar_action(actionSave);
	
	QFileInfo finfo(currentNtv->path());
	octave_connection->command_enter(QString("cd '") + finfo.path() + "'");
	octave_connection->command_enter(finfo.baseName());
	
  }else if(action == actionDebug){
    /** Run */
    if(currentNtv->path().isEmpty())
    {
      QMessageBox::critical(NULL, tr("Error"), tr("You must save the file first"));
      return;
    }

    // Debug?
    if(actionStep->isEnabled())
      octave_connection->command_enter(QString("dbcont"));
    else
    {
	QFileInfo finfo(currentNtv->path());
	QList<int> *breakpoints = currentNtv->getBreakpoints();
	if(breakpoints!=NULL)
	{
		// Source
		//octave_connection->command_enter(QString("source('") + finfo.absoluteFilePath() + "')");
		
		// Clear breakpoints
		//octave_connection->command_enter(QString("dbclear('") + finfo.baseName() + "',dbstatus('"+finfo.baseName()+"') )");
		
		//Change to dir
		octave_connection->command_enter(QString("cd '") + finfo.path() + "'");
		octave_connection->command_enter(QString(
			"while (  length (dbstatus('" + finfo.baseName() + "')) >0  )"
			"dbclear('" + finfo.baseName() + "', dbstatus('" + finfo.baseName() + "')(1).line);"
			"endwhile"
			) );
		
		// Insert breakpoints
		for(QList<int>::const_iterator i = breakpoints->constBegin();
			i != breakpoints->constEnd();
			i++)
			{
			octave_connection->command_enter(QString("dbstop('")
							+ finfo.baseName()
							+ "','" + QString::number(*i) + "')");
			}
		
		// Connect debug
		connect(octave_connection, SIGNAL(debug(int, int)),
			this, SLOT(debug(int, int)));
		connect(octave_connection, SIGNAL(endDebug()),
			this, SLOT(endDebug()));
		
		// Run
		octave_connection->command_enter(finfo.baseName());
	}
    } // End debug?

  }else if(action == actionUndo){
    // Undo
    currentNtv->textEdit()->document()->undo();
  }else if(action == actionCut){
    // Cut
    currentNtv->textEdit()->cut();
  }else if(action == actionCopy){
    // Copy
    currentNtv->textEdit()->copy();
  }else if(action == actionPaste){
    // Paste
    currentNtv->textEdit()->paste();
  }else if(action == actionSearch){
	if(search_dialog==NULL)
	{
		search_dialog=new SearchDialog(this);
		connect(search_dialog, SIGNAL(search_signal()), this, SLOT(search()));
		connect(search_dialog, SIGNAL(replace_signal()), this, SLOT(replace()));
	}
	search_dialog->show();
  }else if(action == actionSendToOctave){
    octave_connection->command_enter( currentNtv->textEdit()->document()->toPlainText() );
  }else if(action == actionStep){
    octave_connection->command_enter( "dbstep" );
  }else if(action == actionClose){
    closeTabs(false);
  }
  else{
    printf("Unhandled action\n");
  }
}


void Editor::openFile(QString file)
{
	/** Open **/
	QString path;
	
	if(file.isEmpty())
	{
		QFileDialog openDialog(this, tr("Open") /*Qt::Dialog*/);
		
		QStringList filters;
		filters << "Octave (*.m; *.M)"
			<< "Plain text (*.txt)"
			<< "All files (*)";
		
		openDialog.setAcceptMode(QFileDialog::AcceptOpen);
		openDialog.setDefaultSuffix("m");
		openDialog.setFilters(filters);
		//openDialog.setViewMode(QFileDialog::Detail);
		QFileInfo current_file(currentNtv->path());
		openDialog.setDirectory(current_file.absolutePath());
		openDialog.selectFile(current_file.baseName());
		
		if(openDialog.exec() == QDialog::Accepted)
			path = openDialog.selectedFiles().first();
		else return;
	}
	else
		path=file;
	
	loadFiles(QStringList() << path);
}

void Editor::setProject(QString name)
{
	project_name=name;
	closeTabs(true);
	QStringList files=Projects::listFiles(project_name);
	loadFiles(files);
}

QString Editor::getProject()
{
	return project_name;
}

void Editor::setSession(Session *session)
{
	BaseWidget::setSession(session);
	setProject(session->getProjectName());
	connect(session, SIGNAL(projectChanged(QString)), this, SLOT(setProject(QString)) );
}

void Editor::search()
{
  QString search, replace;
  QTextCursor cursor;
  QTextEdit *textEdit = currentNtv->textEdit();

  // Strings
  search = search_dialog->searchString();
  replace = search_dialog->replaceString();

  // Flags
  QTextDocument::FindFlags flags;
  if(search_dialog->caseSensitive())
    flags |= QTextDocument::FindCaseSensitively;
  if(search_dialog->wholeWords())
    flags |= QTextDocument::FindWholeWords;

  // Search
  cursor = textEdit->textCursor();
  if(search_dialog->searchStringIsRegExp())
  {
    // Search string is a regular expression
    QRegExp searchReg(search);
    
    cursor = textEdit->document()->find(searchReg, cursor, flags);
    //cursor = textEdit->document()->find(search, cursor, flags);
  }else{
    // Search string is not a regular expression
    cursor = textEdit->document()->find(search, cursor, flags);
  }

  textEdit->setTextCursor(cursor);
}

void Editor::replace()
{
	QTextCursor cursor = currentNtv->textEdit()->textCursor();
	
	if(!cursor.selectedText().isEmpty())
	{
		int pos=cursor.position();
		cursor.insertText(search_dialog->replaceString());
		cursor.setPosition(pos);
		currentNtv->textEdit()->setTextCursor(cursor);
	}
	
	//Next line is comented because editor loose cursor
	//search();
}

void Editor::toggleBreakpoint(int lineno)
{
  currentNtv->toggleBreakpoint(lineno);
}

void Editor::debug(int lineno, int /*colno*/)
{
  currentNtv->setCurrentLine(lineno);
  actionStep->setEnabled(true);
}

void Editor::endDebug()
{
  currentNtv->setCurrentLine(-1);
  actionStep->setEnabled(false);
}

void Editor::tabChanged(int index)
{
	//printf("Activado %d\n", index);
	//if(currentNtv!=NULL)
	//	disconnect(currentNtv->textEdit(), SIGNAL(toggleBreakpoint(int)), this, SLOT(toggleBreakpoint(int)));
	currentNtv = (NumberedTextView*)tabWidget->widget(index);
	setWindowTitle(tabWidget->tabText(index));
	//connect(currentNtv->textEdit(), SIGNAL(toggleBreakpoint(int)), this, SLOT(toggleBreakpoint(int)));
	ListModel *list=(ListModel*)list_files->model();
	list_files->setCurrentIndex(list->position_index(index));
}

void Editor::textModified(bool ok)
{
	if(!ok) return;
	QTextEdit *text=currentNtv->textEdit();
	if(!text->document()->isModified()) return;
	if(currentNtv->path().isEmpty())
	{
		setTabText(tabWidget->currentIndex(), tr("New")+"*");
	}
	else
	{
		setTabText(tabWidget->currentIndex(), currentNtv->path().split("/").last()+"*");
	}
}

/////////////////////////////////////////////////////////////

void Editor::closeEvent ( QCloseEvent * event )
{
	bool modified=false;
	for(int i=0;i<tabWidget->count();i++)
	{
		modified|=( (NumberedTextView*)tabWidget->widget(i) )->modified();
	}
	int ok;
	if(modified)
		ok=QMessageBox::warning (this,tr("Close this window?"), tr("You are going to close Editor. Are you sure?"), QMessageBox::Ok, QMessageBox::Cancel);
	else
		ok=QMessageBox::Ok;

	if (ok==QMessageBox::Ok)
	{
		event->accept();
	}
	else
	{
		event->ignore();
	}
}

/////////////////////////////////////////////////////////////

void Editor::closeTabs(bool close_all_tabs)
{
	while(tabWidget->count()>0)
	{
		if(
			(
				currentNtv->modified() &&
				!currentNtv->textEdit()->toPlainText().isEmpty() &&
				currentNtv->path().isEmpty()
			)
			||
			(
				currentNtv->modified() && !currentNtv->path().isEmpty()
			)
		)
		{
			QMessageBox msg(tr("Close"), tr("The file has been modified. Save changes?"),
					QMessageBox::Question,
					QMessageBox::Yes, QMessageBox::No,
					QMessageBox::Cancel | QMessageBox::Default,
					this);
			
			switch(msg.exec())
			{
				case QMessageBox::Yes:
					toolbar_action(actionSave);
					break;
				case QMessageBox::No:
					// No hacer nada
					break;
				default:
					return;
			}
		}
		
		// Borrar
		//tabWidget->removeTab(tabWidget->currentIndex());
		disconnect(currentNtv->textEdit()->document(), SIGNAL(modificationChanged (bool)), this, SLOT(textModified(bool)));
		delete currentNtv;
		
		if(!close_all_tabs) break;
	}
	
	// Crear uno si no queda ninguno
	if(tabWidget->count() == 0)
	{
		CodeEdit *codeEdit = new CodeEdit(NULL, QString(CONFIG_PATH) + "/octave.xml");
		connect(codeEdit, SIGNAL(dynamic_help_required(const QString &)), this, SLOT(emit_dynamic_help_required(const QString &)));
		NumberedTextView *ntv = new NumberedTextView(NULL, codeEdit);
		connect(ntv->textEdit(), SIGNAL(toggleBreakpoint(int)), this, SLOT(toggleBreakpoint(int)));
		//connect(ntv, SIGNAL(textModified()), this, SLOT(textModified()));
		connect(ntv->textEdit()->document(), SIGNAL(modificationChanged (bool)), this, SLOT(textModified(bool)));
		
		tabWidget->addTab(ntv, tr("New"));
		
		currentNtv = ntv;
	}
	else
	{
		tabChanged(tabWidget->currentIndex());
	}
	
	updateFileList();
}

/////////////////////////////////////////////////////////////

void Editor::loadFiles(const QStringList &files)
{
	QString path;
	foreach(path, files)
	{
		if( path.isEmpty() ) continue;
		try{
			// Si la pestaña activa no contiene ningún texto ni es un archivo,
			// abrir en ella
			if(currentNtv->path().isEmpty() && currentNtv->textEdit()->document()->isEmpty())
			{
				setTabText(tabWidget->currentIndex(), path.split("/").last());
			}
			else
			{
				CodeEdit *codeEdit = new CodeEdit(NULL, QString(CONFIG_PATH) + "/octave.xml");
				connect(codeEdit, SIGNAL(dynamic_help_required(const QString &)), this, SLOT(emit_dynamic_help_required(const QString &)));
				NumberedTextView *ntv = new NumberedTextView(NULL, codeEdit);
				connect(ntv->textEdit(), SIGNAL(toggleBreakpoint(int)), this, SLOT(toggleBreakpoint(int)));
			
				tabWidget->setCurrentIndex(tabWidget->addTab(ntv, path.split("/").last()));
				updateFileList();
			}
			
			disconnect(currentNtv->textEdit()->document(), SIGNAL(modificationChanged (bool)), this, SLOT(textModified(bool)));
			
			currentNtv->open(path);
	
			connect(currentNtv->textEdit()->document(), SIGNAL(modificationChanged (bool)), this, SLOT(textModified(bool)));
		}
		catch(...)
		{
			QMessageBox::critical(NULL, tr("Error"), path + " can not be opened");
		}
	}
}

/////////////////////////////////////////////////////////////

void Editor::close_editor()
{
	if(parent()!=NULL) ((QWidget*)parent())->close();
	else close();
}

void Editor::emit_dynamic_help_required(const QString &text)
{
	//printf("%s\n", text.toLocal8Bit().data());
	emit dynamic_help_required(text);
}

BaseWidget *Editor::copyBaseWidget(QWidget * parent )
{
	saveProject();
	
	Editor *bw=new Editor(parent);
	bw->setSession(session);
	bw->octave_connection=octave_connection;
	
	for(int i=bw->tabWidget->count();i>0;i--)
	{
		bw->toolbar_action(bw->actionClose);
	}
	
	for(int i=0;i<tabWidget->count();i++)
	{
		NumberedTextView *code=((NumberedTextView*)tabWidget->widget(i) );
		if(i!=0) bw->toolbar_action(bw->actionNew);
		bw->currentNtv->textEdit()->setPlainText(code->textEdit()->toPlainText());
		bw->currentNtv->setPath(code->path());
		bw->currentNtv->setModified(code->modified());
		code->setModified(false);
		if(!code->path().isEmpty())
		{
			bw->setTabText(bw->tabWidget->currentIndex(), code->path().split("/").last());
		}
	}
	
	return bw;
}

void Editor::setTabText(int index, const QString & label)
{
	tabWidget->setTabText(index, label);
	
	updateFileList();
}

void Editor::updateFileList()
{
	ListModel *model=(ListModel *)list_files->model();
	model->clear();
	
	for(int i=0;i<tabWidget->count();i++)
	{
		model->append(tabWidget->tabText(i), i);
	}
	
	model->update();
}

void Editor::file_selected(const QModelIndex & index)
{
	ListModel *model=(ListModel *)list_files->model();
	tabWidget->setCurrentIndex(model->position(index));
}


ListModel::ListModel(QObject *parent):QAbstractListModel(parent)
{
}

int ListModel::rowCount(const QModelIndex &parent) const
{
	return list.size();
}

QVariant ListModel::data(const QModelIndex &index, int role) const
{
	if (!index.isValid() || index.row() >= list.size())
		return QVariant();
	
	if (role == Qt::DisplayRole)
		return list.at(index.row()).name;
	else
		return QVariant();
}

void ListModel::clear()
{
	list.clear();
}

void ListModel::append(QString name, int position)
{
	ListItem item;
	item.name=name;
	item.position=position;
	list.append(item);
}

int ListModel::position(const QModelIndex &index)
{
	return list.at(index.row()).position;
}

void ListModel::update()
{
	beginInsertRows(QModelIndex(), 0, list.size()-1);
	endInsertRows();
}

QModelIndex ListModel::position_index(int position)
{
	for(int i=0;i<list.size();i++)
	{
		if(list[i].position==position)
		{
			return index(i,0);
		}
	}
	
	return QModelIndex();
}

