# Copyright 2019 The Chromium Authors. All rights reserved.
# Use of this source code is governed by a BSD-style license that can be
# found in the LICENSE file.

import unittest
import re
import os


class TestProguardRuleGeneration(unittest.TestCase):
  """
  This script is used to test a ProGuard keep rules for the purposes
  of maintaining compatibility between async DFMs and synchronously
  proguarded modules.

  The rules are often generated by constant_pool_refs_to_keep_rules.py

  This test can be run manually. Example:
  python build/android/constant_pool_refs_to_keep_rules_test.py -v
  """

  # Make sure this variable is set accordingly.
  # It should point to a proguard file.
  PROGUARD_FILE_PATH = os.path.join(
      os.path.dirname(__file__),
      "../../chrome/android/features/tab_ui/proguard_async.flags")

  def test_TabUI_HasRules(self):
    """
    Ensures that a few of the rules used in tabs_ui module are included.
    Although this is far from 100% deterministic, these rules are
    created by code that exercise different parts of the rule generation code.
    """

    rules = set()
    with open(self.PROGUARD_FILE_PATH, 'r') as proguard_rules:
      for line in proguard_rules:
        if line.startswith('-keep'):
          rule = re.search('class (.+?) {', line).group(1)
          rules.add(rule)

    # The following rules test most of the use cases for
    # rules that can be added automatically.
    self.assertIn('org.chromium.ui.modelutil.PropertyModel', rules)
    self.assertIn('org.chromium.ui.modelutil.PropertyModel', rules)
    self.assertIn('org.chromium.ui.modelutil.PropertyKey', rules)
    self.assertIn('org.chromium.chrome.browser.toolbar.ToolbarManager', rules)
    self.assertIn('org.chromium.base.Supplier', rules)
    self.assertIn('android.support.v7.widget.helper.ItemTouchHelper', rules)
    self.assertIn(
        'android.support.v7.widget.helper.ItemTouchHelper$SimpleCallback',
        rules)
    self.assertIn('android.support.v7.widget.helper.ItemTouchHelper$Callback',
                  rules)
    self.assertIn('android.support.v4.content.ContextCompat', rules)
    self.assertIn('android.support.v7.widget.GridLayoutManager', rules)
    self.assertIn('android.support.v4.content.res.ResourcesCompat', rules)
    self.assertIn(
        'org.chromium.chrome.browser.tasks.tabgroup.TabGroupModelFilter', rules)
    self.assertIn('android.support.v7.widget.RecyclerView$ViewHolder', rules)
    self.assertIn('android.support.v7.widget.RecyclerView', rules)
    self.assertIn('org.chromium.ui.modelutil.SimpleRecyclerViewMcpBase', rules)
    self.assertIn('org.chromium.ui.modelutil.RecyclerViewAdapter', rules)

    # The following rules need to be added manually.
    self.assertNotIn(
        'org.chromium.chrome.browser.fullscreen.ChromeFullscreenManager' +
        '$FullscreenListener$$CC', rules)
    self.assertNotIn(
        'org.chromium.chrome.browser.widget.bottomsheet.BottomSheet' +
        '$BottomSheetContent$$CC', rules)
    self.assertNotIn('org.chromium.ui.widget.RoundedCornerImageView', rules)
    self.assertNotIn(
        'android.support.v4.graphics.drawable.RoundedBitmapDrawable', rules)

  def test_TabUI_HasNoDuplicateRules(self):
    """
    Ensures that there are no duplicate keep rules
    """

    rules = set()
    with open(self.PROGUARD_FILE_PATH, 'r') as proguard_rules:
      for line in proguard_rules:
        if line.startswith('-keep'):
          rule = re.search('class (.+?) {', line).group(1)
          self.assertNotIn(rule, rules)
          rules.add(rule)


if __name__ == '__main__':
  unittest.main()
