/* -*- mode: c++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */

/*
 Copyright (C) 2000, 2001, 2002, 2003 RiskMap srl
 Copyright (C) 2003, 2004, 2005, 2006 StatPro Italia srl

 This file is part of QuantLib, a free-software/open-source library
 for financial quantitative analysts and developers - http://quantlib.org/

 QuantLib is free software: you can redistribute it and/or modify it
 under the terms of the QuantLib license.  You should have received a
 copy of the license along with this program; if not, please email
 <quantlib-dev@lists.sf.net>. The license is also available online at
 <http://quantlib.org/reference/license.html>.

 This program is distributed in the hope that it will be useful, but WITHOUT
 ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 FOR A PARTICULAR PURPOSE.  See the license for more details.
*/

/*! \file ratehelpers.hpp
    \brief deposit, FRA, futures, and swap rate helpers
*/

#ifndef quantlib_ratehelpers_hpp
#define quantlib_ratehelpers_hpp

#include <ql/TermStructures/piecewiseyieldcurve.hpp>
#include <ql/Instruments/vanillaswap.hpp>

namespace QuantLib {

    //! Rate helper for bootstrapping over interest-rate futures prices
    /*! \todo implement/refactor constructors with:
              Index instead of (nMonths, calendar, convention, dayCounter),
              IMM code
    */
    class FuturesRateHelper : public RateHelper {
      public:
        FuturesRateHelper(const Handle<Quote>& price,
                          const Date& immDate,
                          Integer nMonths,
                          const Calendar& calendar,
                          BusinessDayConvention convention,
                          const DayCounter& dayCounter,
                          const Handle<Quote>& convexityAdjustment);
        FuturesRateHelper(const Handle<Quote>& price,
                          const Date& immDate,
                          Integer nMonths,
                          const Calendar& calendar,
                          BusinessDayConvention convention,
                          const DayCounter& dayCounter,
                          Rate convexityAdjustment = 0.0);
        FuturesRateHelper(Real price,
                          const Date& immDate,
                          Integer nMonths,
                          const Calendar& calendar,
                          BusinessDayConvention convention,
                          const DayCounter& dayCounter,
                          Rate convexityAdjustment = 0.0);
        Real impliedQuote() const;
        DiscountFactor discountGuess() const;
        Real convexityAdjustment() const { return convAdj_->value(); }
      private:
        Time yearFraction_;
        Handle<Quote> convAdj_;
    };

    //! Rate helper with date schedule relative to the global evaluation date
    /*! This class takes care of rebuilding the date schedule when the global
        evaluation date changes
    */
    class RelativeDateRateHelper : public RateHelper {
      public:
        RelativeDateRateHelper(const Handle<Quote>& quote);
        RelativeDateRateHelper(Real quote);
        void update();
      protected:
        virtual void initializeDates() = 0;
        Date evaluationDate_;
    };


    //! Rate helper for bootstrapping over deposit rates
    class DepositRateHelper : public RelativeDateRateHelper {
      public:
        DepositRateHelper(const Handle<Quote>& rate,
                          const Period& tenor,
                          Integer settlementDays,
                          const Calendar& calendar,
                          BusinessDayConvention convention,
                          const DayCounter& dayCounter);
        DepositRateHelper(Rate rate,
                          const Period& tenor,
                          Integer settlementDays,
                          const Calendar& calendar,
                          BusinessDayConvention convention,
                          const DayCounter& dayCounter);
        Real impliedQuote() const;
        DiscountFactor discountGuess() const;
      private:
        void initializeDates();
        Period tenor_;
        Integer settlementDays_;
        Calendar calendar_;
        BusinessDayConvention convention_;
        DayCounter dayCounter_;
        Time yearFraction_;
    };


    //! Rate helper for bootstrapping over %FRA rates
    class FraRateHelper : public RelativeDateRateHelper {
      public:
        FraRateHelper(const Handle<Quote>& rate,
                      Integer monthsToStart, Integer monthsToEnd,
                      Integer settlementDays,
                      const Calendar& calendar,
                      BusinessDayConvention convention,
                      const DayCounter& dayCounter);
        FraRateHelper(Rate rate,
                      Integer monthsToStart, Integer monthsToEnd,
                      Integer settlementDays,
                      const Calendar& calendar,
                      BusinessDayConvention convention,
                      const DayCounter& dayCounter);
        Real impliedQuote() const;
        DiscountFactor discountGuess() const;
      private:
        void initializeDates();
        Integer monthsToStart_, monthsToEnd_;
        Integer settlementDays_;
        Calendar calendar_;
        BusinessDayConvention convention_;
        DayCounter dayCounter_;
        Time yearFraction_;
    };

    //! Rate helper for bootstrapping over swap rates
    class SwapRateHelper : public RelativeDateRateHelper {
      public:
        SwapRateHelper(const Handle<Quote>& rate,
                       const Period& tenor,
                       Integer settlementDays,
                       const Calendar& calendar,
                       // fixed leg
                       Frequency fixedFrequency,
                       BusinessDayConvention fixedConvention,
                       const DayCounter& fixedDayCount,
                       // floating leg
                       const boost::shared_ptr<Xibor>& index);
        SwapRateHelper(Rate rate,
                       const Period& tenor,
                       Integer settlementDays,
                       const Calendar& calendar,
                       // fixed leg
                       Frequency fixedFrequency,
                       BusinessDayConvention fixedConvention,
                       const DayCounter& fixedDayCount,
                       // floating leg
                       const boost::shared_ptr<Xibor>& index);
        Real impliedQuote() const;
        // implementing discountGuess() is not worthwhile,
        // and may not avoid the root-finding process
        void setTermStructure(YieldTermStructure*);
      protected:
        void initializeDates();
        Period tenor_;
        Integer settlementDays_;
        Calendar calendar_;
        BusinessDayConvention fixedConvention_;
        Frequency fixedFrequency_;
        DayCounter fixedDayCount_;
        boost::shared_ptr<Xibor> index_;
        boost::shared_ptr<VanillaSwap> swap_;
        Handle<YieldTermStructure> termStructureHandle_;
    };

}


#endif
