/* -*- mode: c++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */

/*
 Copyright (C) 2006 Mario Pucci

 This file is part of QuantLib, a free-software/open-source library
 for financial quantitative analysts and developers - http://quantlib.org/

 QuantLib is free software: you can redistribute it and/or modify it
 under the terms of the QuantLib license.  You should have received a
 copy of the license along with this program; if not, please email
 <quantlib-dev@lists.sf.net>. The license is also available online at
 <http://quantlib.org/license.shtml>.

 This program is distributed in the hope that it will be useful, but WITHOUT
 ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 FOR A PARTICULAR PURPOSE.  See the license for more details.
*/

/*! \file smilesection.hpp
    \brief Swaption volatility structure
*/

#ifndef quantlib_smile_section_hpp
#define quantlib_smile_section_hpp

#include <ql/patterns/observable.hpp>
#include <ql/time/date.hpp>
#include <ql/time/daycounters/actual365fixed.hpp>
#include <vector>

namespace QuantLib {

    //! interest rate volatility smile section
    /*! This abstract class provides volatility smile section interface */
    class SmileSection : public virtual Observable {
      public:
        SmileSection(const Date& d,
                     const DayCounter& dc = Actual365Fixed(),
                     const Date& referenceDate = Date());
        SmileSection(Time exerciseTime,
                     const DayCounter& dc = Actual365Fixed());
        virtual ~SmileSection() {};

        virtual Real minStrike() const = 0;
        virtual Real maxStrike() const = 0;
        virtual Real variance(Rate strike) const = 0;
        virtual Volatility volatility(Rate strike) const = 0;

        //virtual Rate atmLevel() const = 0;
        const Date& exerciseDate() const { return exerciseDate_; }
        Time exerciseTime() const { return exerciseTime_; };
        const DayCounter& dayCounter() const { return dc_; }
      protected:
        Date exerciseDate_;
        DayCounter dc_;
        Time exerciseTime_;
    };

    class FlatSmileSection : public SmileSection {
      public:
        FlatSmileSection(const Date& d,
                         Volatility vol,
                         const DayCounter& dc,
                         const Date& referenceDate = Date())
        : SmileSection(d, dc, referenceDate), vol_(vol) {};

        FlatSmileSection(Time exerciseTime,
                         Volatility vol,
                         const DayCounter& dc = Actual365Fixed())
        : SmileSection(exerciseTime, dc), vol_(vol) {};

        Real variance(Rate) const { return vol_*vol_*exerciseTime_; }
        Volatility volatility(Rate) const { return vol_; }
        Real minStrike () const { return 0.0; };
        Real maxStrike () const { return QL_MAX_REAL; };

      private:
        Volatility vol_;
    };

    class SabrSmileSection : public SmileSection {
      public:
        SabrSmileSection(Time timeToExpiry,
                         Rate forward,
                         const std::vector<Real>& sabrParameters);
        SabrSmileSection(const Date& d,
                         Rate forward,
                         const std::vector<Real>& sabrParameters,
                         const DayCounter& dc = Actual365Fixed());
        Real variance(Rate strike) const;
        Volatility volatility(Rate strike) const;
        Real minStrike () const { return 0.0; };
        Real maxStrike () const { return QL_MAX_REAL; };
    private:
        Real alpha_, beta_, nu_, rho_, forward_;
    };

}

#endif
