/***************************************************************************
 *            sort.c
 *
 *  Sun Sep 24 12:13:51 2006
 *  Copyright  2006-2007  Neil Williams
 *  linux@codehelp.co.uk
 ****************************************************************************/
/** @file sort.c
	@author Copyright 2006-2007  Neil Williams <linux@codehelp.co.uk>
	@author Copyright 1999  Sam Phillips (sam@usaworks.com)
	@brief column sort 
*/
/*
    This package is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "config.h"

#include <math.h>
#include <stdlib.h>
#include <string.h>
#include <glib.h>
#include <gtk/gtk.h>
#include <glib/gi18n.h>
#include <gtkextra/gtksheet.h>
#include "types.h"
#include "dim_list_menu.h"
#include "edit.h"
#include "main.h"
#include "dialog_initial.h"

#define FROM_LIST_WIDTH   180
#define FROM_LIST_HEIGHT  180

typedef struct
{
	union
	{
		gdouble number;
		gchar *string;
	} level[MAX_SORT_NESTING];
	gint old_row;
} QlSortRow;

QlSortRow *big_array;

QlSortInfo * curr_sort;
QlSortInfo * edit_sort;

enum {
	SORT_COL,
	FROM_COL,
};

typedef enum 
{
	QL_SORT_NONE,
	QL_SORT_ADD,
	QL_SORT_EDIT,
	QL_SORT_APPLY,
	QL_SORT_DELETE
} QlSortType;

QlSortType select_mode;

/* how many sort levels there are */
gint levels;
gint level_type[MAX_SORT_NESTING + 1];
/* +1 or -1 */
gint level_ascending[MAX_SORT_NESTING + 1];
gint level_formatting[MAX_SORT_NESTING + 1];

/*  User clicked ascending at some level */
static void
ascending_clicked (GtkObject G_GNUC_UNUSED * object, gpointer entry)
{
	gint level = GPOINTER_TO_INT (entry);
	edit_sort->line[level].ascending = TRUE;
}

/*  User clicked ascending at some level */
static void
descending_clicked (GtkObject G_GNUC_UNUSED * object, gpointer entry)
{
	gint level = GPOINTER_TO_INT (entry);
	edit_sort->line[level].ascending = FALSE;
}

/*  User clicked to change a field at some level */
static void
field_change (GtkObject * object, gpointer G_GNUC_UNUSED entry)
{
	gint iselect, level, fieldx;

	iselect = gtk_combo_box_get_active (GTK_COMBO_BOX (object));
	level = iselect / 100;
	fieldx = iselect % 100;
	edit_sort->line[level].field = fieldx;
}

/* Build the field button for the sort dialog box  */
static GtkWidget *
field_menu (QlTabData * tab, gint level)
{
	GtkWidget *temp_box, *ascend_button, *descend_button;
	GtkWidget *menu, *menu_item, *name_label;
	gint ret_value, fieldx;
	GtkComboBox *field_box;

	menu = gtk_menu_new ();
	menu_item = gtk_menu_item_new_with_label (_("-Unused-"));
	gtk_widget_show (menu_item);
	ret_value = level * 100;
	g_signal_connect (GTK_OBJECT (menu_item),
		"select", G_CALLBACK (field_change), &ret_value);
	gtk_menu_attach (GTK_MENU (menu), menu_item, 0, 1, 0, 1);

	/* add the individual fields to the menu */
	field_box = GTK_COMBO_BOX (gtk_combo_box_new_text ());
	/** \todo replace with a g_list_foreach */
	for (fieldx = 0; fieldx <= tab->file->last_field; fieldx++)
	{
		ret_value = level * 100 + fieldx + 1;
		gtk_combo_box_insert_text (field_box, ret_value,
			tab->file->fields[fieldx].name);
	}
	gtk_widget_show (GTK_WIDGET (field_box));
	g_signal_connect (GTK_OBJECT (field_box),
		"changed", G_CALLBACK (field_change), NULL);
	if (edit_sort->line[level].field >= 0)
		gtk_combo_box_set_active (field_box, edit_sort->line[level].field);
	/* Pack Widgets into boxes */
	temp_box = gtk_hbox_new (FALSE, 5);

	/* now ascending and descending buttons */
	/* ad_vbox = gtk_vbox_new (FALSE, 5); */
	ascend_button = gtk_radio_button_new_with_label 
		(NULL, _("Ascending"));
	descend_button =
		gtk_radio_button_new_with_label (gtk_radio_button_get_group
		(GTK_RADIO_BUTTON (ascend_button)), _("Descending"));
	if (edit_sort->line[level].ascending)
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (ascend_button),
			1);
	else
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (descend_button),
			1);

	/* Callbacks for radio buttons */
	g_signal_connect (GTK_OBJECT (ascend_button), "clicked",
		G_CALLBACK (ascending_clicked), GINT_TO_POINTER (level));
	g_signal_connect (GTK_OBJECT (descend_button), "clicked",
		G_CALLBACK (descending_clicked), GINT_TO_POINTER (level));

	gtk_box_pack_end (GTK_BOX (temp_box), descend_button, FALSE, FALSE, 0);
	gtk_box_pack_end (GTK_BOX (temp_box), ascend_button, FALSE, FALSE, 0);
	gtk_box_pack_end (GTK_BOX (temp_box), GTK_WIDGET (field_box),
		FALSE, FALSE, 0);

	/* now right justify the name label */
	if (level)
		name_label = gtk_label_new (_("then on: "));
	else
		name_label = gtk_label_new (_("Primary sort on: "));
	gtk_box_pack_end (GTK_BOX (temp_box), name_label, FALSE, FALSE, 0);

	return (temp_box);
}								/* end of field_menu () */

/*  This routine compares two big_array elements to return hi, low, equal
and is full of bugs.
*/
static gint
sort_compare (gconstpointer fir, gconstpointer sec)
{
	gchar *local_string1;
	gchar *local_string2;
	gdouble local_double1;
	gdouble local_double2;
	gint levelx;
	gint temp_return;
	gdouble temp_double;
	QlSortRow *first = (QlSortRow *) (fir), *second = (QlSortRow *) (sec);
	for (levelx = 0; levelx < levels; levelx++)
	{
		if (level_type[levelx] == FIELD_TYPE_TEXT)
		{
			local_string1 = first->level[levelx].string;
			local_string2 = second->level[levelx].string;
			if (!local_string1)
			{
				if (!local_string2)
					continue;	/* they are equal */
				return (level_ascending[levelx]);	/* left is less than right */
			}
			/* first is not zero, maybe second is */
			if (!local_string2)
				return (-level_ascending[levelx]);

			temp_return = strcmp (local_string1, local_string2);
			if (temp_return)
				return (level_ascending[levelx] * temp_return);
		}

		else
		{						/* all numeric types */
			local_double1 = first->level[levelx].number;
			local_double2 = second->level[levelx].number;
			if ((glong)local_double1 == (glong)HUGE_VAL)
			{
				if ((glong)local_double2 == (glong)HUGE_VAL)
					continue;	/* they are equal */
				return (level_ascending[levelx]);	/* left is less than right */
			}
			/* first is not zero, maybe second is */
			if ((glong)local_double2 == (glong)HUGE_VAL)
				return (-level_ascending[levelx]);
			temp_double = first->level[levelx].number -
				second->level[levelx].number;
			if (temp_double > 0)
				return (level_ascending[levelx]);
			if (temp_double < 0)
				return (-level_ascending[levelx]);
		}
	}

	/* got through everything.  They must be equal */
	return 0;
}

/* 1.  Build an array of elements to sort
  2.  Sort them
  3.  Copy the sorted rows to the end of the sheet
  4.  Delete the original sorted rows  */
static void
do_the_nasty_sort (QlTabData * tab)
{
	gint fieldx;
	gint rows, rowx, fromrow, torow;
	gboolean visible, hold_visible;
	gint colx;
	gint levelx;
	gint local_last_field;
	gchar *text;
	gchar *hold1row[MAX_FIELDS + 2];

	for (fieldx = 0; fieldx < MAX_FIELDS; fieldx++)
		hold1row[fieldx] = NULL;
	local_last_field = tab->file->last_field;
	/* omits last row, which is blank */
	rows = tab->file->last_row;
	levels = curr_sort->line_ct;
	if (rows < 2)
		return;
	for (levelx = 0; levelx < levels; levelx++)
	{
		fieldx = curr_sort->line[levelx].field;
		curr_sort->line[levelx].column = tab->file->fields[fieldx].sheet_column;
		level_type[levelx] = tab->file->fields[fieldx].type;
		level_formatting[levelx] = tab->file->fields[fieldx].formatting;
		if (curr_sort->line[levelx].ascending)
			level_ascending[levelx] = 1;
		else
			level_ascending[levelx] = -1;
	}

	/* go get enough storage to sort this mess */
	big_array = (QlSortRow *) g_malloc (rows * sizeof (QlSortRow));

	/* Start the big loop to build array that qsort will sort */
	for (rowx = 0; rowx < rows; rowx++)
	{
		big_array[rowx].old_row = rowx;
		for (levelx = 0; levelx < levels; levelx++)
		{
			text = gtk_sheet_cell_get_text (tab->view->sheet, rowx,
				curr_sort->line[levelx].column);
			if (text)
				switch (level_type[levelx])
				{
				case FIELD_TYPE_TEXT:
					big_array[rowx].level[levelx].string = text;
					break;
				default:
					big_array[rowx].level[levelx].number =
						qls2d (text, level_type[levelx],
						level_formatting[levelx]);
					break;
				}
			else
			{
				if (level_type[levelx] == FIELD_TYPE_TEXT)
					big_array[rowx].level[levelx].string = NULL;
				else
					big_array[rowx].level[levelx].number = HUGE_VAL;
			}
		}
	}							/* end of building array of pointers */

	/* this is sure easy.  sort them */
	qsort (big_array, (size_t) rows, sizeof (QlSortRow), sort_compare);

	/* rows are rearranged.  move them in sheet */
	big_draw_start (tab);

	/* the plan is to move aside one row, then move in the row that goes 
	   there, then move in the next row into the vacated position, and so on */
	for (rowx = 0; rowx < rows; rowx++)
	{
		if (big_array[rowx].old_row < 0)	/* already moved */
			continue;
		if (rowx == big_array[rowx].old_row)	/* this row doesn't move */
			continue;

		/* save this first row */
		hold_visible = row_is_visible (tab, rowx);
		for (colx = 0; colx <= local_last_field; colx++)
		{
			text = g_strdup (gtk_sheet_cell_get_text 
				(tab->view->sheet, rowx, colx));
			if (text)
			{
				hold1row[colx] = text;
				gtk_sheet_cell_clear (tab->view->sheet, rowx,
					colx);
			}
		}
		torow = rowx;
		fromrow = big_array[rowx].old_row;
		big_array[rowx].old_row = -1;

		/* copy row that goes here from where it is now */
		do
		{
			visible = row_is_visible (tab, fromrow);
			gtk_sheet_row_set_visibility (tab->view->sheet,
				torow, visible);
			for (colx = 0; colx <= local_last_field; colx++)
			{
				text = g_strdup (gtk_sheet_cell_get_text 
					(tab->view->sheet, fromrow, colx));
				if (text)
				{
					gtk_sheet_set_cell_text (tab->view->sheet,
						torow, colx, text);
					gtk_sheet_cell_clear (tab->view->sheet,
						fromrow, colx);
				}
				g_free (text);
			}
			torow = fromrow;
			fromrow = big_array[fromrow].old_row;
			big_array[torow].old_row = -1;
		}
		while (big_array[fromrow].old_row >= 0);

		/* got to the end of a chain, put back the stored row */
		gtk_sheet_row_set_visibility (tab->view->sheet,
			torow, hold_visible);
		for (colx = 0; colx <= local_last_field; colx++)
		{
			if (hold1row[colx])
			{
				gtk_sheet_set_cell_text (tab->view->sheet, torow,
					colx, hold1row[colx]);
				g_free (hold1row[colx]);
				hold1row[colx] = NULL;
			}
		}
	}							/* end of big looop to move all rows */
	g_free (big_array);
	big_draw_end (tab);
	front_is_changed (tab);
}								/* end of do_the_nasty_sort () */

/*  User clicked OK to finish editing a sort layout.  What we do now
  depends on how we got here.*/
static void
ok_clicked (GtkWidget * w, gpointer G_GNUC_UNUSED entry)
{
	gchar *text;
	gint sortx;
	gboolean gotone;
	QlContext * qlc;
	QlTabData * tab;

	qlc = ql_get_context (GTK_WIDGET(w));
	tab = ql_get_tabdata (qlc);
	/* save the changed name */
	text = g_strdup (gtk_entry_get_text (GTK_ENTRY (tab->view->entry1)));
	if (check_entry (tab, text))
		return;
	front_is_changed (tab);
	strcpy (edit_sort->name, text);

	/* check to see if even one field.  Don't start moving until we
	   get past this point */
	gotone = FALSE;
	for (sortx = 0; sortx < MAX_SORT_NESTING; sortx++)
		if (edit_sort->line[sortx].field > 0)
			gotone = TRUE;

	if (!gotone)
	{
		level2_error (tab, _("You must specify at least one column "
				"on which to sort."));
		return;
	}

	/* there is at least one field.  Now clean up and convert all back to valid
	   values. */
	edit_sort->line_ct = 0;
	for (sortx = 0; sortx < MAX_SORT_NESTING; sortx++)
		if (edit_sort->line[sortx].field-- > 0)
			edit_sort->line[edit_sort->line_ct++] = edit_sort->line[sortx];

	curr_sort = edit_sort;
	do_the_nasty_sort (tab);

	if (select_mode == QL_SORT_ADD)
		tab->file->sort_ct++;
	tab->file->sort_ptr = curr_sort;
	dim_list_sort_menu (tab->qlc);
}

/* User clicked OK to select a specific sort layout.  What we do now
  depends on how we got here.*/
static void
select_ok (GtkButton * button, gpointer data)
{
	GtkTreeSelection *selection;
	GtkTreeIter iter;
	GtkTreeModel *model;
	gchar * sort_name;
	gint reportx;
	QlContext * qlc;
	QlTabData * tab;

	qlc = ql_get_context (GTK_WIDGET(button));
	tab = ql_get_tabdata (qlc);
	selection = GTK_TREE_SELECTION (data);
	if (gtk_tree_selection_get_selected (selection, &model, &iter))
	{
		reportx = -1;
		gtk_tree_model_get (model, &iter, SORT_COL, 
			&sort_name, -1);
		for (reportx = 0; reportx <= tab->file->report_ct; reportx++)
		{
			if (0 == strcmp (tab->file->sorts[reportx].name, sort_name))
			{
				tab->file->sort_ptr = &tab->file->sorts[reportx];
				curr_sort = &tab->file->sorts[reportx];
			}
		}
		if (tab->file->report_no < 0)
			return;
		do_the_nasty_sort (tab);
//		gtk_widget_destroy (dialog1_win);
		dim_list_sort_menu (tab->qlc);
	}
}

static void
build_select_box (QlTabData * tab)
{
	gint sortx;
	guint width;
	GtkWidget *top_vbox;
	GtkWidget *hbox2;			/* treeview and header */
	GtkWidget *scrwin_from, * ok_button;
	GtkTreeStore * treestore;
	GtkTreeIter parent_iter, child_iter;
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *column;
	GtkTreeView * treeview;
	GtkTreeModel * model;
	GtkTreeSortable * sort;
	GtkTreeSelection * bselect;
	GtkWidget *hbox6;			/* same as action area */
	GtkWidget *cancel_button;
	GtkWidget * select_dlg;

	select_dlg = gtk_dialog_new ();
	gtk_window_set_modal (GTK_WINDOW (select_dlg), TRUE);
	gtk_window_set_position (GTK_WINDOW (select_dlg), GTK_WIN_POS_CENTER);
	gtk_window_set_resizable (GTK_WINDOW (select_dlg), TRUE);
	gtk_container_set_border_width (GTK_CONTAINER (select_dlg), 5);
	width = 0;
	gtk_window_set_title (GTK_WINDOW (select_dlg), _("Select a sort"));
	top_vbox = GTK_DIALOG (select_dlg)->vbox;
	gtk_box_set_spacing (GTK_BOX (top_vbox), 10);
	hbox6 = GTK_DIALOG (select_dlg)->action_area;

	hbox2 = gtk_hbox_new (FALSE, 5);
	gtk_box_pack_start (GTK_BOX (top_vbox), hbox2, TRUE, TRUE, 0);

	/* The sort names tree */
	treeview = GTK_TREE_VIEW (gtk_tree_view_new ());
	model = gtk_tree_view_get_model (treeview);
	treestore = gtk_tree_store_new (SINGLE_COL, G_TYPE_STRING);
	sort = GTK_TREE_SORTABLE (treestore);
	gtk_tree_view_set_model (GTK_TREE_VIEW (treeview), 
		GTK_TREE_MODEL (sort));
	gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (treeview), TRUE);
	gtk_tree_view_set_reorderable (GTK_TREE_VIEW (treeview), TRUE);
	gtk_tree_store_append (treestore, &parent_iter, NULL);
	gtk_tree_store_set (treestore, &parent_iter, SORT_COL, 
		_("Available sort patterns"), -1);
	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes
		(_("Sort Name"), renderer, "text", SORT_COL, NULL);
	gtk_tree_view_column_set_sort_column_id (column, SORT_COL);
	gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);
	bselect = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));
	gtk_tree_selection_set_mode (bselect, GTK_SELECTION_SINGLE);

	scrwin_from = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrwin_from),
		GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);
	gtk_container_set_border_width (GTK_CONTAINER (scrwin_from), 5);
	gtk_box_pack_start (GTK_BOX (hbox2), scrwin_from, TRUE, TRUE, 0);
	gtk_container_add (GTK_CONTAINER (scrwin_from), 
		GTK_WIDGET(treeview));

	ok_button = gtk_button_new_from_stock (GTK_STOCK_OK);
	gtk_box_pack_start (GTK_BOX (hbox6), ok_button, TRUE, TRUE, 0);
	g_signal_connect (GTK_OBJECT (ok_button), "clicked",
		G_CALLBACK (select_ok), bselect);

	cancel_button = gtk_button_new_from_stock (GTK_STOCK_CANCEL);
	gtk_box_pack_start (GTK_BOX (hbox6), cancel_button, TRUE, TRUE, 0);
	g_signal_connect (GTK_OBJECT (cancel_button), "clicked",
		G_CALLBACK (close_dlg), select_dlg);
	g_signal_connect (GTK_OBJECT (select_dlg), "delete_event", 
		G_CALLBACK (close_dlg), select_dlg);

	/* now populate the tree of sort names */
	for (sortx = 0; sortx < tab->file->sort_ct; sortx++)
	{
		gtk_tree_store_append (treestore, &child_iter, &parent_iter);
		gtk_tree_store_set (treestore, &child_iter, SORT_COL, 
			tab->file->sorts[sortx].name, -1);
		width = (strlen(tab->file->sorts[sortx].name) > width) ? 
			strlen(tab->file->sorts[sortx].name) : width;
	}
	gtk_widget_set_size_request (GTK_WIDGET(treeview), 
		width + 200, 200);
	gtk_tree_view_expand_all (GTK_TREE_VIEW(treeview));
	gtk_widget_show_all (select_dlg);
}								/* End of build_select_box */

static void
display_edit_window (QlTabData * tab)
{
	GtkWidget *ok_button,
		*cancel_button, * edit_win,
		*top_box, *button_box, *temp_box, *hbox1, *name_label;
	gint sortx;

	/* let's clean up the sort rules for display purposes.
	   Even when there may be only two lines, all six display. 
	   For that reason, add 1 to all entries so that 0 displays 
	   as None and 1 displays field 0 */
	for (sortx = 0; sortx < MAX_SORT_NESTING; sortx++)
	{
		if (sortx < edit_sort->line_ct)
			edit_sort->line[sortx].field++;
		else
		{
			edit_sort->line[sortx].field = 0;
			edit_sort->line[sortx].ascending = TRUE;
		}
	}

	edit_win = gtk_dialog_new ();
	gtk_window_set_modal (GTK_WINDOW (edit_win), TRUE);
	gtk_window_set_position (GTK_WINDOW (edit_win), GTK_WIN_POS_CENTER);
	gtk_window_set_resizable (GTK_WINDOW (edit_win), TRUE);
	gtk_container_set_border_width (GTK_CONTAINER (edit_win), 5);

	gtk_window_set_title (GTK_WINDOW (edit_win),
		_("Add or edit sort rules"));
	top_box = GTK_DIALOG (edit_win)->vbox;
	button_box = GTK_DIALOG (edit_win)->action_area;
	gtk_box_set_spacing (GTK_BOX (top_box), 5);
	g_signal_connect (GTK_OBJECT (edit_win),
		"delete_event", G_CALLBACK (gtk_widget_destroy), edit_win);
	/* Show and allow changes to sort name */
	hbox1 = gtk_hbox_new (FALSE, 5);
	name_label = gtk_label_new (_("Sort Name"));
	gtk_box_pack_start (GTK_BOX (hbox1), name_label, FALSE, FALSE, 0);

	tab->view->entry1 = gtk_entry_new ();
	gtk_box_pack_start (GTK_BOX (hbox1), tab->view->entry1, TRUE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX (top_box), hbox1, FALSE, FALSE, 0);
	gtk_entry_set_text (GTK_ENTRY (tab->view->entry1), edit_sort->name);

	/* make six entries for levels of sorting */
	for (sortx = 0; sortx < MAX_SORT_NESTING; sortx++)
	{
		temp_box = field_menu (tab, sortx);
		gtk_container_add (GTK_CONTAINER (top_box), temp_box);
	}

	ok_button = gtk_button_new_from_stock (GTK_STOCK_OK);
	g_signal_connect (GTK_OBJECT (ok_button),
		"clicked", G_CALLBACK (ok_clicked), edit_win);

	cancel_button = gtk_button_new_from_stock (GTK_STOCK_CANCEL);
	g_signal_connect (GTK_OBJECT (cancel_button),
		"clicked", G_CALLBACK (close_dlg), edit_win);

	gtk_container_add (GTK_CONTAINER (button_box), ok_button);
	gtk_container_add (GTK_CONTAINER (button_box), cancel_button);
	gtk_widget_show_all (edit_win);
}								/* end of display_edit_window */

/*  Add a new sort to the file */
void
sort_add (GtkAction G_GNUC_UNUSED * a, gpointer data)
{
	QlContext * qlc;
	QlTabData * tab;

	qlc = ql_get_context (GTK_WIDGET(data));
	tab = ql_get_tabdata (qlc);
	CHECK_CHANGED(tab);
	edit_sort = g_new0(QlSortInfo, 1);
	edit_sort->name = g_strdup(_("Untitled sort"));
	select_mode = QL_SORT_ADD;
	tab->file->sort_ptr = &tab->file->sorts[tab->file->sort_ct];
	display_edit_window (tab);
}

/* Called by report.  Just sort the file */
void
sort_do_it (QlTabData * tab, gint sortno)
{
	g_return_if_fail (tab);
	CHECK_CHANGED(tab);
	curr_sort = &tab->file->sorts[sortno];
	do_the_nasty_sort (tab);
}

void
sort_by_column (GtkAction G_GNUC_UNUSED * w, gpointer data)
{
	GtkSheetRange range;
	QlContext * qlc;
	QlTabData * tab;

	qlc = ql_get_context (GTK_WIDGET(data));
	tab = ql_get_tabdata (qlc);
//	CHECK_CHANGED(tab);
	curr_sort = g_new0 (QlSortInfo, 1);
	curr_sort->line_ct = 1;
	curr_sort->line[0].field = tab->file->col_to_field[tab->view->sel_range.col0];
	curr_sort->line[0].ascending = TRUE;
	do_the_nasty_sort (tab);
	range.col0 = range.coli = tab->view->sel_range.col0;
	range.row0 = range.rowi = 0;
	gtk_sheet_select_range (tab->view->sheet, &range);
}

void
sort_delete (GtkAction G_GNUC_UNUSED * w, gpointer data)
{
	QlContext * qlc;
	QlTabData * tab;

	qlc = ql_get_context (GTK_WIDGET(data));
	tab = ql_get_tabdata (qlc);
	CHECK_CHANGED(tab);
	select_mode = QL_SORT_DELETE;
	build_select_box (tab);
}

void
sort_edit (GtkAction G_GNUC_UNUSED * w, gpointer data)
{
	QlContext * qlc;
	QlTabData * tab;

	qlc = ql_get_context (GTK_WIDGET(data));
	tab = ql_get_tabdata (qlc);
	CHECK_CHANGED(tab);
	select_mode = QL_SORT_EDIT;
	build_select_box (tab);
}

void
sort_apply (GtkWidget G_GNUC_UNUSED * w, gpointer data)
{
	QlContext * qlc;
	QlTabData * tab;

	qlc = ql_get_context (GTK_WIDGET(data));
	tab = ql_get_tabdata (qlc);
	CHECK_CHANGED(tab);
	select_mode = QL_SORT_APPLY;
	build_select_box (tab);
}
