#ifndef QUTDES_H
#define QUTDES_H

#include <string.h>
#include <stdlib.h>
#include <time.h>
#include <assert.h>

//Calculate the number of bytes required for the CRYPTOGRAM given the number
//of bytes in the text.
//Based on multiples of 8 for as many blocks of 8 there are in the text plus
//another 8 bytes if CBC is used plus another 8 bytes if padding is required
//and the number of bytes in the text ends on an 8 byte boundary
#define CRYPTSIZE(bPadding, bCBC, cbText)                                   \
    ((((cbText)/8) + (((cbText) % 8 > 0) ? 1 : ((bPadding) ? 1 : 0)) +      \
    ((bCBC) ? 1 : 0)) * 8)

//Calculate the number of bytes required for the TEXT given the number of
//bytes IN THE CRYPTOGRAM.
//The padding flag is ignored - if padding has been used in the cryptogram
//then the 8 bytes are not subtracted because there is no way of knowing
//whether the padding has meant using the extra 8 bytes without looking at
//the decrypted crypt's content.
#define TEXTSIZEFROMCRYPT(bPadding, bCBC, cbCrypt)                       \
    ((((cbCrypt)/8) - ((bCBC) ? 1 : 0)) * 8)

//Calculate the number of bytes required for the TEXT given the initial
//number of bytes IN THE TEXT.
//This is required to allow the additional 8 bytes when for padding is used.
#define TEXTSIZE(bPadding, bCBC, cbText)                                 \
    TEXTSIZEFROMCRYPT(bPadding, bCBC, CRYPTSIZE(bPadding, bCBC,          \
    cbText))

//Macros for manipulating bits
#define GETBIT(lpBytes, nBitzb)                                          \
    ((lpBytes)[(nBitzb)/8] & (1 << (7 - ((nBitzb) % 8))))
#define SETBIT(lpBytes, nBitzb)                                          \
    ((lpBytes)[(nBitzb)/8] |= (1 << (7 - ((nBitzb) % 8))))
#define CLEARBIT(lpBytes, nBitzb)                                        \
    ((lpBytes)[(nBitzb)/8] &= ~(1 << (7 - ((nBitzb) % 8))))

#define Rotate26BitsLeft2(lIn, lOut)                                     \
    (lOut) = ((lIn) << 2) | (((lIn) & 0x0C000000) >> 26);                                  

#define Rotate26BitsLeft1(lIn, lOut)                                     \
    (lOut) = ((lIn) << 1) | (((lIn) & 0x08000000) >> 27);

#define GetKeyPair(lpBytes, lC, lD)                                      \
	lpBytes[0] = ((lC) & 0xFF00000) >> 20;                               \
    lpBytes[1] = ((lC) & 0x00FF000) >> 12;                               \
    lpBytes[2] = ((lC) & 0x0000FF0) >> 4;                                \
    lpBytes[3] = (((lC) & 0x000000F) << 4) | (((lD) & 0xF000000) >> 24); \
    lpBytes[4] = ((lD) & 0x0FF0000) >> 16;                               \
    lpBytes[5] = ((lD) & 0x000FF00) >> 8;                                \
    lpBytes[6] = ((lD) & 0x00000FF);

// class to implement triple DES.
class QutDES
{
public:
	//Constructor.
    QutDES() : m_lpOutput(0), m_cbOutput(0), m_bOutputIsOwned(true), 
        m_bPadding(true), m_bCBC(true), m_nIterations(3), m_cbResult(0)
    {
#ifdef _DEBUG
     #define DESKEYMISSING "_MARTYN_"

        SetKey1((unsigned char *)DESKEYMISSING);
        SetKey2((unsigned char *)DESKEYMISSING);

    #define ASSERTDESKEYMISSING(lpKey)                                    \
        assert(memcmp(lpKey, DESKEYMISSING, 8) != 0);

#else // _DEBUG

    #define ASSERTDESKEYMISSING(lpKey)

#endif // _DEBUG
    }

	~QutDES()
    {
        ClearOutput();        
    }

    //Set whether we are using PKCS#5 (on by default) which is a good idea
    //because it allows use to determine the exact length of the original
    //message when decrypting
	inline void SetPadding(bool bPadding)
    {
        m_bPadding = bPadding;
    }

    //Set whether we are using chain block coding (CBC) which is a stronger
    //form of encryption because the results of each 64 bit block encryption 
    //are used for the next (on by default)
	inline void SetCBC(bool bCBC)
    {
        m_bCBC = bCBC;
    }

    //Whether we are using single or triple des (triple des by default)
	inline void SetDES() { SetIterations(1); }
    inline void SetTripleDES() { SetIterations(3); }

    //Set the first key (required for SINGLE des and TRIPLE des).
	inline void SetKey1(unsigned char * lpsz8Bytes)
    {
        memcpy(m_key1, lpsz8Bytes, sizeof(m_key1));
    }

    //Set the second key (required for TRIPLE des).
	inline void SetKey2(unsigned char * lpsz8Bytes)
    {
        memcpy(m_key2, lpsz8Bytes, sizeof(m_key2));
    }
    
    //Set the output buffer if you want to own the memory used for the 
    //results.  This should be large enough to store the results.
    //Call with 0 for either parameter to revert back to allowing the class
    //to manage the memory (which is the default).
	inline void SetOutputBuffer(unsigned char * lpOutput, 
        unsigned long cbOutput)
    {        
        ClearOutput();

        if (lpOutput && cbOutput)
        {
            m_lpOutput = lpOutput;
            m_cbOutput = cbOutput;
            m_bOutputIsOwned = false;
        }
    }

    //Access the cryptogram and plain text.
	inline const unsigned char * GetCryptogram() const 
        { return m_lpOutput; }

    inline const unsigned char * GetPlainText() const 
        { return m_lpOutput; }

    inline unsigned long GetCryptogramSize() const 
        { return m_cbResult; }

    inline unsigned long GetPlainTextSize() const 
        { return m_cbResult; }

    //Other accessors for completness.
	inline const unsigned char * GetKey1() const 
        { ASSERTDESKEYMISSING(m_key1) return m_key1; }

    inline const unsigned char * GetKey2() const 
        { ASSERTDESKEYMISSING(m_key2) return m_key2; }

    inline bool GetPadding() const { return m_bPadding; }
    inline bool GetCBC() const { return m_bCBC; }

    //Calculate the cryptogram size based on:
    //1.  The size of the plaintext specified as the parameter,
    //2.  Whether PKCS#5 padding is enabled (see SetPadding)
    //3.  Whether CBC is enabled (see SetCBC).
	inline unsigned long CalcCryptogramSize(unsigned long cbPlainText)
    {
        return CRYPTSIZE(GetPadding(), GetCBC(), cbPlainText);                
    }

    //Calculate size of buffer LARGE ENOUGH to hold results of decrypting
    //a cryptogram based on:
    //1.  The size of the cryptogram specified as the parameter,
    //2.  Whether PKCS#5 padding is enabled (see SetPadding)
    //3.  Whether CBC is enabled (see SetCBC).
    //Note that the actual size returned my be smaller when padding is 
    //enabled, this can be determined by calling GetPlainTextSize() after
    //decryption.
	unsigned long CalcPlainTextSize(unsigned long cbCryptogram)
    {
        TEXTSIZEFROMCRYPT(GetPadding(), GetCBC(), cbCryptogram);
    }

    //Encrypt plaintext - if successful this returns true.  The cryptogram
    //can then be obtained by calling GetCryptogram() and 
    //GetCryptogramSize().
    //Failures are usually because the output buffer isn't large enough.
	inline bool Encrypt(const unsigned char * lpPlainText, 
        unsigned long cbPlainText)
    {
        return DoDES(lpPlainText, cbPlainText, true);
    }

    inline bool Encrypt(const char * lpszPlainText)        
    {
        unsigned long cbPlainText = strlen(lpszPlainText);
        return DoDES((unsigned char *)lpszPlainText, cbPlainText, true);
    }

    //Decrypt cryptogram - if successful this returns true.  The plaintext
    //can then be obtained by calling GetPlainText() and 
    //GetPlainTextSize().
    //Failures are usually because the output buffer isn't large enough or
    //that a different mode of DES was used for encryption.
	inline bool Decrypt(const unsigned char * lpCryptogram, 
        unsigned long cbCryptogram)
    {
        return DoDES(lpCryptogram, cbCryptogram, false);
    }    

protected:

    //Return the size of the output buffer which is not necessarily the same
    //tas the size of the result
	inline unsigned long GetOutputSize() const 
        { return m_cbOutput; }

    //Set the number of iterations (1 for single des, 3 for triple des).
	inline void SetIterations(unsigned long nIterations)
    {
        m_nIterations = nIterations;
    }

    //Return the total number of iterations.
	inline unsigned long GetIterations() const { return m_nIterations; }

    //Return whether we own the output memory.
	inline bool GetOutputIsOwned() const { return m_bOutputIsOwned; }

    //Clear the cryptogram.
	inline void ClearOutput()
    {
        if (GetOutputIsOwned())
        {
            if (m_lpOutput)
            {
                assert(GetOutputSize());
                delete [] m_lpOutput;                
            }
        }
        else
        {
            m_bOutputIsOwned = true;
        }

        m_lpOutput = 0;
        m_cbOutput = 0;
        m_cbResult = 0;
    }

    //Resize the output buffer - should only be called if we own the memory.
	inline void ResizeOutput(unsigned long cb)
    {
        assert(GetOutputIsOwned());

        ClearOutput();
        assert(cb);

        m_lpOutput = new unsigned char[cb+1];
        m_lpOutput[cb] = 0;
        m_cbOutput = cb;        
    }

    //Performs single/triple DES encryption/decryption and places the results 
    //in an out buffer.  Triple DES is just DES done two times with two keys 
    //in a particular order
	bool DoDES(const unsigned char * lpIn, unsigned long cbIn, 
        bool bEncrypt);

    //For a given in buffer, out buffer and table this function maps bits 
    //which are true given that the entries in the table show the new 
    //arrangement of the bits from their initial order
	static void MapTrueBits(const unsigned char * lpMap, 
        unsigned int cbMap, const unsigned char * lpIn, 
        unsigned char * const lpOut);

    //Encode 64 bits of data.
	static void Encode64Bits(const unsigned char in[8], 
        unsigned char out[8], const unsigned char SubKeys[16][6], 
        int nEncrypt);

    //Function for encrypting each RIGHT half of each block of encryption 
    //data by expanding the in data from 32 bits to 48; exclusively ORing 
    //(XOR) these 48 bits with the Subkey, passing this data through the SBOX 
    //to return back to 32 bits and applying a final table permutation
	static void MapThroughSBox(const unsigned char Right[4], 
        const unsigned char SubKey[6], unsigned char output32[4]);

    //Obtain nibble from the SBox table
	static unsigned char GetSBoxNibble(unsigned char sbox, 
        unsigned int idxSBox);

    //Create array of 16 subkeys from the key
	static void CreateSubKeys(const unsigned char Key[8], 
        unsigned char SubKeys[16][6]);

    //For a given key, create the permuted K+ based on table PC1
	static void CreateKeyPlus(const unsigned char Key[8], 
        unsigned char KeyPlus[7]);

    //Create the Initial permuation from the message.
	static void CreateInitialPermutation(const unsigned char Message[8], 
        unsigned char IP[8]);

    //Variables
    unsigned char m_key1[8];            // key used for single & triple des.
    unsigned char m_key2[8];            // key used for triple des.
    unsigned char *m_lpOutput;          // output buffer
    unsigned long m_cbOutput;           // size of output buffer
    unsigned long m_cbResult;           // size of the result.
    bool          m_bOutputIsOwned;     // whether we own the output buffer.
    bool          m_bEncrypting;        // whether encrypting or decrypting.
    bool          m_bPadding;           // whether were using PKCS#5 padding.
    bool          m_bCBC;               // whether were using CBC encryption.
    unsigned long m_nIterations;        // single or triple des.

};

#endif // QUTDES_H