# Thanks to Mr. Shuhei Yamamoto.

$LOAD_PATH << '..' unless $LOAD_PATH.include? '..'
require 'qwik/util-escape'
require 'qwik/util-charset'

module Qwik
  class Filename
    def self.encode(str)
      str = str.to_filename_charset
      str = str.gsub(/([^ 0-9A-Za-z_.\/-]+)/n) {
	'=' + $1.unpack('H2' * $1.size).join('=').upcase
      }
      return str
    end

    def self.decode(str)
      str = str.gsub(/((?:=[0-9a-fA-F]{2})+)/n) {
	[$1.delete('=')].pack('H*')
      }
      str = str.to_filename_charset
      return str
    end

    def self.contain_multibyte?(filename)
      filename.each_byte {|byte|
	if 0x7f < byte || byte == ?\e
	  return true
	end
      }
      return false
    end

    def self.extname(filename)
      return File.extname(filename).sub(/\A\./, '')
    end

    private

    ALLOWABLE_CHARACTERS_RE = /\A[ 0-9A-Za-z_.-]+\z/

    def self.allowable_characters?(f)
      return true if ALLOWABLE_CHARACTERS_RE =~ f
      return false
    end
  end
end

if $0 == __FILE__
  require 'qwik/test-common'
  $test = true
end

if defined?($test) && $test
  class TestFilename < Test::Unit::TestCase
    include TestSession

    def test_all
      c = Qwik::Filename

      # test_encode
      ok_eq('t', c.encode('t'))
      ok_eq(' ', c.encode(' '))
      ok_eq('=E3=81=82', c.encode("\202\240"))
      ok_eq('=E3=81=82.txt', c.encode("\202\240.txt"))

      # test_decode
      ok_eq('t', c.decode('t'))
      ok_eq(' ', c.decode(' '))
      ok_eq("\343\201\202", c.decode('=E3=81=82'))
      ok_eq("\343\201\202.txt", c.decode('=E3=81=82.txt'))

      # test_contain_multibyte?
      ok_eq(false, c.contain_multibyte?('t'))
      ok_eq(false, c.contain_multibyte?('t t'))
      ok_eq(true,  c.contain_multibyte?("\202\240"))

      # test_allowable_characters?
      ok_eq(true,  c.allowable_characters?('t'))
      ok_eq(true,  c.allowable_characters?('t t'))
      ok_eq(true,  c.allowable_characters?('t.-_t'))
      ok_eq(true,  c.allowable_characters?('t..t'))
      ok_eq(false, c.allowable_characters?("\202\240"))

      # test_extname
      ok_eq('txt',  c.extname('t.txt'))
    end
  end
end
