// jit.h

#define DEBUG_JIT       1   // 0 no runtime checks (don't use yet)
                            // 1 Dassert (use in production build for now)
                            // 2 print jit code execution
                            // 3...9 further details

#define DEBUG_JIT_SYM   0   // do expensive symbol binding consistency checks

#if JIT_INTERNAL // definitions for internal use by jit.c and friends ---------

// We use assert where the assertion is absolutely necessary or speed is
// not important.  We use "Dassert" where speed is critical.

#if DEBUG_JIT
    #define Dassert(x) assert(x)
#else
    #define Dassert(x)
#endif

#define STRLEN          80
#define printf          Rprintf
#define RNIL            R_NilValue

typedef double (* FUNC_TYPE)();
typedef int (* IFUNC_TYPE)();

#define NBR_ELEMS(x)  (sizeof(x) / sizeof((x)[0]))

#define MAX_JITS     1000   // 1+max number of compiled jit expressions

#define MAX_EXP_LEN  1000   // 1+max jit expression len (+1 is for JIT_endop)
                            // e.g. for "1+2" this needs 4 (push,push,add,endop)

#define MAX_NEST     100    // stateStack size

#define HASH_LEN     1001   // maximum number of jitted symbols
                            // informal testing: 1001 is much better than 1000

#endif // end JIT_INTERNAL, more JIT_INTERNAL stuff below ---------------------

// These states are disjoint but specified as a bit vector to allow
// fast check for multiple flags.
// If you change these, update JITS_NAMES too.

#define JITS_IDLE            0x1   // not in a jit block
#define JITS_IN_NESTED_FUNC  0x2   // in an R function called from the jit block
#define JITS_AWAITING_LOOP   0x4   // got jit(1), waiting for a loop statement
#define JITS_IN_LOOP         0x8   // in a loop statement, waiting for jitProlog
#define JITS_COMPILING       0x10  // got a LANG SXP, generating jit ops
#define JITS_NO_AS           0x20  // JITS_COMPILING but disallow JIT_as
#define JITS_AWAITING_AS     0x40  // JITS_COMPILING but need genjitAssign()
#define JITS_TERMINATED      0x80  // early termination, set CANNOT_JIT_BIT
#define JITS_SUSPENDED       0x100 // don't gen instructions for the moment

// states where compilation is taking place, even if currently suspended
#define JITS_COMPILING_STATES (JITS_COMPILING | JITS_NO_AS |        \
                               JITS_AWAITING_AS | JITS_TERMINATED | \
                               JITS_SUSPENDED)
// JIT opcodes
//
// Opcode suffixes
// r_r      REAL vec and REAL vec
// r_r1     REAL vec and REAL scalar
// r1_r     REAL scalar and REAL vec
// r1_r1    REAL scalar and REAL scalar
//
// The order within blocks of the opcodes below matters, see
// genjitBin() and adjustForTypes() and friends.
// If you add an opcode, extend the tables in the JIT_INTERNAL section below.

typedef enum JIT_OPCODE {
    JIT_endop,          // marks end of opcode sequence

    JIT_arg,            // holds args for previous op, never executed directly

    JIT_push,           // push LGLSXP, INTSXP or REALSXP
    JIT_pushsym,        // push SYMSXP

    JIT_uminus_r, JIT_uminus_r1, JIT_uminus_i, JIT_uminus_i1,

    JIT_not_r, JIT_not_r1, JIT_not_i, JIT_not_i1,     // not i.e. "!"

    JIT_math1_r,        // math function of one real vector argument
    JIT_math1_r1,       // ditto, but for scalar argument
    JIT_math1_i,        // math function of one integer vector argument
    JIT_math1_i1,       // ditto, but for scalar argument

    JIT_math1i_r,       // like math1 funcs above but return int not double
    JIT_math1i_r1,
    JIT_math1i_i,
    JIT_math1i_i1,

    JIT_eval,           // call eval()
    JIT_evaljit,        // call evalJit()

    JIT_if_i,           // "if"       with an INTEGER condition, no else
    JIT_ifelse_i,       // "if else " with a INTEGER condition
    JIT_if_r,
    JIT_ifelse_r,

    JIT_for_i,          // "for" with INTEGER loop variable

    // opcodes for for rhs expressions like x[i]
    // e.g. JIT_subset_r_i1 is x[i] where x is a real vec and i is int scalar.
    // Currently we only support indexes of length 1, so not JIT_subset_r_r.

    JIT_subset_r_r1,
    JIT_subset_r_i1,
    JIT_subset_i_r1,
    JIT_subset_i_i1,

    // opcodes for for rhs expressions like x[i,j]

    JIT_subset2_r_r1_r1, JIT_subset2_r_r1_i1,
    JIT_subset2_r_i1_r1, JIT_subset2_r_i1_i1,
    JIT_subset2_i_r1_r1, JIT_subset2_i_r1_i1,
    JIT_subset2_i_i1_r1, JIT_subset2_i_i1_i1,

    // we generate assign ops only when both sides have the same length

    JIT_as_r_r, JIT_as_r_r1, JIT_as_r1_r, JIT_as_r1_r1,
    JIT_as_r_i, JIT_as_r_i1, JIT_as_r1_i, JIT_as_r1_i1,
    JIT_as_i_r, JIT_as_i_r1, JIT_as_i1_r, JIT_as_i1_r1,
    JIT_as_i_i, JIT_as_i_i1, JIT_as_i1_i, JIT_as_i1_i1,

    // Subassign opcodes: subscript naming x[i] = y is JIT_subas_x_i_y.
    // Only indexes of length 1 are supported.

    JIT_subas_r_r1_r, JIT_subas_r_r1_i, JIT_subas_r_i1_r, JIT_subas_r_i1_i,
    JIT_subas_i_r1_r, JIT_subas_i_r1_i, JIT_subas_i_i1_r, JIT_subas_i_i1_i,

    // binary ops

    JIT_add_r_r, JIT_add_r_r1, JIT_add_r1_r, JIT_add_r1_r1,
    JIT_add_r_i, JIT_add_r_i1, JIT_add_r1_i, JIT_add_r1_i1,
    JIT_add_i_r, JIT_add_i_r1, JIT_add_i1_r, JIT_add_i1_r1,
    JIT_add_i_i, JIT_add_i_i1, JIT_add_i1_i, JIT_add_i1_i1,

    JIT_sub_r_r, JIT_sub_r_r1, JIT_sub_r1_r, JIT_sub_r1_r1,
    JIT_sub_r_i, JIT_sub_r_i1, JIT_sub_r1_i, JIT_sub_r1_i1,
    JIT_sub_i_r, JIT_sub_i_r1, JIT_sub_i1_r, JIT_sub_i1_r1,
    JIT_sub_i_i, JIT_sub_i_i1, JIT_sub_i1_i, JIT_sub_i1_i1,

    JIT_mul_r_r, JIT_mul_r_r1, JIT_mul_r1_r, JIT_mul_r1_r1,
    JIT_mul_r_i, JIT_mul_r_i1, JIT_mul_r1_i, JIT_mul_r1_i1,
    JIT_mul_i_r, JIT_mul_i_r1, JIT_mul_i1_r, JIT_mul_i1_r1,
    JIT_mul_i_i, JIT_mul_i_i1, JIT_mul_i1_i, JIT_mul_i1_i1,

    // div always return REAL even when args are INTs, for
    // compatibility with existing R code in arithmetic.c.

    JIT_div_r_r, JIT_div_r_r1, JIT_div_r1_r, JIT_div_r1_r1,
    JIT_div_r_i, JIT_div_r_i1, JIT_div_r1_i, JIT_div_r1_i1,
    JIT_div_i_r, JIT_div_i_r1, JIT_div_i1_r, JIT_div_i1_r1,
    JIT_div_i_i, JIT_div_i_i1, JIT_div_i1_i, JIT_div_i1_i1,

    // pow always return REAL even when args are INTs

    JIT_pow_r_r, JIT_pow_r_r1, JIT_pow_r1_r, JIT_pow_r1_r1,
    JIT_pow_r_i, JIT_pow_r_i1, JIT_pow_r1_i, JIT_pow_r1_i1,
    JIT_pow_i_r, JIT_pow_i_r1, JIT_pow_i1_r, JIT_pow_i1_r1,
    JIT_pow_i_i, JIT_pow_i_i1, JIT_pow_i1_i, JIT_pow_i1_i1,

    JIT_mod_r_r, JIT_mod_r_r1, JIT_mod_r1_r, JIT_mod_r1_r1,
    JIT_mod_r_i, JIT_mod_r_i1, JIT_mod_r1_i, JIT_mod_r1_i1,
    JIT_mod_i_r, JIT_mod_i_r1, JIT_mod_i1_r, JIT_mod_i1_r1,
    JIT_mod_i_i, JIT_mod_i_i1, JIT_mod_i1_i, JIT_mod_i1_i1,

    JIT_idiv_r_r, JIT_idiv_r_r1, JIT_idiv_r1_r, JIT_idiv_r1_r1,
    JIT_idiv_r_i, JIT_idiv_r_i1, JIT_idiv_r1_i, JIT_idiv_r1_i1,
    JIT_idiv_i_r, JIT_idiv_i_r1, JIT_idiv_i1_r, JIT_idiv_i1_r1,
    JIT_idiv_i_i, JIT_idiv_i_i1, JIT_idiv_i1_i, JIT_idiv_i1_i1,

    JIT_eq_r_r, JIT_eq_r_r1, JIT_eq_r1_r, JIT_eq_r1_r1,
    JIT_eq_r_i, JIT_eq_r_i1, JIT_eq_r1_i, JIT_eq_r1_i1,
    JIT_eq_i_r, JIT_eq_i_r1, JIT_eq_i1_r, JIT_eq_i1_r1,
    JIT_eq_i_i, JIT_eq_i_i1, JIT_eq_i1_i, JIT_eq_i1_i1,

    JIT_ne_r_r, JIT_ne_r_r1, JIT_ne_r1_r, JIT_ne_r1_r1,
    JIT_ne_r_i, JIT_ne_r_i1, JIT_ne_r1_i, JIT_ne_r1_i1,
    JIT_ne_i_r, JIT_ne_i_r1, JIT_ne_i1_r, JIT_ne_i1_r1,
    JIT_ne_i_i, JIT_ne_i_i1, JIT_ne_i1_i, JIT_ne_i1_i1,

    JIT_lt_r_r, JIT_lt_r_r1, JIT_lt_r1_r, JIT_lt_r1_r1,
    JIT_lt_r_i, JIT_lt_r_i1, JIT_lt_r1_i, JIT_lt_r1_i1,
    JIT_lt_i_r, JIT_lt_i_r1, JIT_lt_i1_r, JIT_lt_i1_r1,
    JIT_lt_i_i, JIT_lt_i_i1, JIT_lt_i1_i, JIT_lt_i1_i1,

    JIT_le_r_r, JIT_le_r_r1, JIT_le_r1_r, JIT_le_r1_r1,
    JIT_le_r_i, JIT_le_r_i1, JIT_le_r1_i, JIT_le_r1_i1,
    JIT_le_i_r, JIT_le_i_r1, JIT_le_i1_r, JIT_le_i1_r1,
    JIT_le_i_i, JIT_le_i_i1, JIT_le_i1_i, JIT_le_i1_i1,

    JIT_gt_r_r, JIT_gt_r_r1, JIT_gt_r1_r, JIT_gt_r1_r1,
    JIT_gt_r_i, JIT_gt_r_i1, JIT_gt_r1_i, JIT_gt_r1_i1,
    JIT_gt_i_r, JIT_gt_i_r1, JIT_gt_i1_r, JIT_gt_i1_r1,
    JIT_gt_i_i, JIT_gt_i_i1, JIT_gt_i1_i, JIT_gt_i1_i1,

    JIT_ge_r_r, JIT_ge_r_r1, JIT_ge_r1_r, JIT_ge_r1_r1,
    JIT_ge_r_i, JIT_ge_r_i1, JIT_ge_r1_i, JIT_ge_r1_i1,
    JIT_ge_i_r, JIT_ge_i_r1, JIT_ge_i1_r, JIT_ge_i1_r1,
    JIT_ge_i_i, JIT_ge_i_i1, JIT_ge_i1_i, JIT_ge_i1_i1,

    JIT_last,           // for sanity checking, never executed by JIT machine
} JIT_OPCODE;

#if JIT_INTERNAL // definitions for internal use by jit.c and friends ---------

static const char *JITS_NAMES[] = {
    "JITS_IDLE",
    "JITS_IN_NESTED_FUNC",
    "JITS_AWAITING_LOOP",
    "JITS_IN_LOOP",
    "JITS_COMPILING",
    "JITS_NO_AS",
    "JITS_AWAITING_AS",
    "JITS_TERMINATED",
    "JITS_SUSPENDED",
};

// return a string corresponding to the bit set in state

static R_INLINE const char *jitStateName(unsigned state)
{
    int i;
    assert(state);              // a bit must be set
    for (i = 0; !(state & 1); i++)
        state >>= 1;
    assert((state & ~1) == 0);  // only one bit should be set
    return JITS_NAMES[i];
}

typedef struct JIT_OP {  // jit instruction (RA_TODO could be unionized)
    JIT_OPCODE  opcode;  // e.g. JIT_push
    SEXP        operand; // e.g. SEXP being pushed by JIT_push
                         //      or pointer to symbol binding for JIT_pushsym
                         //      or ...
    FUNC_TYPE   func;    // math function returning a double e.g. sin()
    IFUNC_TYPE  ifunc;   // function returning an int e.g. abs with int arg
    int         n;       // memcpy size for assign ops
    SEXP        result;  // buffer to store the result e.g. for JIT_add_r_r
                         // also used by JIT_arg
    SEXP        sym;     // symbol for JIT_pushsym and evalJitFor (for check)
    SEXP        env;     // environment for JIT_eval, JIT_pushsym, etc.
} JIT_OP;

typedef struct JIT_RECORD {
    SEXP    original;       // pointer to SEXP holding exp before it was jitted
    SEXP    ans;            // buf to store the result of evaluating jit exp
    JIT_OP  ops[MAX_EXP_LEN];
} JIT_RECORD;

#if JIT_NAMES
static const char *JIT_OPCODE_NAMES[] = {
    "JIT_endop",

    "JIT_arg",

    "JIT_push",
    "JIT_pushsym",

    "JIT_uminus_r", "JIT_uminus_r1", "JIT_uminus_i", "JIT_uminus_i1",

    "JIT_not_r", "JIT_not_r1", "JIT_not_i", "JIT_not_i1",

    "JIT_math1_r", "JIT_math1_r1", "JIT_math1_i", "JIT_math1_i1",
    "JIT_math1i_r", "JIT_math1i_r1", "JIT_math1i_i", "JIT_math1i_i1",

    "JIT_eval",
    "JIT_evaljit",

    "JIT_if_i",
    "JIT_ifelse_i",
    "JIT_if_r",
    "JIT_ifelse_r",

    "JIT_for_i",

    "JIT_subset_r_r1",
    "JIT_subset_r_i1",
    "JIT_subset_i_r1",
    "JIT_subset_i_i1",

    "JIT_subset2_r_r1_r1", "JIT_subset2_r_r1_i1",
    "JIT_subset2_r_i1_r1", "JIT_subset2_r_i1_i1",
    "JIT_subset2_i_r1_r1", "JIT_subset2_i_r1_i1",
    "JIT_subset2_i_i1_r1", "JIT_subset2_i_i1_i1",

    "JIT_as_r_r",  "JIT_as_r_r1",
    "JIT_as_r1_r", "JIT_as_r1_r1",
    "JIT_as_r_i",  "JIT_as_r_i1",
    "JIT_as_r1_i", "JIT_as_r1_i1",
    "JIT_as_i_r",  "JIT_as_i_r1",
    "JIT_as_i1_r", "JIT_as_i1_r1",
    "JIT_as_i_i",  "JIT_as_i_i1",
    "JIT_as_i1_i", "JIT_as_i1_i1",

    "JIT_subas_r_r1_r", "JIT_subas_r_r1_i",
    "JIT_subas_r_i1_r", "JIT_subas_r_i1_i",
    "JIT_subas_i_r1_r", "JIT_subas_i_r1_i",
    "JIT_subas_i_i1_r", "JIT_subas_i_i1_i",

    "JIT_add_r_r", "JIT_add_r_r1", "JIT_add_r1_r", "JIT_add_r1_r1",
    "JIT_add_r_i", "JIT_add_r_i1", "JIT_add_r1_i", "JIT_add_r1_i1",
    "JIT_add_i_r", "JIT_add_i_r1", "JIT_add_i1_r", "JIT_add_i1_r1",
    "JIT_add_i_i", "JIT_add_i_i1", "JIT_add_ii_1", "JIT_add_i1_i1",

    "JIT_sub_r_r", "JIT_sub_r_r1", "JIT_sub_r1_r", "JIT_sub_r1_r1",
    "JIT_sub_r_i", "JIT_sub_r_i1", "JIT_sub_r1_i", "JIT_sub_r1_i1",
    "JIT_sub_i_r", "JIT_sub_i_r1", "JIT_sub_i1_r", "JIT_sub_i1_r1",
    "JIT_sub_i_i", "JIT_sub_i_i1", "JIT_sub_ii_1", "JIT_sub_i1_i1",

    "JIT_mul_r_r", "JIT_mul_r_r1", "JIT_mul_r1_r", "JIT_mul_r1_r1",
    "JIT_mul_r_i", "JIT_mul_r_i1", "JIT_mul_r1_i", "JIT_mul_r1_i1",
    "JIT_mul_i_r", "JIT_mul_i_r1", "JIT_mul_i1_r", "JIT_mul_i1_r1",
    "JIT_mul_i_i", "JIT_mul_i_i1", "JIT_mul_ii_1", "JIT_mul_i1_i1",

    "JIT_div_r_r", "JIT_div_r_r1", "JIT_div_r1_r", "JIT_div_r1_r1",
    "JIT_div_r_i", "JIT_div_r_i1", "JIT_div_r1_i", "JIT_div_r1_i1",
    "JIT_div_i_r", "JIT_div_i_r1", "JIT_div_i1_r", "JIT_div_i1_r1",
    "JIT_div_i_i", "JIT_div_i_i1", "JIT_div_ii_1", "JIT_div_i1_i1",

    "JIT_pow_r_r", "JIT_pow_r_r1", "JIT_pow_r1_r", "JIT_pow_r1_r1",
    "JIT_pow_r_i", "JIT_pow_r_i1", "JIT_pow_r1_i", "JIT_pow_r1_i1",
    "JIT_pow_i_r", "JIT_pow_i_r1", "JIT_pow_i1_r", "JIT_pow_i1_r1",
    "JIT_pow_i_i", "JIT_pow_i_i1", "JIT_pow_ii_1", "JIT_pow_i1_i1",

    "JIT_mod_r_r", "JIT_mod_r_r1", "JIT_mod_r1_r", "JIT_mod_r1_r1",
    "JIT_mod_r_i", "JIT_mod_r_i1", "JIT_mod_r1_i", "JIT_mod_r1_i1",
    "JIT_mod_i_r", "JIT_mod_i_r1", "JIT_mod_i1_r", "JIT_mod_i1_r1",
    "JIT_mod_i_i", "JIT_mod_i_i1", "JIT_mod_ii_1", "JIT_mod_i1_i1",

    "JIT_idiv_r_r", "JIT_idiv_r_r1", "JIT_idiv_r1_r", "JIT_idiv_r1_r1",
    "JIT_idiv_r_i", "JIT_idiv_r_i1", "JIT_idiv_r1_i", "JIT_idiv_r1_i1",
    "JIT_idiv_i_r", "JIT_idiv_i_r1", "JIT_idiv_i1_r", "JIT_idiv_i1_r1",
    "JIT_idiv_i_i", "JIT_idiv_i_i1", "JIT_idiv_ii_1", "JIT_idiv_i1_i1",

    "JIT_eq_r_r", "JIT_eq_r_r1", "JIT_eq_r1_r", "JIT_eq_r1_r1",
    "JIT_eq_r_i", "JIT_eq_r_i1", "JIT_eq_r1_i", "JIT_eq_r1_i1",
    "JIT_eq_i_r", "JIT_eq_i_r1", "JIT_eq_i1_r", "JIT_eq_i1_r1",
    "JIT_eq_i_i", "JIT_eq_i_i1", "JIT_eq_i1_i", "JIT_eq_i1_i1",

    "JIT_ne_r_r", "JIT_ne_r_r1", "JIT_ne_r1_r", "JIT_ne_r1_r1",
    "JIT_ne_r_i", "JIT_ne_r_i1", "JIT_ne_r1_i", "JIT_ne_r1_i1",
    "JIT_ne_i_r", "JIT_ne_i_r1", "JIT_ne_i1_r", "JIT_ne_i1_r1",
    "JIT_ne_i_i", "JIT_ne_i_i1", "JIT_ne_i1_i", "JIT_ne_i1_i1",

    "JIT_lt_r_r", "JIT_lt_r_r1", "JIT_lt_r1_r", "JIT_lt_r1_r1",
    "JIT_lt_r_i", "JIT_lt_r_i1", "JIT_lt_r1_i", "JIT_lt_r1_i1",
    "JIT_lt_i_r", "JIT_lt_i_r1", "JIT_lt_i1_r", "JIT_lt_i1_r1",
    "JIT_lt_i_i", "JIT_lt_i_i1", "JIT_lt_i1_i", "JIT_lt_i1_i1",

    "JIT_le_r_r", "JIT_le_r_r1", "JIT_le_r1_r", "JIT_le_r1_r1",
    "JIT_le_r_i", "JIT_le_r_i1", "JIT_le_r1_i", "JIT_le_r1_i1",
    "JIT_le_i_r", "JIT_le_i_r1", "JIT_le_i1_r", "JIT_le_i1_r1",
    "JIT_le_i_i", "JIT_le_i_i1", "JIT_le_i1_i", "JIT_le_i1_i1",

    "JIT_gt_r_r", "JIT_gt_r_r1", "JIT_gt_r1_r", "JIT_gt_r1_r1",
    "JIT_gt_r_i", "JIT_gt_r_i1", "JIT_gt_r1_i", "JIT_gt_r1_i1",
    "JIT_gt_i_r", "JIT_gt_i_r1", "JIT_gt_i1_r", "JIT_gt_i1_r1",
    "JIT_gt_i_i", "JIT_gt_i_i1", "JIT_gt_i1_i", "JIT_gt_i1_i1",

    "JIT_ge_r_r", "JIT_ge_r_r1", "JIT_ge_r1_r", "JIT_ge_r1_r1",
    "JIT_ge_r_i", "JIT_ge_r_i1", "JIT_ge_r1_i", "JIT_ge_r1_i1",
    "JIT_ge_i_r", "JIT_ge_i_r1", "JIT_ge_i1_r", "JIT_ge_i1_r1",
    "JIT_ge_i_i", "JIT_ge_i_i1", "JIT_ge_i1_i", "JIT_ge_i1_i1",

    "JIT_last"
};

static const char *opcodeName(JIT_OPCODE opcode)
{
    Dassert(NBR_ELEMS(JIT_OPCODE_NAMES) == JIT_last + 1);
    Dassert(0 == strcmp(JIT_OPCODE_NAMES[JIT_last], "JIT_last"));

    if (opcode >= JIT_endop && opcode <= JIT_last)
        return JIT_OPCODE_NAMES[opcode] + 4;    // +4 skips "JIT_"
    else
        return "[unknown JIT_OPCODE]";
}
#endif // JIT_NAMES

// return TRUE if op->operand is a pointer to a location i.e. a binding

static R_INLINE Rboolean isOperandALocation(JIT_OPCODE opcode)
{
    switch (opcode) {
        case JIT_pushsym:
        case JIT_as_r_r:  case JIT_as_r_r1:     // all the assigns
        case JIT_as_r1_r: case JIT_as_r1_r1:
        case JIT_as_r_i:  case JIT_as_r_i1:
        case JIT_as_r1_i: case JIT_as_r1_i1:
        case JIT_as_i_r:  case JIT_as_i_r1:
        case JIT_as_i1_r: case JIT_as_i1_r1:
        case JIT_as_i_i:  case JIT_as_i_i1:
        case JIT_as_i1_i: case JIT_as_i1_i1:
            return TRUE;
        default:
            break;
    }
    return FALSE;
}

static R_INLINE Rboolean isEvalOpcode(JIT_OPCODE opcode)
{
    return opcode == JIT_eval || opcode == JIT_evaljit;
}

static R_INLINE Rboolean isPushOpcode(JIT_OPCODE opcode)
{
    return opcode == JIT_push || opcode == JIT_pushsym;
}

// called when entering or leaving a R closure

extern unsigned jitState;
void pushJitState(CSEXP call, unsigned newState);
void popJitState(CSEXP call);

static R_INLINE void jitEnterClosure(CSEXP call)
{
    if (jitState & ~JITS_IDLE)
        pushJitState(call, JITS_IN_NESTED_FUNC);
}

static R_INLINE void jitExitClosure(CSEXP call)
{
    if (jitState & ~JITS_IDLE)
        popJitState(call);
}

extern int istack;    // index into JIT execution stack
extern int jitTrace;  // copy of R jit() "trace" parameter
extern int jitUnresolved;
char *bindingAsString(CSEXP loc);
Rboolean jitProlog(CSEXP e, const char msg[]);
SEXP evalJit(CSEXP e);
unsigned jitEnterLoopAux(CSEXP s, CSEXP body);
void checkLocationConsistency(SEXP e, SEXP env, SEXP loc, SEXP val);
void decJitUnresolved(int n);
void genjitFor(CSEXP rhs, CSEXP sym, CSEXP body, CSEXP call, CSEXP env);
void jitAllowAssign(Rboolean allow);
void jitEpilog(CSEXP resultTemplate, const char msg[]);
void jitExitLoop(CSEXP s, unsigned prevJitState);
void printBinding(CSEXP loc);
void printJitHeader(const JIT_RECORD * const prec);
void printJitOp(const JIT_OP * const op);
void printLoopIteration(const char msg[], const SEXP indexVar, int i);
void printLoopStats(void);
void setJittedBit(SEXP loc);

#endif // end JIT_INTERNAL ----------------------------------------------------

extern unsigned jitState;
extern int jitDirective;
Rboolean genjitLog(CSEXP argTemplate, CSEXP base);
Rboolean genjitMath1(double(* const func)(), CSEXP argTemplate, CSEXP ansTemplate);
SEXP do_jit(SEXP call, SEXP op, SEXP args, SEXP env);
SEXP do_nojit(SEXP call, SEXP op, SEXP args, SEXP env);
SEXP getExpBeforeItWasJitted(CSEXP s);
void checkJitBinding(CSEXP loc, CSEXP val);
void checkJitRemove(CSEXP name);
void checkJitSymBinding(CSEXP sym, CSEXP val);
void disallowIfJitting(const char str[]);
void ForwardJitNodes(SEXP *pforwarded_nodes);
void genjitAbs(CSEXP argTemplate);
void genjitAssign(CSEXP sym, SEXP loc, CSEXP y, CSEXP env, const char msg[]);
void genjitBinAux(JIT_OPCODE opcode, CSEXP x, CSEXP y, SEXPTYPE ansType);
void genjitIf(CSEXP evaluatedCondition, CSEXP result, CSEXP call, CSEXP args, CSEXP env);
void genjitLogDone(CSEXP call);
void genjitNot(JIT_OPCODE opcode, CSEXP x);
void genjitPush(SEXP e);
void genjitPushsym(SEXP loc, SEXP env);
void genjitRealBin(ARITHOP_TYPE oper, CSEXP x, CSEXP y);
void genjitRelop(RELOP_TYPE op, CSEXP x, CSEXP y);
void genjitSubas(CSEXP x, CSEXP index, CSEXP y);
void genjitSubset(CSEXP x, CSEXP subset);
void genjitSubset2(CSEXP x, CSEXP subset);
void genjitUnary(JIT_OPCODE opcode, CSEXP x);
void initJit(void);
void jitContextChangedAux(int nest);
void jitOff(Rboolean issueWarning);
void jitPreventLocalRebindingAux(CSEXP sym, CSEXP val, CSEXP env);
void jitSuspendAux(const char msg[]);
void jitUnsuspendAux(void);
void markSubassignAsNotJittable(CSEXP expr);
void popJitState(CSEXP call);
void printJit(CSEXP s);
void pushJitState(CSEXP call, unsigned newState);
#if 0   // used for development
void traceBinding(CSEXP loc, CSEXP val);
void traceSymBinding(CSEXP sym, CSEXP val);
#endif

// these inline functions are for efficiency without clutter

static R_INLINE Rboolean jitCompiling(void)
{
    return jitState & (JITS_COMPILING | JITS_NO_AS | JITS_AWAITING_AS);
}
static R_INLINE void genjitBin(JIT_OPCODE opcode, CSEXP x,
                               CSEXP y, CSEXP resultTemplate)
{
    if (jitCompiling()) {
        PROTECT(resultTemplate);
        genjitBinAux(opcode, x, y, TYPEOF(resultTemplate));
        UNPROTECT(1);
    }
}
static R_INLINE void jitContextChanged(int nest) // called by findcontext()
{
    if (jitState & ~JITS_IDLE)
        jitContextChangedAux(nest);
}
static R_INLINE void jitPreventLocalRebinding(CSEXP sym, CSEXP val, CSEXP env)
{
    if (jitState & JITS_COMPILING_STATES)
        jitPreventLocalRebindingAux(sym, val, env);
}

// temporarily prevent genjit code generation

static R_INLINE void jitSuspend(const char msg[])
{
    if (jitCompiling())
        jitSuspendAux(msg);
}
static R_INLINE void jitUnsuspend(void) // reallow jit code generation
{
    if (jitState == JITS_SUSPENDED)
        jitUnsuspendAux();
}
