# ra-changes.R
#
# This program is used when porting Ra to a new version of R.  It
# shows you which files changed etc. between and old Ra release and a
# new R release.  It does not copy any files --- it just lists what
# needs to be done
#
# How to port Ra to a new version of R:
#   Rename the current ra directory to raOLD eg. rename as ra109
#   Create a new ra directory which is a copy of the new version R sources
#   In the new Ra directory, make all recommended; make check-all
#   If you want you can copy the old ra/ra directory to the new ra now
#      but you can also do it after running ra-changes.R
#   Change into raOLD/ra and run ra-changes.R:
#       that will give you a list of files that must be manually updated
#   Following two steps can be done in various ways, below is just one way:
#       1. copy raOLD rafiles into the new ra directory 
#             (run rafiles.bat to see names of these)
#       2. manually update the files that were listed above 
#             as "must be manually updated"
#   Update the version numbers in:
#      tar-ra.bat
#      tools/GETVERSION
#      win-installer/ra.iss
#      win-installer/make.bat
#   Update NEWS in this directory
#   In the new Ra directory: make all recommended; make check-all
#   Run ra/tests/test-jit
#   Sanity check: diff new ra files against new version R sources
#   Sanity check: run ra-changes.R again and check results
#   Run ra/tar-ra
#   Run win-installer/make.bat
#   Update the ra web page (including the release date!)

trace.shell <- FALSE

old.ra.dir <- "/a/r/ra.old"
new.ra.dir <- "/a/r/ra"
old.r.dir  <- "/a/r/281"
new.r.dir  <- "/a/r/290"

# Note: you must update old.ra.files if you add new files or
# directories to RAFILES (see rafiles.bat)

files <- c(
        "tools/*",
        "src/main/*.[ch]",
        "src/main/Make*",
        "src/include/*.[ch]",
        "src/gnuwin32/Make*",
        "src/gnuwin32/front-ends/R.ico",
        "src/library/base/R/zzz.R")

stop1 <- function(...) stop(..., call.=FALSE)

sh <- function(cmd)
{
    if (trace.shell)
        cat("shell>", cmd, "\n", sep="")
    # shell(cmd, intern=TRUE, mustWork=TRUE)
    system(cmd, intern=TRUE)
}
my.ls <- function(dirs)
{
    s <- character(0)
    for (i in seq_along(dirs))
        s <- c(s, sh(paste("ls ", dirs[i], sep="")))
    s
}
expand.dir <- function(dir, files)
{
    cat("=== Expanding", dir, "\n")

    # check that dir has no backslashes
    igrep <- grep("\\\\", dir)
    if(length(igrep))
        stop1("backslashes are not allowed in \"", dir, "\"")

    # check that dir does not end with a slash
    igrep <- grep("/$", dir)
    if(length(igrep))
        stop1("last character of \"", dir, "\" is a slash")

    # check that dir exists
    if (!file.exists(dir))
        stop1(dir, " does not exist")

    # check that dir is a populated R src directory
    s <- sh(paste("ls", dir))
    igrep <- grep("COPYING", s)
    if(length(igrep) == 0)
        stop1(dir, " is not populated (could not find \"COPYING\")")
    igrep <- grep("src", s)
    if(length(igrep) == 0)
        stop1(dir, " is not populated (could not find \"src\" directory)")

    files <- paste(dir, "/", files, sep="")
    my.ls(files) # glob
}
# Get the list "changed" of files that are Ra files and 
# changed between old.files and new.files.
# This uses the global variable ra.files.

which.ra.files.changed <- function(old.files, new.files)
{
    cat("=== Doing diffs\n")
    changed <- character(0)
    ichanged <- 0
    for (i.ra.files in seq_along(ra.files)) {
	basename <- basename(ra.files[i.ra.files])
	i <- grep(basename, old.files)
	j <- grep(basename, new.files)
	if (length(i) && length(j)) {
	    if (length(i) != 1 || length(j) != 1) {
		cat("\n=== Multiple matches for", ra.files[i.ra.files],
		    "(not to worry, will add to the \"changed\" list below)\n")
		cat("=== Matches", old.files[i])
		cat("\n=== and    ",     new.files[j], "\n\n")
		changed[ichanged <- ichanged + 1] <- ra.files[i.ra.files]
	    } else {
		# -w -B means ignore white space and blank lines
		cmd <- paste("diff -w -B --brief", old.files[i], new.files[j])
		if (trace.shell)
		    cat("shell>", cmd, "\n", sep="")
		# s <- shell(cmd, mustWork=NA)
		s <- system(cmd)
		if (s)
		    changed[ichanged <- ichanged + 1] <- ra.files[i.ra.files]
	    }
	}
    }
    sort(changed)
}
# check that we are using the mingw/Rtools binaries

s <- sh("which diff")
igrep <- grep("Rtools.*bin.*diff", s)   # .* for / or \ or \\
if(length(igrep) == 0)
    stop1("you are not using the mingw/Rtools binaries\n",
          "(\"which diff\" returns \"", s, "\")\n")

old.ra.files <- expand.dir(old.ra.dir, files)
new.ra.files <- expand.dir(new.ra.dir, files)
old.r.files  <- expand.dir(old.r.dir, files)
new.r.files  <- expand.dir(new.r.dir, files)

cat("=== Ra files in ", new.ra.dir, " but not in ", old.ra.dir, ":\n", sep="")
matched <- 0
for (i in seq_along(new.ra.files)) {
    if (is.na(match(basename(new.ra.files[i]), basename(old.ra.files)))) {
        matched <- 1
        cat(new.ra.files[i], "\n", sep="")
    }
}
if (!matched)
    cat("none\n")

cat("=== Ra files in ", old.ra.dir, " but not in ", new.ra.dir, ":\n", sep="")
matched <- 0
for (i in seq_along(old.ra.files)) {
    imatch <- match(basename(old.ra.files[i]), basename(new.ra.files))
    if (is.na(imatch)) {
        matched <- 1
        cat(old.ra.files[i], "\n", sep="")
    }
}
if (!matched)
    cat("none\n")

ra.files <- sh("rafiles.bat") # get the list of files needed for Ra
ra.files <- (strsplit(ra.files, " "))[[1]]  # convert single string to a vec of strings

changed.r <- which.ra.files.changed(old.r.files, new.r.files)
changed.ra <- which.ra.files.changed(old.r.files, new.ra.files)

cat("\n=== Ra files which differ from", old.r.dir, "to", new.ra.dir, "\n")
cat(changed.ra, sep="\n")

cat("\n=== Ra files which differ from", old.r.dir, "to", new.r.dir, "\n")
cat("=== (these files must be ported by hand):\n")
cat(changed.r, sep="\n")
