% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/class_dmDSfit.R, R/class_dmSQTLfit.R
\docType{methods}
\name{plotProportions}
\alias{plotProportions}
\alias{plotProportions,dmDSfit-method}
\alias{plotProportions,dmSQTLfit-method}
\title{Plot feature proportions}
\usage{
plotProportions(x, ...)

\S4method{plotProportions}{dmDSfit}(x, gene_id, group_variable,
  plot_type = "barplot", order_features = TRUE, order_samples = TRUE,
  plot_fit = TRUE, plot_main = TRUE, group_colors = NULL,
  feature_colors = NULL)

\S4method{plotProportions}{dmSQTLfit}(x, gene_id, snp_id,
  plot_type = "boxplot1", order_features = TRUE, order_samples = TRUE,
  plot_fit = FALSE, plot_main = TRUE, group_colors = NULL,
  feature_colors = NULL)
}
\arguments{
\item{x}{\code{\linkS4class{dmDSfit}}, \code{\linkS4class{dmDStest}} or 
\code{\linkS4class{dmSQTLfit}}, \code{\linkS4class{dmSQTLtest}} object.}

\item{...}{Other parameters that can be defined by methods using this 
generic.}

\item{gene_id}{Character indicating a gene ID to be plotted.}

\item{group_variable}{Character indicating the grouping variable which is one
of the columns in the \code{samples} slot of \code{x}.}

\item{plot_type}{Character defining the type of the plot produced. Possible 
values \code{"barplot"}, \code{"boxplot1"}, \code{"boxplot2"}, 
\code{"lineplot"}, \code{"ribbonplot"}.}

\item{order_features}{Logical. Whether to plot the features ordered by their 
expression.}

\item{order_samples}{Logical. Whether to plot the samples ordered by the 
group variable. If \code{FALSE} order from the \code{sample(x)} is kept.}

\item{plot_fit}{Logical. Whether to plot the proportions estimated by the 
full model.}

\item{plot_main}{Logical. Whether to plot a title with the information about 
the Dirichlet-multinomial estimates.}

\item{group_colors}{Character vector with colors for each group defined by 
\code{group_variable}.}

\item{feature_colors}{Character vector with colors for each feature of gene
defined by \code{gene_id}.}

\item{snp_id}{Character indicating the ID of a SNP to be plotted.}
}
\value{
For a given gene, plot the observed and estimated with 
  Dirichlet-multinomial model feature proportions in each group. Estimated
  group proportions are marked with diamond shapes.
}
\description{
This plot is available only for a group design, i.e., a design that is 
equivalent to multiple group fitting.
}
\details{
In the QTL analysis, plotting of fitted proportions is deactivated 
  even when \code{plot_fit = TRUE}. It is due to the fact that neither fitted
  values nor regression coefficients are returned by the \code{dmFit}
  function as they occupy a lot of memory.
}
\examples{
# --------------------------------------------------------------------------
# Create dmDSdata object 
# --------------------------------------------------------------------------
## Get kallisto transcript counts from the 'PasillaTranscriptExpr' package

library(PasillaTranscriptExpr)
\donttest{
data_dir  <- system.file("extdata", package = "PasillaTranscriptExpr")

## Load metadata
pasilla_metadata <- read.table(file.path(data_dir, "metadata.txt"), 
header = TRUE, as.is = TRUE)

## Load counts
pasilla_counts <- read.table(file.path(data_dir, "counts.txt"), 
header = TRUE, as.is = TRUE)

## Create a pasilla_samples data frame
pasilla_samples <- data.frame(sample_id = pasilla_metadata$SampleName, 
  group = pasilla_metadata$condition)
levels(pasilla_samples$group)

## Create a dmDSdata object
d <- dmDSdata(counts = pasilla_counts, samples = pasilla_samples)

## Use a subset of genes, which is defined in the following file
gene_id_subset <- readLines(file.path(data_dir, "gene_id_subset.txt"))

d <- d[names(d) \%in\% gene_id_subset, ]

# --------------------------------------------------------------------------
# Differential transcript usage analysis - simple two group comparison 
# --------------------------------------------------------------------------

## Filtering
## Check what is the minimal number of replicates per condition 
table(samples(d)$group)

d <- dmFilter(d, min_samps_gene_expr = 7, min_samps_feature_expr = 3,
  min_gene_expr = 10, min_feature_expr = 10)

plotData(d)

## Create the design matrix
design_full <- model.matrix(~ group, data = samples(d))

## To make the analysis reproducible
set.seed(123)
## Calculate precision
d <- dmPrecision(d, design = design_full)

plotPrecision(d)

head(mean_expression(d))
common_precision(d)
head(genewise_precision(d))

## Fit full model proportions
d <- dmFit(d, design = design_full)

## Get fitted proportions
head(proportions(d))
## Get the DM regression coefficients (gene-level) 
head(coefficients(d))
## Get the BB regression coefficients (feature-level) 
head(coefficients(d), level = "feature")

## Fit null model proportions and perform the LR test to detect DTU
d <- dmTest(d, coef = "groupKD")

## Plot the gene-level p-values
plotPValues(d)

## Get the gene-level results
head(results(d))

## Plot feature proportions for a top DTU gene
res <- results(d)
res <- res[order(res$pvalue, decreasing = FALSE), ]

top_gene_id <- res$gene_id[1]

plotProportions(d, gene_id = top_gene_id, group_variable = "group")

plotProportions(d, gene_id = top_gene_id, group_variable = "group", 
  plot_type = "lineplot")

plotProportions(d, gene_id = top_gene_id, group_variable = "group", 
  plot_type = "ribbonplot")
}
}
\seealso{
\code{\link{plotData}}, \code{\link{plotPrecision}}, 
  \code{\link{plotPValues}}
}
\author{
Malgorzata Nowicka
}
