/*
 * RageIRCd: an advanced Internet Relay Chat daemon (ircd).
 * (C) 2000-2005 the RageIRCd Development Team, all rights reserved.
 *
 * This software is free, licensed under the General Public License.
 * Please refer to doc/LICENSE and doc/README for further details.
 *
 * $Id: m_map.c,v 1.38.2.2 2005/01/15 23:53:32 amcwilliam Exp $
 */

#include "config.h"
#include "struct.h"
#include "common.h"
#include "sys.h"
#include "numeric.h"
#include "msg.h"
#include "channel.h"
#include "h.h"
#include "memory.h"
#include "modules.h"
#include "xmode.h"
#include <time.h>
#include <sys/stat.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>

Module MOD_HEADER(m_map) = {
	"m_map",
	"/MAP command",
	6, "$Revision: 1.38.2.2 $"
};

int MOD_LOAD(m_map)()
{
	if (register_command(&MOD_HEADER(m_map), &CMD_MAP, m_map) == NULL) {
		return MOD_FAILURE;
	}
	return MOD_SUCCESS;
}

int MOD_UNLOAD(m_map)()
{
	return MOD_SUCCESS;
}

static void dump_map(aClient *cptr, aClient *server, char *mask, int prompt_length, int length)
{
	static char prompt[64];
	char *p = &prompt[prompt_length];
	int count = 0, local = 0;
	aClient *acptr;

	*p = '\0';

	if (prompt_length > 60) {
		send_me_numeric(cptr, RPL_MAPMORE, prompt, server->name);
	}
	else {
		for (acptr = client; acptr != NULL; acptr = acptr->next) {
			if ((GeneralConfig.hide_super_servers && IsULine(acptr) && !HasMode(cptr, UMODE_OPER))
			  || !IsPerson(acptr)) {
				continue;
			}

			count++;
			if (!irccmp(acptr->user->server, server->name)) {
				local++;
			}
		}
		send_me_numeric(cptr, RPL_MAP, prompt, length, server->name, local, (local * 100) / count);
		count = 0;
	}

	if (prompt_length > 0) {
		p[-1] = ' ';
		if (p[-2] == '`') {
			p[-2] = ' ';
		}
	}
	if (prompt_length > 60) {
		return;
	}

	strcpy(p, "|-");

	for (acptr = client; acptr != NULL; acptr = acptr->next) {
		if (!IsServer(acptr) || mycmp(acptr->serv->up, server->name)) {
			continue;
		}
		if (GeneralConfig.hide_super_servers && IsULine(acptr) && !HasMode(cptr, UMODE_OPER)) {
			continue;
		}

		ClearMapped(acptr);
		if (!match(mask, acptr->name)) {
			SetMapped(acptr);
			count++;
		}
	}

	for (acptr = client; acptr != NULL; acptr = acptr->next) {
		if (!IsServer(acptr) || !IsMapped(acptr)) {
			continue;
		}
		if (GeneralConfig.hide_super_servers && IsULine(acptr) && !HasMode(cptr, UMODE_OPER)) {
			continue;
		}
		if (mycmp(acptr->serv->up, server->name)) {
			continue;
		}

		count--;
		if (!count) {
			*p = '`';
		}

		dump_map(cptr, acptr, mask, prompt_length + 2, length - 2);
	}

	if (prompt_length > 0) {
		p[-1] = '-';
	}

	return;
}

/*
 * m_map
 *	parv[0] = sender prefix
 *	parv[1] = server mask
 */
int m_map(aClient *cptr, aClient *sptr, int parc, char *parv[])
{
	aClient *acptr;
	int longest = strlen(me.name), len = 0;
	static time_t last_used = 0L;

	if (!GeneralConfig.enable_map) {
		send_me_numeric(sptr, ERR_FUNCDISABLED, "MAP");
		return 0;
	}
	if (!HasMode(sptr, UMODE_OPER)) {
		if (GeneralConfig.flatten_links) {
			send_me_numericNA(sptr, ERR_NOPRIVILEGES);
			return 0;
		}
		if (FloodConfig.pace_wait_intense && (last_used + FloodConfig.pace_wait_intense > timeofday)) {
			send_me_numericNA(sptr, RPL_LOAD2HI);
			return 0;
		}
		else {
			last_used = timeofday;
		}
		if (GeneralConfig.spy_notices && IsPerson(sptr)) {
			sendto_realops_lev(SPY_LEV, "MAP requested by %s (%s@%s)", sptr->name,
				sptr->username, MaskedHost(sptr));
		}
	}

	if (parc < 2) {
		parv[1] = "*";
	}

	for (acptr = client; acptr != NULL; acptr = acptr->next) {
		if (!IsServer(acptr)) {
			continue;
		}

		len = strlen(acptr->name) + acptr->hopcount * 2;
		if (len > longest) {
			longest = len;
		}
	}

	if (longest > 60) {
		longest = 60;
	}
	longest += 2;

	dump_map(sptr, &me, parv[1], 0, longest);
	send_me_numericNA(sptr, RPL_MAPEND);
	return 0;
}
