/*
 * RageIRCd: an advanced Internet Relay Chat daemon (ircd).
 * (C) 2000-2005 the RageIRCd Development Team, all rights reserved.
 *
 * This software is free, licensed under the General Public License.
 * Please refer to doc/LICENSE and doc/README for further details.
 *
 * $Id: conf_parse.c,v 1.22.2.2 2005/01/15 21:01:57 amcwilliam Exp $
 */

#include "struct.h"
#include "common.h"
#include "h.h"
#include "sys.h"
#include "numeric.h"
#include "msg.h"
#include "channel.h"
#include "patchlevel.h"
#include "conf2.h"
#include "config.h"
#include "memory.h"
#include "hook.h"
#include "dlink.h"
#include "fd.h"
#include <time.h>
#include <signal.h>
#include <sys/stat.h>
#include <stdlib.h>
#include <string.h>
#include <utmp.h>
#include <fcntl.h>

dlink_list conf_file_list = DLINK_LIST_INIT;
dlink_list conf_block_list = DLINK_LIST_INIT;
dlink_list conf_unknown_list = DLINK_LIST_INIT;

void report(short error, char *string, ...)
{
	va_list vl;
	char buffer[1024];

	va_start(vl, string);
	ircvsprintf(buffer, string, vl);
	va_end(vl);

	if (Internal.rehashing) {
		sendto_realops("%s", buffer);
	}
	else {
		fprintf(stderr, "%s"LF, buffer);
	}
	if (error) {
		ircdlog(LOG_ERROR, "%s", buffer);
		Internal.conf_errors++;
	}
}

static ConfigDirective *get_directive(char *d)
{
	ConfigDirective *dir;

	for (dir = conf_directives; dir->directive != NULL; dir++) {
		if (!mycmp(dir->directive, d)) {
			return dir;
		}
	}
	return NULL;
}

static void free_configentry(ConfigEntry *entry)
{
	ConfigEntry *next;

	while (entry != NULL) {
		next = entry->next;

		if (entry->entries != NULL) {
			free_configentry(entry->entries);
		}
		cMyFree(entry->varname);
		cMyFree(entry->vardata);
		cMyFree(entry);
		entry = next;
	}
}

static void free_configfile(ConfigFile *file)
{
	ConfigFile *next;

	while (file != NULL) {
		next = file->next;

		if (file->entries != NULL) {
			free_configentry(file->entries);
		}
		cMyFree(file->filename);
		cMyFree(file);
		file = next;
	}
}

static ConfigFile *load_configfile(char *filename, char *confdata)
{
	ConfigFile *curcf, *lastcf;
	ConfigEntry *curce = NULL, *cursect = NULL, **lastce;
	char *ptr, *start;
	int linenum = 1, commentstart = 0, commentlevel = 0;

	lastcf = curcf = (ConfigFile *)MyMalloc(sizeof(ConfigFile));
	cDupString(curcf->filename, filename);
	lastce = &(curcf->entries);

	for (ptr = confdata; *ptr != '\0'; ptr++) {
		switch (*ptr) {
			case ';':
				if (curce == NULL) {
					report(0, "%s:%d: ignoring extra semicolon", filename, linenum);
					break;
				}
				*lastce = curce;
				lastce = &(curce->next);
				curce = NULL;
				break;
			case '{':
				if (curce == NULL) {
					report(0, "%s:%d: ignoring blank section-start", filename, linenum);
					break;
				}
				else if (curce->entries != NULL) {
					report(0, "%s:%d: ignoring extra section-start", filename, linenum);
					break;
				}
				curce->sectlinenum = linenum;
				lastce = &(curce->entries);
				cursect = curce;
				curce = NULL;
				break;
			case '}':
				if (curce != NULL) {
					report(1, "%s:%d: ERROR: missing semicolon before section-finish",
						filename, linenum);
					free_configentry(curce);
					free_configfile(curcf);
					return NULL;
				}
				else if (cursect == NULL) {
					report(0, "%s:%d: ignoring extra section-finish", filename, linenum);
					break;
				}

				curce = cursect;
				cursect = cursect->prev;
				lastce = (cursect == NULL) ? &(curcf->entries) : &(cursect->entries);

				while (*lastce != NULL) {
					lastce = &((*lastce)->next);
				}
				break;
			case '#':
				ptr++;
				while (*ptr != '\0' && *ptr != '\n') {
					ptr++;
				}
				if (*ptr == '\0') {
					break;
				}
				ptr--;
				break;
			case '/':
				if (*(ptr + 1) == '/') {
					ptr += 2;
					while (*ptr != '\0' && *ptr != '\n') {
						ptr++;
					}
					if (*ptr == '\0') {
						break;
					}
					ptr--;
				}
				else if (*(ptr + 1) == '*') {
					commentstart = linenum;
					commentlevel = 1;

					for (ptr += 2; *ptr; ptr++) {
						if (*ptr == '/' && *(ptr + 1) == '*') {
							commentlevel++;
							ptr++;
						}
						else if (*ptr == '*' && *(ptr + 1) == '/') {
							commentlevel--;
							ptr++;
						}
						else if (*ptr == '\n') {
							linenum++;
						}
						if (!commentlevel) {
							break;
						}
					}
					if (*ptr == '\0') {
						report(1, "%s:%d: ERROR: unexpected EOF: possible unterminated "
							"comment at line %d", filename, linenum, commentstart);
						free_configentry(curce);
						free_configfile(curcf);
						return NULL;
					}
				}
				break;
			case '\"':
				start = ++ptr;
				while (*ptr != '\0') {
					if (*ptr == '\\' && *(ptr + 1) == '\"') {
						char *ptr2 = ptr;
						while ((*ptr2 = *(ptr2 + 1))) {
							ptr2++;
						}
					}
					else if (*ptr == '\"' || *ptr == '\n') {
						break;
					}
					ptr++;
				}
				if (*ptr == '\0' || *ptr == '\n') {
					report(1, "%s:%d: ERROR: unexpected EOF: possible unterminated "
						"double-quote", filename, linenum);
					free_configentry(curce);
					free_configfile(curcf);
					return NULL;
				}
				if (curce != NULL) {
					if (curce->vardata != NULL) {
						report(0, "%s:%d: ignoring extra data", filename, linenum);
					}
					else {
						curce->vardata = (char *)MyMalloc((ptr - start) + 1);
						strncpy(curce->vardata, start, ptr - start);
						curce->vardata[ptr - start] = '\0';
					}
					break;
				}
				curce = (ConfigEntry *)MyMalloc(sizeof(ConfigEntry));
				curce->varname = (char *)MyMalloc((ptr - start) + 1);
				strncpy(curce->varname, start, ptr - start);
				curce->varname[ptr - start] = '\0';
				curce->file = curcf;
				curce->varlinenum = linenum;
				curce->prev = cursect;
				break;
			case '\n':
				linenum++;
			case '\t':
			case '\r':
			case '=':
			case ' ':
				break;
			default:
				if (*ptr == '*' && *(ptr + 1) == '/') {
					report(0, "%s:%d: ignoring extra close-comment", filename, linenum);
					ptr++;
					break;
				}

				start = ptr;
				while (*ptr != '\0') {
					if (*ptr == ' ' || *ptr == '=' || *ptr == '\t' || *ptr == '\n'
					  || *ptr == ';') {
						break;
					}
					ptr++;
				}
				if (*ptr == '\0') {
					if (curce != NULL) {
						report(1, "%s:%d: ERROR: unexpected EOF: unterminated variable "
							"at line %d", filename, linenum, curce->varlinenum);
					}
					else if (cursect != NULL) {
						report(1, "%s:%d: ERROR: unexpected EOF: unterminated section "
							"starting at line %d", filename, linenum,
							curce->sectlinenum);
					}
					else {
						report(1, "%s:%d: ERROR: unexpected EOF: unknown cause", filename,
							linenum);
					}
					free_configentry(curce);
					free_configfile(curcf);
					return NULL;
				}
				if (curce != NULL) {
					if (curce->vardata != NULL) {
						report(0, "%s:%d: ignoring extra data", filename, linenum);
					}
					else {
						curce->vardata = (char *)MyMalloc((ptr - start) + 1);
						strncpy(curce->vardata, start, ptr - start);
						curce->vardata[ptr - start] = '\0';
					}
				}
				else {
					curce = (ConfigEntry *)MyMalloc(sizeof(ConfigEntry));
					curce->varname = (char *)MyMalloc((ptr - start) + 1);
					strncpy(curce->varname, start, ptr - start);
					curce->varname[ptr - start] = '\0';
					curce->file = curcf;
					curce->varlinenum = linenum;
					curce->prev = cursect;
				}
				if (*ptr == ';' || *ptr == '\n') {
					ptr--;
				}
				break;
		}
	}
	if (curce != NULL) {
		report(1, "%s: ERROR: unexpected EOF: unterminated variable at line %d", filename,
			curce->varlinenum);
		free_configentry(curce);
		free_configfile(curcf);
		return NULL;
	}
	if (cursect != NULL) {
		report(1, "%s: ERROR: unexpected EOF: unterminated section starting at line %d", filename,
			cursect->sectlinenum);
		free_configfile(curcf);
		return NULL;
	}
	return curcf;
}

static ConfigFile *open_configfile(char *filename)
{
	int fd, len, i = (!Internal.rehashing) ? 1 : 0;
	struct stat sb;
	ConfigFile *file;
	char *confdata, *err = (!Internal.rehashing) ? "ERROR: failed" : "Failed";

	if (BadPtr(filename)) {
		report(0, "Ignoring blank config file name.");
		return NULL;
	}

	if (Internal.verbose) {
		report(0, "Attempting to open config file %s...", filename);
	}

#ifdef OS_CYGWIN
	if ((fd = open(filename, O_RDONLY|O_BINARY)) == -1) {
#else
	if ((fd = open(filename, O_RDONLY)) == -1) {
#endif
		report(i, "%s to open config file %s: %s.", err, filename, strerror(errno));
		return NULL;
	}

	if (fstat(fd, &sb) == -1) {
		report(i, "%s to fstat config file %s: %s.", err, filename, strerror(errno));
		close(fd);
		return NULL;
	}
	if (sb.st_size <= 0) {
		report(0, "Ignoring empty config file %s.", filename);
		close(fd);
		return NULL;
	}

	confdata = (char *)MyMalloc(sb.st_size + 1);
	if ((len = read(fd, confdata, sb.st_size)) != sb.st_size) {
		report(i, "%s to read config file %s: %s.", err, filename,
			len == -1 ? strerror(errno) : strerror(EFAULT));
		MyFree(confdata);
		close(fd);
		return NULL;
	}
	close(fd);

	confdata[len] = '\0';
	file = load_configfile(filename, confdata);
	MyFree(confdata);

	return file;
}

static void parse_configfile(ConfigFile *file)
{
	ConfigDirective *dir;
	ConfigBlock *block;
	ConfigEntry *entry;
	void *item;

	if (file->entries == NULL) {
		if (Internal.verbose) {
			report(0, "%s: ignoring empty file", file->filename);
		}
		return;
	}
	for (EACH_ENTRY(entry, file)) {
		if (BadPtr(entry->varname)) {
			report(0, "%s:%d: ignoring null variable", entry->file->filename, entry->varlinenum);
			continue;
		}

		if ((dir = get_directive(entry->varname)) == NULL) {
			report(0, "%s:%d: ignoring unknown block %s", entry->file->filename,
				entry->varlinenum, entry->varname);
			continue;
		}

		if ((item = dir->parse_func(entry)) == NULL) {
			continue;
		}
		if (dir->test_func == NULL || dir->free_func == NULL) {
			continue;
		}

		block = (ConfigBlock *)MyMalloc(sizeof(ConfigBlock));
		block->file = file;
		block->dir = dir;
		block->item = item;

		Debug((DEBUG_DEBUG, "conf_parse: adding block %x to list %x [h:%x t:%x l:%lu]",
			block, &conf_block_list, conf_block_list.head, conf_block_list.tail,
			dlink_length(&conf_block_list)));

		dlink_add(&conf_block_list, block);
	}
}

static void init_configfile(char *filename)
{
	ConfigFile *file;

	chmod(filename, 0600);
	if ((file = open_configfile(filename)) != NULL) {
		if (Internal.verbose) {
			report(0, "Successfully loaded config file %s.", filename);
		}
		dlink_add(&conf_file_list, file);
		parse_configfile(file);
	}
}

static void init_default_conf()
{
	if (DefaultClass == NULL) {
		DefaultClass = (ConfigItem_class *)MyMalloc(sizeof(ConfigItem_class));
		cDupString(DefaultClass->name, "default");
		DefaultClass->ping_time = DEFAULT_PINGTIME;
		DefaultClass->max_clients = DEFAULT_MAX_CLIENTS;
		DefaultClass->sendq_length = MAXSENDQLENGTH;
		DefaultClass->item.perm = 1;
		dlink_add(&conf_class_list, DefaultClass);
	}
	if (!dlink_length(&conf_include_list)) {
		ConfigItem_include *include = (ConfigItem_include *)MyMalloc(sizeof(ConfigItem_include));
		cDupString(include->filename, Internal.conf_file);
		include->loaded = 1;
		dlink_add(&conf_include_list, include);
	}
}

static void parse_unknowns()
{
	dlink_node *node, *next = NULL;
	ConfigUnknown *unknown;
	HookData hdata = HOOKDATA_INIT;
	int free_it = 0;

	DLINK_FOREACH_SAFE_DATA(conf_unknown_list.head, node, next, unknown, ConfigUnknown) {
		if (unknown->item != NULL) {
			hdata.v = (void *)unknown;
			if (hook_run_untilnot(h_conf_parse, &hdata, CONF_NOMATCH) == CONF_NOMATCH) {
				UNKNOWN_VARNA(unknown->entry);
				free_it = 1;
			}
		}
		if (free_it || (unknown->item == NULL)) {
			dlink_del(&conf_unknown_list, NULL, node);
			MyFree(unknown);
			free_it = 0;
		}
	}
}

static void test_unknowns()
{
	dlink_node *node, *next = NULL;
	ConfigUnknown *unknown;
	HookData hdata = HOOKDATA_INIT;

	DLINK_FOREACH_SAFE_DATA(conf_unknown_list.head, node, next, unknown, ConfigUnknown) {
		if (Internal.valid_conf && (unknown->item != NULL)) {
			hdata.v = (void *)unknown;
			hook_run_untilnot(h_conf_test, &hdata, CONF_NOMATCH);
		}
		dlink_del(&conf_unknown_list, NULL, node);
		MyFree(unknown);
	}
}

static void test_conf()
{
	dlink_node *node, *next;
	ConfigBlock *block;
	int status;

	Internal.valid_conf = 1;
	if (Internal.verbose) {
		report(0, "Validating all loaded server configuration...");
	}

	DLINK_FOREACH_SAFE_DATA(conf_block_list.head, node, next, block, ConfigBlock) {
		dlink_del(&conf_block_list, NULL, node);

		if (!Internal.valid_conf) {
			cMyFree(block);
			continue;
		}

		ASSERT(block->dir != NULL);

		status = block->dir->test_func(block->file, block->item);
		switch (status) {
			case CONF_SUCCESS:
				break;
			case CONF_FALLBACK:
				Internal.valid_conf = 0;
				report(0, "WARNING: falling back to previous configuration due to errors.");
			case CONF_FAILURE:
				block->dir->free_func(block->item);
				break;
			default:
				ircdlog(LOG_ERROR, "Encountered invalid test retval [%d] in first-stage validation",
					status);
				break;
		}

		cMyFree(block);
	}

	test_unknowns();

	if (!Internal.valid_conf) {
		report(0, "ERROR: failed to validate server configuration.");
	}
	else if (Internal.verbose) {
		report(0, "Successfully validated all server configuration.");
	}
}

static void setup_masking_keys()
{
	if (!Internal.rehashing) {
		init_masking_keys();
	}
}

static void setup_try_connections()
{
	del_event_byfunc(try_connections, NULL);

	if (GeneralConfig.auto_connect_freq) {
		add_event("try_connections", try_connections, NULL, GeneralConfig.auto_connect_freq, 1);
	}
}

void init_conf()
{
	dlink_node *node, *next = NULL;
	ConfigFile *file;
	ConfigItem_include *include;
	
	if (chdir(CONF_DIR) == -1) {
		report(1, "ERROR: cannot change to conf directory: %s", strerror(errno));
		report(1, "       CONF_DIR: %s", CONF_DIR);
		return;
	}

	/* Initialise defaults */
	if (!Internal.rehashing) {
		init_default_conf();
	}
	Internal.conf_errors = 0;

	/* Load all the configuration (main conf, includes, unknowns) */
	init_configfile(Internal.conf_file);
	DLINK_FOREACH_DATA(conf_include_list.head, node, include, ConfigItem_include) {
		if (!include->loaded) {
			include->loaded = 1;
			init_configfile(include->filename);
		}
	}
	parse_unknowns();

	/* Validate configuration */
	if (!Internal.conf_errors) {
		test_conf();
	}

	/* Free files */
	DLINK_FOREACH_SAFE_DATA(conf_file_list.head, node, next, file, ConfigFile) {
		dlink_del(&conf_file_list, NULL, node);
		free_configfile(file);
	}
	DLINK_FOREACH_SAFE_DATA(conf_include_list.head, node, next, include, ConfigItem_include) {
		dlink_del(&conf_include_list, NULL, node);
		cMyFree(include->filename);
		cMyFree(include);
	}

	/* Verify configuration */
	if (ServerInfo == NULL) {
		report(1, "ERROR: missing servinfo{} configuration.");
	}
	if (!dlink_length(&conf_allow_list)) {
		report(1, "ERROR: missing allow{} configuration.");
	}
	if (!tmpNetworkConfig.max_link_depth) {
		report(1, "ERROR: missing network{} configuration.");
	}
	if (BadPtr(tmpMaskingConfig.user_mask_prefix)) {
		report(1, "ERROR: missing masking{} configuration.");
	}
	if (!tmpFloodConfig.user_recvq_limit) {
		report(1, "ERROR: missing flood{} configuration.");
	}
	if (!tmpGeneralConfig.max_bans) {
		report(1, "ERROR: missing general{} configuration.");
	}

	if (Internal.valid_conf) {
		hook_run_untilnot(h_conf_verify, NULL, CONF_SUCCESS);
	}

	/* Successfully verified conf */
	if (Internal.valid_conf) {
		/* Free existing config */
		if (Internal.rehashing) {
			free_network(&NetworkConfig);
			free_masking(&MaskingConfig);
			free_flood(&FloodConfig);
			free_general(&GeneralConfig);
		}

		/* Copy over new config */
		memcpy(&NetworkConfig, &tmpNetworkConfig, sizeof(NetworkConfig));
		memcpy(&MaskingConfig, &tmpMaskingConfig, sizeof(MaskingConfig));
		memcpy(&FloodConfig, &tmpFloodConfig, sizeof(FloodConfig));
		memcpy(&GeneralConfig, &tmpGeneralConfig, sizeof(GeneralConfig));


		/* And zeroise temporary elements */
		memset(&tmpNetworkConfig, '\0', sizeof(tmpNetworkConfig));
		memset(&tmpMaskingConfig, '\0', sizeof(tmpMaskingConfig));
		memset(&tmpFloodConfig, '\0', sizeof(tmpFloodConfig));
		memset(&tmpGeneralConfig, '\0', sizeof(tmpGeneralConfig));
	}
	if (!Internal.conf_errors) {
		setup_masking_keys();
		setup_try_connections();

		hook_run(h_conf_complete, NULL);
	}
}
