/*
 * RageIRCd: an advanced Internet Relay Chat daemon (ircd).
 * (C) 2000-2005 the RageIRCd Development Team, all rights reserved.
 *
 * This software is free, licensed under the General Public License.
 * Please refer to doc/LICENSE and doc/README for further details.
 *
 * $Id: engine_rtsigio.c,v 1.14.2.2 2005/02/25 03:11:18 pneumatus Exp $
 */

#ifndef _GNU_SOURCE
#define _GNU_SOURCE 1
#endif

#include "setup.h"
#include "struct.h"
#include "common.h"
#include "sys.h"
#include "h.h"
#include "fd.h"

#ifdef USE_RTSIGIO

#include <signal.h>
#include <unistd.h>
#include <fcntl.h>
#include POLLH

#define SIGIO_SIGNAL SIGRTMIN

#ifndef POLLRDNORM
#define POLLRDNORM POLLIN
#endif
#ifndef POLLWRNORM
#define POLLWRNORM POLLOUT
#endif

typedef struct _pollfd_list pollfd_list;
struct _pollfd_list {
	struct pollfd list[MAXCONNECTIONS];
	int max_index;
};

static pollfd_list poll_fds;

static int my_pid;
static int queue_overflow;
static sigset_t sigio_set;

static inline void mask_our_signal()
{
	sigemptyset(&sigio_set);
	sigaddset(&sigio_set, SIGIO_SIGNAL);
	sigaddset(&sigio_set, SIGIO);
	sigprocmask(SIG_BLOCK, &sigio_set, NULL);
}

static int poll_find_slot()
{
	int i;

	for (i = 0; i < MAXCONNECTIONS; i++) {
		if (poll_fds.list[i].fd == FD_UNUSED) {
			return i;
		}
	}

	ASSERT("poll_find_slot() with no slots" == NULL);
	return FD_UNUSED;
}

static void poll_update_fds(int fd, short event, FDCB *cb)
{
	fd_entry *fde = &fd_table[fd];
	int i;

	if (fde->index == FD_UNUSED) {
		fde->index = poll_find_slot();
	}

	i = fde->index;

	if (cb != NULL) {
		poll_fds.list[i].events |= event;
		poll_fds.list[i].fd = fd;

		if (i > poll_fds.max_index) {
			poll_fds.max_index = i;
		}
	}
	else if (i >= 0) {
		poll_fds.list[i].events &= ~event;

		if (poll_fds.list[i].events) {
			return;
		}

		poll_fds.list[i].revents = 0;
		poll_fds.list[i].fd = FD_UNUSED;
		fde->index = FD_UNUSED;

		if (i == poll_fds.max_index) {
			while (poll_fds.max_index >= 0
			  && (poll_fds.list[poll_fds.max_index].fd == FD_UNUSED)) {
				poll_fds.max_index--;
			}
		}
	}
}

void setup_sigio_fd(int fd)
{
	fcntl(fd, F_SETFL, fcntl(fd, F_GETFL)|O_ASYNC|O_NONBLOCK);
	fcntl(fd, F_SETSIG, SIGIO_SIGNAL);
	fcntl(fd, F_SETOWN, my_pid);
}

void engine_init()
{
	int i;

	for (i = 0; i < MAXCONNECTIONS; i++) {
		poll_fds.list[i].fd = FD_UNUSED;
		poll_fds.list[i].events = 0;
		poll_fds.list[i].revents = 0;
	}
	poll_fds.max_index = 0;

	my_pid = getpid();
	mask_our_signal();
}

void engine_set_call(int fd, unsigned short flags, FDCB *cb, void *data, time_t timeout)
{
	fd_entry *fde = &fd_table[fd];

	if (flags & FDEV_READ) {
		SetCallback(fde->read, cb, data);
		poll_update_fds(fd, POLLRDNORM, cb);
	}
	if (flags & FDEV_WRITE) {
		SetCallback(fde->write, cb, data);
		poll_update_fds(fd, POLLWRNORM, cb);
	}
	if (timeout) {
		fde->timeout_time = timeofday + (timeout / 1000);
	}
}

void engine_do_netio(time_t delay)
{
	int cnt = 0, i, fd, sig;
	struct siginfo si;
	struct timespec ts;
	fd_entry *fde;
	FDCB *cb;
	void *data;

	ts.tv_sec = 0;
	ts.tv_nsec = (delay * 1000);

	for (;;) {
		if (queue_overflow) {
			break;
		}
		if ((sig = sigtimedwait(&sigio_set, &si, &ts)) <= 0) {
			break;
		}

		if (sig == SIGIO) {
			ircdlog(LOG_ERROR, "Kernel RT signal queue overflow. Please check "
				"/proc/sys/kernel/rtsig-max is big enough!");
			queue_overflow = 1;
			break;
		}

		set_time();

		fd = si.si_fd;
		fde = &fd_table[fd];
		poll_fds.list[fd].revents |= si.si_band;

		if (poll_fds.list[fd].revents & (POLLRDNORM|POLLIN|POLLHUP|POLLERR)) {
			GetCallback(fde->read, cb, data);
			ClrCallback(fde->read);

			poll_update_fds(fd, POLLRDNORM, NULL);

			if (cb != NULL) {
				cb(fd, data, ENGINE_OK);
			}
		}
		if (poll_fds.list[fd].revents & (POLLWRNORM|POLLOUT|POLLHUP|POLLERR)) {
			GetCallback(fde->write, cb, data);
			ClrCallback(fde->write);

			poll_update_fds(fd, POLLWRNORM, NULL);

			if (cb != NULL) {
				cb(fd, data, ENGINE_OK);
			}
		}
	}

	if (!queue_overflow) {
		set_time();
		return;
	}

	signal(SIGIO_SIGNAL, SIG_IGN);
	signal(SIGIO_SIGNAL, SIG_DFL);
	queue_overflow = 0;

	for (;;) {
		if ((cnt = poll(poll_fds.list, poll_fds.max_index + 1, 0)) >= 0) {
			break;
		}
		if (engine_ignore_errno(errno)) {
			continue;
		}

		set_time();
		return;
	}

	set_time();

	if (!cnt) {
		return;
	}

	for (i = 0; i < poll_fds.max_index + 1; i++) {
		if (!poll_fds.list[i].revents || (poll_fds.list[i].fd == FD_UNUSED)) {
			continue;
		}

		fd = poll_fds.list[i].fd;
		fde = &fd_table[fd];

		if (poll_fds.list[i].revents & (POLLRDNORM|POLLIN|POLLHUP|POLLERR)) {
			GetCallback(fde->read, cb, data);
			ClrCallback(fde->read);

			poll_update_fds(fd, POLLRDNORM, NULL);

			if (cb != NULL) {
				cb(fd, data, ENGINE_OK);
			}
		}
		if (poll_fds.list[i].revents & (POLLWRNORM|POLLOUT|POLLHUP|POLLERR)) {
			GetCallback(fde->write, cb, data);
			ClrCallback(fde->write);

			poll_update_fds(fd, POLLWRNORM, NULL);

			if (cb != NULL) {
				cb(fd, data, ENGINE_OK);
			}
		}
	}

	mask_our_signal();
}

#endif
