/*
 * RageIRCd: an advanced Internet Relay Chat daemon (ircd).
 * (C) 2000-2005 the RageIRCd Development Team, all rights reserved.
 *
 * This software is free, licensed under the General Public License.
 * Please refer to doc/LICENSE and doc/README for further details.
 *
 * $Id: event.c,v 1.25.2.1 2004/12/07 03:05:11 pneumatus Exp $
 */

#include "struct.h"
#include "common.h"
#include "sys.h"
#include "numeric.h"
#include "msg.h"
#include "channel.h"
#include "patchlevel.h"
#include <time.h>
#include <sys/stat.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include "h.h"
#include "memory.h"

static dlink_list event_list = DLINK_LIST_INIT;

static dlink_node *find_event_node(void (*func)(), void *arg)
{
	dlink_node *node;
	Event *event;

	DLINK_FOREACH_DATA(event_list.head, node, event, Event) {
		if (event->func == func && (event->arg == arg)) {
			return node;
		}
	}
	return NULL;
}

Event *add_event(char *name, void (*func)(), void *arg, time_t when, int loop)
{
	Event *event = (Event *)MyMalloc(sizeof(Event));
	DupString(event->name, name);
	event->func = func;
	event->arg = arg;
	event->when = (timeofday + when);
	event->freq = when;
	event->loop = (loop) ? 1 : 0;
	dlink_add(&event_list, event);

	if (Internal.next_event > when || (Internal.next_event == -1)) {
		Internal.next_event = when;
	}

	return event;
}

Event *add_event_approx(char *name, void (*func)(), void *arg, time_t when, int loop)
{
	if (when > 3) {
		const time_t two_thirds = (2 * when) / 3;
		when = two_thirds + ((rand() % 1000) * two_thirds) / 1000;
	}

	return add_event(name, func, arg, when, loop);
}

void del_event(Event *event, dlink_node *node)
{
	ASSERT(event != NULL);

	dlink_del(&event_list, event, node);

	MyFree(event->name);
	MyFree(event);
}

void del_event_byfunc(void (*func)(), void *arg)
{
	dlink_node *node;
	if ((node = find_event_node(func, arg)) != NULL) {
		del_event((Event *)node->data, node);
	}
}

void run_events()
{
	dlink_node *node, *next = NULL;
	Event *event;

	DLINK_FOREACH_SAFE_DATA(event_list.head, node, next, event, Event) {
		if (event->when > timeofday) {
			continue;
		}

		event->func(event->arg);

		if (event->loop) {
			event->when = (timeofday + event->freq);
		}
		else {
			del_event(event, node);
		}
		Internal.next_event = -1;
	}
}

void get_next_event()
{
	dlink_node *node;
	Event *event;

	if (Internal.next_event != -1) {
		return;
	}
	DLINK_FOREACH_DATA(event_list.head, node, event, Event) {
		if (Internal.next_event > event->when || (Internal.next_event == -1)) {
			Internal.next_event = event->when;
		}
	}
}

void set_back_events(time_t by)
{
	dlink_node *node;
	Event *event;

	DLINK_FOREACH_DATA(event_list.head, node, event, Event) {
		if (event->when > by) {
			event->when -= by;
		}
		else {
			event->when = 0;
		}
	}
}

void show_events(aClient *sptr)
{
	dlink_node *node;
	Event *event;

	send_me_debugNA(sptr, "E :Operation                    Next Execution");

	DLINK_FOREACH_DATA(event_list.head, node, event, Event) {
		send_me_debug(sptr, "E :%-28s %-5d seconds (%s)", event->name,
			(event->when - timeofday), event->loop ? "continuous" : "once");
	}
}
