/*
 * RageIRCd: an advanced Internet Relay Chat daemon (ircd).
 * (C) 2000-2005 the RageIRCd Development Team, all rights reserved.
 *
 * This software is free, licensed under the General Public License.
 * Please refer to doc/LICENSE and doc/README for further details.
 *
 * $Id: s_debug.c,v 1.57.2.2 2005/01/15 21:01:57 amcwilliam Exp $
 */

#include "struct.h"
#include "patchlevel.h"
#include "blalloc.h"
#include "memory.h"
#include "numeric.h"
#include "common.h"
#include "sys.h"
#include "hash.h"
#include <sys/file.h>
#ifdef HAVE_GETRUSAGE
#include <sys/resource.h>
#else
#ifdef HAVE_TIMES
#include <sys/times.h>
#endif
#endif
#include "h.h"

#ifdef DEBUGMODE

void debug(int level, char *pattern, ...)
{
	int err = errno;

	if (Internal.debug_level >= 0 && (level <= Internal.debug_level)) {
		static char debugbuf[1024];
		va_list vl;

		va_start(vl, pattern);
		ircvsprintf(debugbuf, pattern, vl);
		va_end(vl);

		if (level == DEBUG_ERROR) {
			ircdlog(LOG_ERROR, "[DEBUG] %s", debugbuf);
		}
		fprintf(stderr, "%s\n", debugbuf);
	}

	errno = err;
}

#ifdef HAVE_GETRUSAGE
#ifdef hz
#define hzz hz
#else
#ifdef HZ
#define hzz HZ
#else
int hzz = 1;
#endif
#endif
#endif /* HAVE_GETRUSAGE */

void send_usage(aClient *cptr)
{
#ifdef HAVE_GETRUSAGE
	struct rusage rus;
	time_t secs, rup;

	if (getrusage(RUSAGE_SELF, &rus) == -1) {
		send_me_notice(cptr, ":Getrusage error: %s", strerror(errno));
		return;
	}

	secs = rus.ru_utime.tv_sec + rus.ru_stime.tv_sec;
	rup = timeofday - me.since;

	if (secs == 0) {
		secs = 1;
	}

	send_me_debug(cptr, "R :CPU Secs %d:%d User %d:%d System %d:%d", secs / 60, secs % 60,
		rus.ru_utime.tv_sec / 60, rus.ru_utime.tv_sec % 60, rus.ru_stime.tv_sec / 60, rus.ru_stime.tv_sec % 60);
	send_me_debug(cptr, "R :RSS %d ShMem %d Data %d Stack %d", rus.ru_maxrss, rus.ru_ixrss / (rup * hzz),
		rus.ru_idrss / (rup * hzz), rus.ru_isrss / (rup * hzz));
	send_me_debug(cptr, "R :Swaps %d Reclaims %d Faults %d", rus.ru_nswap, rus.ru_minflt, rus.ru_majflt);
	send_me_debug(cptr, "R :Block in %d out %d", rus.ru_inblock, rus.ru_oublock);
	send_me_debug(cptr, "R :Msg Rcv %d Send %d", rus.ru_msgrcv, rus.ru_msgsnd);
	send_me_debug(cptr, "R :Signals %d Context Vol. %d Invol %d", rus.ru_nsignals, rus.ru_nvcsw, rus.ru_nivcsw);
#elif defined(HAVE_TIMES)
	struct tms tmsbuf;
	int hzz = 1;
	int ticpermin = hhz * 60;
	int umin = tmsbuf.tms_utime / ticpermin;
	int smin = (tmsbuf.tms_utime % ticpermin) / (float)hzz;
	int usec = tmsbuf.tms_stime / ticpermin;
	int ssec = (tmsbuf.tms_stime % ticpermin) / (float)hzz;
	time_t secs = usec + ssec;
	time_t mins = (secs / 60) + umin + smin;

	secs %= hzz;

	if (times(&tmsbuf) == -1) {
		send_me_notice(cptr, ":times(2) error: %s", strerror(errno));
		return;
	}

	secs = tmsbuf.tms_utime + tmsbuf.tms_stime;

	send_me_debug(cptr, "R :CPU Secs %d:%d User %d:%d System %d:%d", mins, secs, umin, usec, smin, ssec);
#endif
}

#endif

static void count_conf_memory(aClient *cptr)
{
	dlink_node *node;
	ConfigItem_class *class;
	ConfigItem_allow *allow;
	ConfigItem_oper *oper;
	char *cp;
	ConfigItem_link *linkp;
	ConfigItem_listen *listenp;
	int i, len;

	int		servinfo_cnt = 0,
			admin_cnt = 0,
			class_cnt = 0,
			allow_cnt = 0,
			oper_cnt = 0,
			super_cnt = 0,
			linkp_cnt = 0,
			listenp_cnt = 0
#ifndef STATIC_MODULES
			, modfile_cnt = 0, modpath_cnt = 0
#endif
			;
	unsigned long	servinfo_mem = 0,
			admin_mem = 0,
			class_mem = 0,
			allow_mem = 0,
			oper_mem = 0,
			super_mem = 0,
			linkp_mem = 0,
			listenp_mem = 0
#ifndef STATIC_MODULES
			, modfile_mem = 0, modpath_mem = 0
#endif
			;

	int		auth_cnt = 0,
			from_cnt = 0,
			conf_cnt = 0;
	unsigned long	auth_mem = 0,
			from_mem = 0,
			conf_mem = 0;

	conf_cnt = dlink_length(&conf_class_list) + dlink_length(&conf_allow_list)
		+ dlink_length(&conf_oper_list) + dlink_length(&conf_super_list)
		+ dlink_length(&conf_link_list) + dlink_length(&conf_listen_list)
#ifndef STATIC_MODULES
		+ dlink_length(&conf_modulefile_list) + dlink_length(&conf_modulepath_list);
#else
		;
#endif

	if (ServerInfo != NULL) {
		conf_cnt++;
		servinfo_cnt++;

		servinfo_mem += sizeof(ConfigItem_servinfo);
		servinfo_mem += strlen(ServerInfo->name) + strlen(ServerInfo->desc);

#ifdef USE_OPENSSL
		if (!BadPtr(ServerInfo->ssl_certificate)) {
			servinfo_mem += strlen(ServerInfo->ssl_certificate);
		}
		if (!BadPtr(ServerInfo->ssl_private_key)) {
			servinfo_mem += strlen(ServerInfo->ssl_private_key);
		}
#endif
		if (!BadPtr(ServerInfo->default_bind_ip)) {
			servinfo_mem += strlen(ServerInfo->default_bind_ip);
		}
		if (!BadPtr(ServerInfo->kline_address)) {
			servinfo_mem += strlen(ServerInfo->kline_address);
		}
	}

	if (AdminInfo != NULL) {
		conf_cnt++;
		admin_cnt++;

		admin_mem += sizeof(ConfigItem_admin);
		if (!BadPtr(AdminInfo->name)) {
			admin_mem += strlen(AdminInfo->name);
		}
		if (!BadPtr(AdminInfo->desc)) {
			admin_mem += strlen(AdminInfo->desc);
		}
		if (!BadPtr(AdminInfo->email)) {
			admin_mem += strlen(AdminInfo->email);
		}
	}

	class_cnt = dlink_length(&conf_class_list);
	DLINK_FOREACH_DATA(conf_class_list.head, node, class, ConfigItem_class) {
		if (!BadPtr(class->name)) {
			class_mem += strlen(class->name);
		}
	}
	class_mem += (class_cnt * sizeof(ConfigItem_class));

	allow_cnt = dlink_length(&conf_allow_list);
	DLINK_FOREACH_DATA(conf_allow_list.head, node, allow, ConfigItem_allow) {
		if (!BadPtr(allow->hostname)) {
			allow_mem += strlen(allow->hostname);
		}
		if (!BadPtr(allow->ipaddr)) {
			allow_mem += strlen(allow->ipaddr);
		}
		if (allow->auth != NULL) {
			auth_cnt++;
			if (!BadPtr(allow->auth->string)) {
				len = strlen(allow->auth->string);
				auth_mem += len;
				allow_mem += len + sizeof(ConfigAuth);
			}
		}
		if (!BadPtr(allow->redir_serv)) {
			allow_mem += strlen(allow->redir_serv);
		}
	}
	allow_mem += (allow_cnt * sizeof(ConfigItem_allow));

	oper_cnt = dlink_length(&conf_oper_list);
	DLINK_FOREACH_DATA(conf_oper_list.head, node, oper, ConfigItem_oper) {
		if (!BadPtr(oper->name)) {
			oper_mem += strlen(oper->name);
		}
		if (oper->auth != NULL) {
			auth_cnt++;
			if (!BadPtr(oper->auth->string)) {
				len = strlen(oper->auth->string);
				auth_mem += len;
				allow_mem += len + sizeof(ConfigAuth);
			}
		}
		if (!BadPtr(oper->join_on_oper)) {
			oper_mem += strlen(oper->join_on_oper);
		}
		if (oper->from.host_count) {
			for (i = 0; i < oper->from.host_count; i++) {
				len = strlen(oper->from.hosts[i]);
				from_cnt++;
				from_mem += len;
				oper_mem += len;
			}
		}
	}
	oper_mem += (oper_cnt * sizeof(ConfigItem_oper));

	super_cnt = dlink_length(&conf_super_list);
	DLINK_FOREACH_DATA(conf_super_list.head, node, cp, char) {
		super_mem += strlen(cp);
	}

	linkp_cnt = dlink_length(&conf_link_list);
	DLINK_FOREACH_DATA(conf_link_list.head, node, linkp, ConfigItem_link) {
		if (!BadPtr(linkp->servername)) {
			linkp_mem += strlen(linkp->servername);
		}
		if (!BadPtr(linkp->host)) {
			linkp_mem += strlen(linkp->host);
		}
		if (!BadPtr(linkp->bind_ip)) {
			linkp_mem += strlen(linkp->bind_ip);
		}
		if (linkp->auth != NULL) {
			auth_cnt++;
			if (!BadPtr(linkp->auth->string)) {
				len = strlen(linkp->auth->string);
				auth_mem += len;
				allow_mem += len + sizeof(ConfigAuth);
			}
		}
	}
	linkp_mem += (linkp_cnt * sizeof(ConfigItem_link));

	listenp_cnt = dlink_length(&conf_listen_list);
	DLINK_FOREACH_DATA(conf_listen_list.head, node, listenp, ConfigItem_listen) {
		if (!BadPtr(listenp->ipaddr)) {
			listenp_mem += strlen(listenp->ipaddr);
		}
	}
	listenp_mem += (listenp_cnt * sizeof(ConfigItem_listen));

#ifndef STATIC_MODULES
	modfile_cnt = dlink_length(&conf_modulefile_list);
	DLINK_FOREACH_DATA(conf_modulefile_list.head, node, cp, char) {
		modfile_mem += strlen(cp);
	}

	modpath_cnt = dlink_length(&conf_modulepath_list);
	DLINK_FOREACH_DATA(conf_modulepath_list.head, node, cp, char) {
		modpath_mem += strlen(cp);
	}
#endif

	conf_mem = servinfo_mem + admin_mem + class_mem + allow_mem + oper_mem + super_mem
		+ linkp_mem + listenp_mem
#ifndef STATIC_MODULES
		+ modfile_mem + modpath_mem
#endif
		;

	auth_mem += (auth_cnt * sizeof(ConfigAuth));

	send_me_debug(cptr, "Z :ConfigItems %d(%ld) Auth %d(%ld) From %d(%ld)",
		conf_cnt, conf_mem, auth_cnt, auth_mem, from_cnt, from_mem);
	send_me_debug(cptr, "Z :    servinfo %d(%ld)", servinfo_cnt, servinfo_mem);
	send_me_debug(cptr, "Z :    admin %d(%ld)", servinfo_cnt, admin_mem);
	send_me_debug(cptr, "Z :    class %d(%ld)", class_cnt, class_mem);
	send_me_debug(cptr, "Z :    allow %d(%ld)", allow_cnt, allow_mem);
	send_me_debug(cptr, "Z :    oper %d(%ld)", oper_cnt, oper_mem);
	send_me_debug(cptr, "Z :    super %d(%ld)", super_cnt, super_mem);
	send_me_debug(cptr, "Z :    link %d(%ld)", linkp_cnt, linkp_mem);
	send_me_debug(cptr, "Z :    listen %d(%ld)", listenp_cnt, listenp_mem);
#ifndef STATIC_MODULES
	send_me_debug(cptr, "Z :    module file %d(%ld) path %d(%ld)", modfile_cnt, modfile_mem,
		modpath_cnt, modpath_mem);
#endif
}

static void count_motd_memory(MessageFile *mf, int *line_cnt, unsigned long *file_mem)
{
	int cnt = 0;
	unsigned long mem = 0;
	dlink_node *node;
	char *line;

	mem += sizeof(MessageFile);
	if (!BadPtr(mf->filename)) {
		mem += strlen(mf->filename);
	}

	cnt = dlink_length(&mf->data);
	DLINK_FOREACH_DATA(mf->data.head, node, line, char) {
		mem += strlen(line);
	}

	*line_cnt = cnt;
	*file_mem = mem;
}

static void count_sbuf_memory(aClient *cptr)
{
	send_me_debug(cptr, "Z :SBuf allocated %ld",
		(sbuf_user_total * sizeof(SBufUser)) + (sbuf_small_total * SBUF_SMALL_TOTAL) +
		(sbuf_large_total * SBUF_LARGE_TOTAL) + (sbuf_blocks_used * sizeof(SBufBlock)) +
		(sbuf_userblocks_used * sizeof(SBufUserBlock)));

	send_me_debug(cptr, "Z :    blocks %d(%ld) userblocks %d(%ld)",
		sbuf_blocks_used, (sbuf_blocks_used * sizeof(SBufBlock)),
		sbuf_userblocks_used, (sbuf_userblocks_used * sizeof(SBufUserBlock)));

	send_me_debug(cptr, "Z :    users %d MAX(%d) ALLOC(%ld)", sbuf_user_used, sbuf_user_total,
		sbuf_user_total * sizeof(SBufUser));

	send_me_debug(cptr, "Z :    small %d MAX(%d) ALLOC(%ld)", sbuf_small_used, sbuf_small_total,
		sbuf_small_total * SBUF_SMALL_TOTAL);

	send_me_debug(cptr, "Z :    large %d MAX(%d) ALLOC(%ld)", sbuf_large_used, sbuf_large_total,
		sbuf_large_total * SBUF_LARGE_TOTAL);
}

void count_memory(aClient *cptr)
{
	SLink *lp;
	dlink_node *node;
	chanMember *cm;
	channelBan *ban;
	aClient *acptr;
	aChannel *chptr;

	size_t cm_size = sizeof(chanMember);
	size_t slink_size = sizeof(SLink);
	size_t chanban_size = sizeof(channelBan);

	int		client_cnt = 0,		/* aClient count */
			localclient_cnt = 0,	/* LocalClient count */
			user_cnt = 0,		/* anUser count */
			localuser_cnt = 0,	/* LocalUser count */
			serv_cnt = 1,		/* aServer count (+1 for Me) */
			away_cnt = 0,		/* user->away count */
			user_chans = 0,		/* user->channel count */
			user_invites = 0,	/* user->invite count */
			user_silences = 0,	/* user->silence count */
			user_fludees = 0;	/* user->fludee count */

	unsigned long	client_mem = 0,		/* aClient memory */
			localclient_mem = 0,	/* LocalClient memory */
			user_mem = 0,		/* anUser memory */
			localuser_mem = 0,	/* LocalUser memory */
			away_mem = 0;		/* user->away memory */

	int		chan_cnt = 0,		/* aChannel count */
			chan_users = 0,		/* chan->members count */
			chan_invites = 0,	/* chan->invites count */
			chan_ban_cnt = 0,	/* chan->banlist count */
			chan_except_cnt = 0,	/* chan->exceptlist count */
			chan_invex_cnt = 0;	/* chan->invexlist count */

	unsigned long	chan_mem = 0,		/* aChannel memory */
			chan_ban_mem = 0,	/* chan->banlist memory */
			chan_except_mem = 0,	/* chan->exceptlist memory */
			chan_invex_mem = 0;	/* chan->invexlist memory */

	int		bh_client_cnt = 0,	/* BlockHeap client count */
			bh_localclient_cnt = 0,	/* BlockHeap localclient count */
			bh_user_cnt = 0,	/* BlockHeap user count */
			bh_localuser_cnt = 0,	/* BlockHeap localuser count */
			bh_chan_cnt = 0,	/* BlockHeap chan count */
			bh_chanmember_cnt = 0,	/* BlockHeap chan member count */
			bh_channelban_cnt = 0,	/* BlockHeap chan ban count */
			bh_slink_cnt = 0,	/* BlockHeap link count */
			bh_dlinknode_cnt = 0,	/* BlockHeap dlink_node count */
#ifdef FLUD
			bh_fludbot_cnt = 0,	/* BlockHeap fludbot count */
#endif
#ifdef THROTTLE
			bh_throttle_cnt = 0,	/* BlockHeap throttle count */
			bh_hashent_cnt = 0,	/* BlockHeap hashent count */
#endif
			bh_ipentry_cnt = 0;	/* BlockHeap IPEntry count */

	unsigned long	bh_client_mem = 0,	/* BlockHeap client memory */
			bh_localclient_mem = 0,	/* BlockHeap localclient memory */
			bh_user_mem = 0,	/* BlockHeap user memory */
			bh_localuser_mem = 0,	/* BlockHeap localuser memory */
			bh_chan_mem = 0,	/* BlockHeap chan memory */
			bh_chanmember_mem = 0,	/* BlockHeap chan member memory */
			bh_channelban_mem = 0,	/* BlockHeap chan ban memory */
			bh_slink_mem = 0,	/* BlockHeap link memory */
			bh_dlinknode_mem = 0,	/* BlockHeap dlink_node memory */
#ifdef FLUD
			bh_fludbot_mem = 0,	/* BlockHeap fludbot memory */
#endif
#ifdef THROTTLE
			bh_throttle_mem = 0,	/* BlockHeap throttle memory */
			bh_hashent_mem = 0,	/* BlockHeap hashent memory */
#endif
			bh_ipentry_mem = 0;	/* BlockHeap IPEntry memory */

	int		slink_cnt = 0;		/* Total SLink count */
	unsigned long	slink_mem = 0;		/* Total SLink memory */

	int		whowas_cnt = 0;		/* aWhowas count */
	unsigned long	whowas_mem = 0;		/* aWhowas memory */

	int		motd_cnt = 0,
			opermotd_cnt = 0,
			rules_cnt = 0;
	unsigned long	motd_mem = 0,
			opermotd_mem = 0,
			rules_mem = 0;

	int		scache_cnt = 0;		/* ServerCache count */
	unsigned long	scache_mem = 0;		/* ServerCache memory */

	int		ipentry_cnt = 0;	/* IP hash count */
	unsigned long	ipentry_mem = 0;	/* IP hash memory */

	for (acptr = client; acptr != NULL; acptr = acptr->next) {
		if (IsMe(acptr)) {
			continue;
		}

		client_cnt++;
		if (acptr->localClient != NULL) {
			localclient_cnt++;
		}

		if (acptr->user != NULL) {
			user_cnt++;
			for (lp = acptr->user->silence; lp != NULL; lp = lp->next) {
				user_silences++;
			}
			for (cm = acptr->user->channel; cm != NULL; cm = cm->nextchan) {
				user_chans++;
			}
			if (acptr->user->away != NULL) {
				away_cnt++;
				away_mem += strlen(acptr->user->away);
			}
		}
		if (acptr->localUser != NULL) {
			localuser_cnt++;
			for (lp = acptr->localUser->invited; lp != NULL; lp = lp->next) {
				user_invites++;
			}
		}

		if (acptr->user != NULL) {
#ifdef FLUD
			if (MyClient(acptr)) {
				for (lp = acptr->user->fludees; lp != NULL; lp = lp->next) {
					user_fludees++;
				}
			}
#endif
		}

		if (IsServer(acptr)) {
			serv_cnt++;
		}
	}

	client_mem = client_cnt * sizeof(aClient);
	localclient_mem = localclient_cnt * sizeof(LocalClient);

	user_mem = user_cnt * sizeof(anUser);;
	localuser_mem = localuser_cnt * sizeof(LocalUser);

	for (chptr = channel; chptr != NULL; chptr = chptr->nextch) {
		chan_cnt++;
		chan_mem += strlen(chptr->chname);

		chan_users += chptr->users;

		for (lp = chptr->invites; lp != NULL; lp = lp->next) {
			chan_invites++;
		}

		chan_ban_cnt += dlink_length(&chptr->banlist);
		DLINK_FOREACH_DATA(chptr->banlist.head, node, ban, channelBan) {
			chan_ban_mem += (strlen(ban->id) + strlen(ban->set_by));
		}

		chan_except_cnt += dlink_length(&chptr->exceptlist);
		DLINK_FOREACH_DATA(chptr->exceptlist.head, node, ban, channelBan) {
			chan_except_mem += (strlen(ban->id) + strlen(ban->set_by));
		}

		chan_invex_cnt += dlink_length(&chptr->invexlist);
		DLINK_FOREACH_DATA(chptr->invexlist.head, node, ban, channelBan) {
			chan_invex_mem += (strlen(ban->id) + strlen(ban->set_by));
		}
	}

	chan_mem += chan_cnt * sizeof(aChannel);

	chan_ban_mem += chan_ban_cnt * chanban_size;
	chan_except_mem += chan_except_cnt * chanban_size;
	chan_invex_mem += chan_invex_cnt * chanban_size;

	BlockHeapUsage(client_heap, &bh_client_cnt, &bh_client_mem);
	BlockHeapUsage(localclient_heap, &bh_localclient_cnt, &bh_localclient_mem);
	BlockHeapUsage(user_heap, &bh_user_cnt, &bh_user_mem);
	BlockHeapUsage(localuser_heap, &bh_localuser_cnt, &bh_localuser_mem);
	BlockHeapUsage(channel_heap, &bh_chan_cnt, &bh_chan_mem);
	BlockHeapUsage(chanmember_heap, &bh_chanmember_cnt, &bh_chanmember_mem);
	BlockHeapUsage(channelban_heap, &bh_channelban_cnt, &bh_channelban_mem);
	BlockHeapUsage(slink_heap, &bh_slink_cnt, &bh_slink_mem);
	BlockHeapUsage(dlinknode_heap, &bh_dlinknode_cnt, &bh_dlinknode_mem);
#ifdef FLUD
	BlockHeapUsage(fludbot_heap, &bh_fludbot_cnt, &bh_fludbot_mem);
#endif
#ifdef THROTTLE
	BlockHeapUsage(throttle_heap, &bh_throttle_cnt, &bh_throttle_mem);
	BlockHeapUsage(hashent_heap, &bh_hashent_cnt, &bh_hashent_mem);
#endif
	BlockHeapUsage(ipentry_heap, &bh_ipentry_cnt, &bh_ipentry_mem);

	slink_cnt = user_silences + user_fludees + user_invites + chan_invites;
	slink_mem = slink_cnt * slink_size;

	count_whowas_memory(&whowas_cnt, &whowas_mem);
	count_scache_memory(&scache_cnt, &scache_mem);
	count_ip_hash_memory(&ipentry_cnt, &ipentry_mem);

	send_me_debug(cptr, "Z :Clients %d(%ld) allocated %d(%ld)",
		client_cnt, client_mem, bh_client_cnt, bh_client_mem);
	send_me_debug(cptr, "Z :LocalClients %d(%ld) allocated %d(%ld)",
		localclient_cnt, localclient_mem, bh_localclient_cnt, bh_localclient_mem);
	send_me_debug(cptr, "Z :Users %d(%ld) allocated %d(%ld)",
		user_cnt, user_mem, bh_user_cnt, bh_user_mem);
	send_me_debug(cptr, "Z :LocalUsers %d(%ld) allocated %d(%ld)",
		localuser_cnt, localuser_mem, bh_localuser_cnt, bh_localuser_mem);
	send_me_debug(cptr, "Z :Servers %d(%ld)", serv_cnt, (serv_cnt * sizeof(aServer)));

	send_me_debug(cptr, "Z :User aways %d(%ld)", away_cnt, away_mem);

	send_me_debug(cptr, "Z :Channels %d(%ld) allocated %d(%ld)", chan_cnt, chan_mem, bh_chan_cnt,
		bh_chan_mem);
	send_me_debug(cptr, "Z :Channel members %d(%ld) allocated %d(%ld)", chan_users + user_chans,
		(chan_users + user_chans) * cm_size, bh_chanmember_cnt, bh_chanmember_mem);
	send_me_debug(cptr, "Z :    Chan members %d(%ld)", chan_users, chan_users * cm_size);
	send_me_debug(cptr, "Z :    User members %d(%ld)", user_chans, user_chans * cm_size);

	send_me_debug(cptr, "Z :Channel bans %d(%ld) allocated %d(%ld)",
		chan_ban_cnt + chan_except_cnt + chan_invex_cnt,
		chan_ban_mem + chan_except_mem + chan_invex_mem,
		bh_channelban_cnt, bh_channelban_mem);
	send_me_debug(cptr, "Z :    Ban %d(%ld)", chan_ban_cnt, chan_ban_mem);
	send_me_debug(cptr, "Z :    Except %d(%ld)", chan_except_cnt, chan_except_mem);
	send_me_debug(cptr, "Z :    Invex %d(%ld)", chan_invex_cnt, chan_invex_mem);

	send_me_debug(cptr, "Z :SLinks %d(%ld) allocated %d(%ld)", slink_cnt, slink_mem,
		bh_slink_cnt, bh_slink_mem);
	send_me_debug(cptr, "Z :    User Silences %d(%ld)", user_silences, user_silences * slink_size);
#ifdef FLUD
	send_me_debug(cptr, "Z :    User fludees %d(%ld)", user_fludees, user_fludees * slink_size);
#endif
	send_me_debug(cptr, "Z :    User invites %d(%ld) Chan invites %d (%ld)", user_invites,
		user_invites * slink_size, chan_invites, chan_invites * slink_size);

	send_me_debug(cptr, "Z :Whowas users %d(%ld) array %d(%ld)", whowas_cnt,
		(whowas_cnt * (sizeof(anUser) + sizeof(LocalUser))), NICKNAMEHISTORYLENGTH, whowas_mem);

	send_me_debug(cptr, "Z :ServerCache %d(%ld)", scache_cnt, scache_mem);

	send_me_debug(cptr, "Z :dlink_node allocated %d(%ld)", bh_dlinknode_cnt, bh_dlinknode_mem);

#ifdef FLUD
	send_me_debug(cptr, "Z :Fludbots allocated %d(%ld)", bh_fludbot_cnt, bh_fludbot_mem);
#endif

#ifdef THROTTLE
	send_me_debug(cptr, "Z :Throttles allocated %d(%ld) Hashents %d(%ld)",
		bh_throttle_cnt, bh_throttle_mem, bh_hashent_cnt, bh_hashent_mem);
#endif

	send_me_debug(cptr, "Z :IPEntry %d(%ld) allocated %d(%ld)", ipentry_cnt, ipentry_mem,
		bh_ipentry_cnt, bh_ipentry_mem);

	count_conf_memory(cptr);

	if (motd != NULL) {
		count_motd_memory(motd, &motd_cnt, &motd_mem);
	}
	if (opermotd != NULL) {
		count_motd_memory(opermotd, &opermotd_cnt, &opermotd_mem);
	}
	if (rules != NULL) {
		count_motd_memory(rules, &rules_cnt, &rules_mem);
	}

	send_me_debug(cptr, "Z :MessageFiles %d(%ld)", (motd_cnt + opermotd_cnt + rules_cnt),
		(motd_mem + opermotd_mem + rules_mem));
	send_me_debug(cptr, "Z :    motd %d(%ld)", motd_cnt, motd_mem);
	send_me_debug(cptr, "Z :    opermotd %d(%ld)", opermotd_cnt, opermotd_mem);
	send_me_debug(cptr, "Z :    rules %d(%ld)", rules_cnt, rules_mem);

	count_sbuf_memory(cptr);

	send_me_debug(cptr, "Z :Current max. RSS: %lu", get_max_rss());
}
