/*
 * RageIRCd: an advanced Internet Relay Chat daemon (ircd).
 * (C) 2000-2005 the RageIRCd Development Team, all rights reserved.
 *
 * This software is free, licensed under the General Public License.
 * Please refer to doc/LICENSE and doc/README for further details.
 *
 * $Id: support.c,v 1.42.2.1 2004/12/07 03:05:41 pneumatus Exp $
 */

#include "struct.h"
#include "common.h"
#include "sys.h"
#include "h.h"
#include "memory.h"

int msg_has_colour(char *msg)
{
	char *p;

	if (BadPtr(msg)) {
		return 0;
	}

	for (p = msg; *p != '\0'; p++) {
		if (*p == '\003' || *p == '\033') {
			return 1;
		}
	}
	return 0;
}

static time_t atime_exp(char *base, char *ptr)
{
	time_t tmp;
	char *p = ptr, c = *ptr;

	*ptr-- = '\0';
	while (ptr-- > base) {
		if (IsAlpha(*ptr)) {
			break;
		}
	}
	tmp = atoi(ptr + 1);
	*p = c;
	return tmp;
}

time_t atime(char *xtime)
{
	char *d, *h, *m, *s;
	time_t D, H, M, S, retval;
	int i = 0;

	d = h = m = s = NULL;
	D = H = M = S = 0;

	for (d = xtime; *d; d++) {
		if (IsAlpha(*d) && i != 1) {
			i = 1;
		}
	}
	if (!i) {
		return strtol(xtime, NULL, 10);
	}

	d = strchr(xtime, 'd');
	h = strchr(xtime, 'h');
	m = strchr(xtime, 'm');
	s = strchr(xtime, 's');

	if (d != NULL) {
		D = atime_exp(xtime, d);
	}
	if (h != NULL) {
		H = atime_exp(xtime, h);
	}
	if (m != NULL) {
		M = atime_exp(xtime, m);
	}
	if (s != NULL) {
		S = atime_exp(xtime, s);
	}

	retval = ((D * 86400) + (H * 3600) + (M * 60) + S);
	return (retval > 0) ? retval : 0;
}

char *time_to_str(time_t val)
{
	static char buf[256];
	int sum = 0, len = 0;

	if (val < 1) {
		return "0 secs";
	}

	*buf = '\0';
	if ((sum = (val / 86400)) > 0) {
		len = ircsprintf(buf, "%d day%s ", sum, (sum == 1) ? "" : "s");
	}
	if ((sum = ((val / 3600) % 24)) > 0) {
		len = ircsprintf(buf, "%s%d hour%s ", buf, sum, (sum == 1) ? "" : "s");
	}
	if ((sum = ((val / 60) % 60)) > 0) {
		len = ircsprintf(buf, "%s%d min%s ", buf, sum, (sum == 1) ? "" : "s");
	}
	if ((sum = (val % 60)) > 0) {
		len = ircsprintf(buf, "%s%d sec%s ", buf, sum, (sum == 1) ? "" : "s");
	}
	if (buf[len - 1] == ' ') {
		buf[len - 1] = '\0';
	}
	return buf;
}

char *strtoken(char **save, char *str, char *fs)
{
	char *pos = *save;
	char *tmp;

	if (str != NULL) {
		pos = str;
	}
	while (pos != NULL && (*pos != '\0') && (strchr(fs, *pos) != NULL)) {
		pos++;
	}
	if (pos == NULL || (*pos == '\0')) {
		return (pos = *save = NULL);
	}

	tmp = pos;
	while (*pos != '\0' && (strchr(fs, *pos) == NULL)) {
		pos++;
	}
	if (*pos != '\0') {
		*pos++ = '\0';
	}
	else {
		pos = NULL;
	}

	*save = pos;
	return tmp;
}

#ifndef HAVE_STRERROR
char *strerror(int err_no)
{
#if !defined(__FreeBSD__) && !defined(__NetBSD__)
	extern char *str_errlist[];
	extern int sys_nerr;
#endif
	static char buff[40];
	char *errp = (err_no > sys_nerr) ? NULL : sys_errorlist[err_no]);

	if (errp == NULL) {
		ircsprintf(buff, "Unknown error %d", err_no);
		errp = buff;
	}
	return errp;
}
#endif

char *my_itoa(int d)
{
	static char buf[128];
	ircsprintf(buf, "%d", d);
	return buf;
}

int dgets(int fd, char *buf, int len)
{
	static char dgbuf[8192], *head = dgbuf, *tail = dgbuf;
	char *s, *t;
	int n, nr;

	if (head == tail) {
		*head = '\0';
	}
	if (!len) {
		head = tail = dgbuf;
		*head = '\0';
		return 0;
	}
	if (len > sizeof(dgbuf) - 1) {
		len = sizeof(dgbuf) - 1;
	}
	for (;;)
	{
		if (head > dgbuf) {
			for (nr = tail - head, s = head, t = dgbuf; nr > 0; nr--) {
				*t++ = *s++;
			}
			tail = t;
			head = dgbuf;
		}
		if (head < tail && ((s = strchr(head, '\n')) || (s = strchr(head, '\r'))) && s < tail) {
			n = IRCD_MIN(s - head + 1, len);
			memcpy(buf, head, n);
			head += n;
			if (head == tail) {
				head = tail = dgbuf;
			}
			return n;
		}
		if (tail - head >= len) {
			n = len;
			memcpy(buf, head, n);
			if ((head += n) == tail) {
				head = tail = dgbuf;
			}
			return n;
		}

		n = sizeof(dgbuf) - (tail - dgbuf) - 1;
		if ((nr = read(fd, tail, n)) == -1) {
			head = tail = dgbuf;
			return -1;
		}
		if (nr == 0) {
			if (tail > head) {
				n = IRCD_MIN(tail - head, len);
				memcpy(buf, head, n);
				if ((head += n) == tail) {
					head = tail = dgbuf;
				}
				return n;
			}
			head = tail = dgbuf;
			return 0;
		}

		tail += nr;
		*tail = '\0';

		for (t = head; (s = strchr(t, '\n'));) {
			if ((s > head) && (s > dgbuf)) {
				t = s - 1;
				for (nr = 0; *t == '\\'; nr++) {
					t--;
				}
				if (nr & 1) {
					t = s + 1;
					s--;
					nr = tail - t;
					while (nr--) {
						*s++ = *t++;
					}
					tail -= 2;
					*tail = '\0';
				}
				else {
					s++;
				}
			}
			else {
				s++;
			}
			t = s;
		}
		*tail = '\0';
	}
}

int has_wilds(char *str)
{
	char *p;

	ASSERT(str != NULL);

	for (p = str; *p != '\0'; p++) {
		if (IsWildChar(*p)) {
			return 1;
		}
	}
	return 0;
}

int in_str(char *str, char c)
{
	char *s;

	for (s = str; *s != '\0'; s++) {
		if (*s == c) {
			return 1;
		}
	}
	return 0;
}

int valid_server_name(char *servername)
{
	char *s;

	for (s = servername; *s != '\0'; s++) {
		if (IsServChar(*s) && !IsWildChar(*s)) {
			return 1;
		}
	}
	return 0;
}

int valid_ip_addr(char *addr, int mask)
{
	char *s = addr;
	int dots = 0, pos = 1;

	if (mask) {
		int got_at = 0;

		while (*s != '\0' && !got_at) {
			if (*s == '@') {
				got_at = 1;
			}
			s++;
		}
		if (*s == '\0' && got_at) {
			return 0;
		}
		if (!got_at) {
			s = addr;
		}
	}

	if (*s == '*' && (*(s + 1) == '\0')) {
		return 1;
	}

	while (*s != '\0') {
		if (*s == '.') {
			dots++;
			pos = 1;
		}
		else if (pos > 3) {
			return 0;
		}
		s++;
	}

	return (dots == 3) ? 1 : 0;
}

int valid_email_addr(char *addr, int allow_url)
{
	char *s;
	for (s = addr; *s != '\0'; s++) {
		if (*s == '@' || (allow_url && (*s == ':'))) {
			return 1;
		}
	}
	return 0;
}

int get_port(char *str)
{
	int port = atoi(str);

	if (port < 1 || port > 65535) {
		return 0;
	}
	return port;
}

void get_port_range(char *str, int *first_port, int *last_port)
{
	char *s;

	if ((s = strchr(str, '-')) == NULL) {
		*first_port = *last_port = get_port(str);
		return;
	}
	*s = '\0';

	*first_port = get_port(str);
	*last_port = get_port(s + 1);

	return;
}

long get_size(char *str)
{
	char *s, *sz;
	long ret = 0;
	int factor = 1;

	for (s = str; *s != '\0'; s++) {
		if (!IsAlpha(*s)) {
			continue;
		}

		if (ToLower(*s) == 'k') {
			factor = 1024;
		}
		else if (ToLower(*s) == 'm') {
			factor = (1024 * 1024);
		}
		else if (ToLower(*s) == 'g') {
			factor = (1024 * 1024 * 1024);
		}

		sz = s;
		while (IsAlpha(*s)) {
			s++;
		}

		*sz-- = '\0';
		while (sz-- > str && (*sz != '\0')) {
			if (IsSpace(*s)) {
				*sz = '\0';
			}
			if (!IsDigit(*s)) {
				break;
			}
		}
		ret += (atoi(sz + 1) * factor);

		if (*s == '\0') {
			s++;
			break;
		}
	}

	factor = 1;
	sz = s;
	sz--;

	while (sz-- > str) {
		if (IsSpace(*sz)) {
			*sz = '\0';
		}
		if (!IsDigit(*sz)) {
			break;
		}
	}

	ret += (atoi(sz + 1) * factor);
	return ret;
}

unsigned int get_option(char *p, unsigned int flag)
{
	if (*p == '1' || (ToLower(*p) == 'o' && (ToLower(*(p + 1) == 'n')))
	  || ToLower(*p) == 'y' || ToLower(*p) == 't') {
		return flag;
	}
	return 0;
}

char *pretty_mask(char *mask)
{
        char *cp = mask, *user, *host;

	if ((user = strchr(cp, '!')) != NULL) {
		*user++ = '\0';
	}
	if ((host = strrchr((user != NULL) ? user : cp, '@')) != NULL) {
		*host++ = '\0';
		if (user == NULL) {
			return make_nick_user_host(NULL, cp, host);
		}
	}
	else if (user == NULL && (strchr(cp, '.') != NULL)) {
		return make_nick_user_host(NULL, NULL, cp);
	}

	return make_nick_user_host(cp, user, host);
}

