/*
 *  Copyright 2001-2003 Adrian Thurston <adriant@ragel.ca>
 */

/*  This file is part of Ragel.
 *
 *  Ragel is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 * 
 *  Ragel is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *  You should have received a copy of the GNU General Public License
 *  along with Ragel; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 */

#include "ragel.h"
#include "tabcodegen.h"
#include "parsetree.h"
#include "redfsm.h"

#define SPEC_ANY_SINGLE  0x01
#define SPEC_ANY_RANGE   0x02
#define SPEC_EOF_ACTION  0x04

/* Integer array line length. */
#define IALL 8

/* Pass init data to base class. */
TabCodeGen::TabCodeGen( char *fsmName, ParseData *parseData, 
		RedFsmAp *redFsm, std::ostream &out )
:
	FsmCodeGen(fsmName, parseData, redFsm, out)
{ }

int TabCodeGen::makeStateSpec( RedStateAp *state )
{
	int retSpec = 0;
	/* Single index. */
	if ( state->outSingle.length() > 0 )
		retSpec |= SPEC_ANY_SINGLE;
	/* Range index. */
	if ( state->outRange.length() > 0 )
		retSpec |= SPEC_ANY_RANGE;
	/* Out action? */
	if ( state->eofAction != 0 )
		retSpec |= SPEC_EOF_ACTION;
	return retSpec;
}

int TabCodeGen::getStateLen( int stateSpec )
{
	/* Basic len of 3 bytes. */
	int stateLen = 3;
	if ( stateSpec & SPEC_ANY_SINGLE )
		stateLen += 1;
	if ( stateSpec & SPEC_ANY_RANGE )
		stateLen += 1;
	if ( stateSpec & SPEC_EOF_ACTION )
		stateLen += 1;
	return stateLen;
}

void TabCodeGen::calcStatePositions()
{
	/* Since states are variable width, we must compute their locations. Loop
	 * the states, keeping a calc of the length. */
	int nextPosition = 0;
	for ( RedStateList::Iter st = redFsm->stateList; st.lte(); st++ ) {
		/* Store the length. */
		st->position = nextPosition;

		/* Skip over the state. */
		int stateSpec = makeStateSpec( st );
		nextPosition += getStateLen( stateSpec );
	}
}

std::ostream &TabCodeGen::STATE_OUT_ACTION( RedStateAp *state )
{
	/* This function is only called if there are any out actions, so need
	 * not guard against there being none. */
	out << state->eofAction->location+1;
	return out;
}

std::ostream &TabCodeGen::TRANS_ACTION( RedTransAp *trans )
{
	/* If there are actions, emit them. Otherwise emit zero. */
	if ( trans->action != 0 )
		out << trans->action->location+1;
	else
		out << "0";
	return out;
}

std::ostream &TabCodeGen::FIRST_FINAL_STATE()
{
	out << redFsm->firstFinState->position;
	return out;
}

std::ostream &TabCodeGen::ERROR_STATE()
{
	out << redFsm->errState->position;
	return out;
}

std::ostream &TabCodeGen::ACTION_SWITCH()
{
	/* Walk the list of functions, printing the cases. */
	for ( ActionList::Iter act = parseData->actionList; act.lte(); act++ ) {
		/* Write out referenced actions. */
		if ( act->numTransRefs > 0 || act->numEofRefs > 0 ) {
			/* Get the function data. Print the case label.  */
			out << "\tcase " << act->id << ":\n";

			/* Write the preprocessor line info for going into the source file. */
			out << "#line " << act->loc.line << " \""; LDIR_PATH(inputFile) << "\"\n";

			/* Write the block and close it off. */
			out << "\t{"; ACTION(act, 0, false) << "} break;\n";
		}
	}

	/* Write the directive for going back into the output file. The line
	 * number is for the next line, so add one. */
	out << "#line " << outFilter->line + 1 << " \""; LDIR_PATH(outputFile) << "\"\n";
	return out;
}

std::ostream &TabCodeGen::STATES()
{
	/* Initial indent. */
	out << "\t";
	int curKeyOffset = 0, curIndOffset = 0;
	int totalStateNum = 0;
	for ( RedStateList::Iter st = redFsm->stateList; st.lte(); st++ ) {
		/* Get the spec and write it and the length. */
		int stateSpec = makeStateSpec( st );
		int stateLen = getStateLen( stateSpec );
		out << ( ( stateLen << 8 ) |  stateSpec ) << ", ";
		if ( ++totalStateNum % IALL == 0 )
			out << "\n\t";

		/* Write the key offset. */
		out << curKeyOffset << ", ";
		if ( ++totalStateNum % IALL == 0 )
			out << "\n\t";

		/* Write the index offset. */
		out << curIndOffset << ", ";
		if ( ++totalStateNum % IALL == 0 )
			out << "\n\t";

		/* Emit length of the singles index. */
		if ( stateSpec & SPEC_ANY_SINGLE ) {
			out << st->outSingle.length() << ", ";
			if ( ++totalStateNum % IALL == 0 )
				out << "\n\t";
		}

		/* Emit length of range index. */
		if ( stateSpec & SPEC_ANY_RANGE ) {
			out << st->outRange.length() << ", ";
			if ( ++totalStateNum % IALL == 0 )
				out << "\n\t";
		}

		/* If there is an out func, write it. */
		if ( stateSpec & SPEC_EOF_ACTION ) {
			STATE_OUT_ACTION( st ) << ", ";
			if ( ++totalStateNum % IALL == 0 )
				out << "\n\t";
		}

		/* Move the key and index offsets ahead. */
		curKeyOffset += st->outSingle.length() + st->outRange.length()*2;
		curIndOffset += st->outSingle.length() + st->outRange.length();
		if ( st->defTrans != 0 )
			curIndOffset += 1;
	}

	/* Output one last number so we don't have to figure out when the last
	 * entry is and avoid writing a comma. */
	out << 0 << "\n";
	return out;
}

std::ostream &TabCodeGen::KEYS()
{
	out << '\t';
	int totalTrans = 0;
	for ( RedStateList::Iter st = redFsm->stateList; st.lte(); st++ ) {
		/* Loop the singles. */
		for ( RedTransList::Iter stel = st->outSingle; stel.lte(); stel++ ) {
			KEY( stel->lowKey ) << ", ";
			if ( ++totalTrans % IALL == 0 )
				out << "\n\t";
		}

		/* Loop the state's transitions. */
		for ( RedTransList::Iter rtel = st->outRange; rtel.lte(); rtel++ ) {
			/* Lower key. */
			KEY( rtel->lowKey ) << ", ";
			if ( ++totalTrans % IALL == 0 )
				out << "\n\t";

			/* Upper key. */
			KEY( rtel->highKey ) << ", ";
			if ( ++totalTrans % IALL == 0 )
				out << "\n\t";
		}
	}

	/* Output one last number so we don't have to figure out when the last
	 * entry is and avoid writing a comma. */
	out << 0 << "\n";
	return out;
}

std::ostream &TabCodeGen::INDICIES()
{
	int totalTrans = 0;
	out << '\t';
	for ( RedStateList::Iter st = redFsm->stateList; st.lte(); st++ ) {
		/* Walk the singles. */
		for ( RedTransList::Iter stel = st->outSingle; stel.lte(); stel++ ) {
			out << stel->value->id << ", ";
			if ( ++totalTrans % IALL == 0 )
				out << "\n\t";
		}

		/* Walk the ranges. */
		for ( RedTransList::Iter rtel = st->outRange; rtel.lte(); rtel++ ) {
			out << rtel->value->id << ", ";
			if ( ++totalTrans % IALL == 0 )
				out << "\n\t";
		}

		/* The state's default index goes next. */
		if ( st->defTrans != 0 ) {
			out << st->defTrans->id << ", ";
			if ( ++totalTrans % IALL == 0 )
				out << "\n\t";
		}
	}

	/* Output one last number so we don't have to figure out when the last
	 * entry is and avoid writing a comma. */
	out << 0 << "\n";
	return out;
}

std::ostream &TabCodeGen::INDEX_TYPE()
{
	if ( redFsm->transSet.length() <= 256 )
		out << "unsigned char";
	else if ( redFsm->transSet.length() <= 256*256 )
		out << "unsigned short";
	else 
		out << "unsigned int";
	return out;
}


std::ostream &TabCodeGen::TRANSITIONS()
{
	/* Transitions must be written ordered by their id. */
	RedTransAp **transPtrs = new RedTransAp*[redFsm->transSet.length()];
	for ( TransApSet::Iter trans = redFsm->transSet; trans.lte(); trans++ )
		transPtrs[trans->id] = trans;

	/* Keep a count of the num of items in the array written. */
	out << '\t';
	int totalTransData = 0;
	for ( int t = 0; t < redFsm->transSet.length(); t++ ) {
		/* Write out the target state. */
		RedTransAp *trans = transPtrs[t];
		out << trans->targ->position << ", ";
		if ( ++totalTransData % IALL == 0 )
			out << "\n\t";

		/* Write the function for the transition. */
		TRANS_ACTION( trans ) << ", ";
		if ( ++totalTransData % IALL == 0 )
			out << "\n\t";
	}

	out << "0" << "\n";
	delete[] transPtrs;
	return out;
}


std::ostream &TabCodeGen::BSEARCH()
{
	out <<
		"static "; ALPH_TYPE() << " *"; FSM_NAME() << "_bsearch( "; ALPH_TYPE() << 
				" c, "; ALPH_TYPE() << " *keys, int len )\n"
		"{\n"
		"	"; ALPH_TYPE() << " *lower = keys;\n"
		"	"; ALPH_TYPE() << " *mid;\n"
		"	"; ALPH_TYPE() << " *upper = keys + len - 1;\n"
		"	while (1) {\n"
		"		if ( upper < lower )\n"
		"			return 0;\n"
		"\n"
		"		mid = lower + ((upper-lower) >> 1);\n"
		"		if ( c < *mid )\n"
		"			upper = mid - 1;\n"
		"		else if ( c > *mid )\n"
		"			lower = mid + 1;\n"
		"		else\n"
		"			return mid;\n"
		"	}\n"
		"}\n";

	return out;
}

std::ostream &TabCodeGen::RANGE_BSEARCH()
{
	out <<
		"static "; ALPH_TYPE() << " *"; FSM_NAME() << "_range_bsearch( "; ALPH_TYPE() << 
				" c, "; ALPH_TYPE() << " *keys, int len )\n"
		"{\n"
		"	"; ALPH_TYPE() << " *lower = keys;\n"
		"	"; ALPH_TYPE() << " *mid;\n"
		"	"; ALPH_TYPE() << " *upper = keys + len - 2;\n"
		"	while (1) {\n"
		"		if ( upper < lower )\n"
		"			return 0;\n"
		"\n"
		"		mid = lower + (((upper-lower) >> 1) & ~1);\n"
		"		if ( c < mid[0] )\n"
		"			upper = mid - 2;\n"
		"		else if ( c > mid[1] )\n"
		"			lower = mid + 2;\n"
		"		else\n"
		"			return mid;\n"
		"	}\n"
		"}\n";

	return out;
}

std::ostream &TabCodeGen::LOCATE_TRANS()
{
	out <<
		"	_specs = *_pcs++;\n"
		"	_keys = "; FSM_NAME() << "_k + *_pcs++;\n"
		"	_inds = "; FSM_NAME() << "_i + *_pcs++;\n"
		"\n"
		"	if ( _specs & SPEC_ANY_SINGLE ) {\n"
		"		int indsLen = *_pcs++;\n"
		"		"; ALPH_TYPE() << " *match = "; FSM_NAME() << 
						"_bsearch( "; GET_KEY() << ", _keys, indsLen );\n"
		"		if ( match != 0 ) {\n"
		"			_trans = "; FSM_NAME() << "_t + (_inds[match - _keys]<<1);\n"
		"			goto match;\n"
		"		}\n"
		"		_keys += indsLen;\n"
		"		_inds += indsLen;\n"
		"	}\n"
		"\n"
		"	if ( _specs & SPEC_ANY_RANGE ) {\n"
		"		int indsLen = *_pcs++;\n"
		"		"; ALPH_TYPE() << " *match = "; FSM_NAME() << 
						"_range_bsearch( "; GET_KEY() << ", _keys, (indsLen<<1) );\n"
		"		if ( match != 0 ) {\n"
		"			_trans = "; FSM_NAME() << "_t + (_inds[(match - _keys)>>1]<<1);\n"
		"			goto match;\n"
		"		}\n"
		"		_keys += (indsLen<<1);\n"
		"		_inds += indsLen;\n"
		"	}\n"
		"\n"
		"	_trans = "; FSM_NAME() << "_t + ((*_inds)<<1);\n";

	return out;
}

std::ostream &TabCodeGen::STACK( int size )
{
	out << "int _top, _st[" << size << "];";
	return out;
}

std::ostream &TabCodeGen::GOTO( NameInst *name, bool inFinish )
{
	/* Lookup the target. Always guaranteed to return just one target. */
	RedEntryMapEl *targ = redFsm->entryMap.find( name->id );
	out << "{_cs = " << targ->value->position << "; goto again;}";
	return out;
}

std::ostream &TabCodeGen::GOTOE( char *stateExpr, bool inFinish )
{
	out << "{_cs = (" << stateExpr << "); goto again;}";
	return out;
}

std::ostream &TabCodeGen::CURS( bool inFinish )
{
	out << "(_ps)";
	return out;
}

std::ostream &TabCodeGen::TARGS( bool inFinish, int targState )
{
	out << "(_cs)";
	return out;
}

std::ostream &TabCodeGen::GET_STATE( NameInst *name, bool inFinish )
{
	RedEntryMapEl *targ = redFsm->entryMap.find( name->id );
	out << "("; FSM_NAME() << "_s+" << targ->value->position << ")";
	return out;
}

std::ostream &TabCodeGen::NEXT( NameInst *name, bool inFinish )
{
	RedEntryMapEl *targ = redFsm->entryMap.find( name->id );
	out << "_cs = " << targ->value->position << ";";
	return out;
}

std::ostream &TabCodeGen::NEXTE( char *stateExpr, bool inFinish )
{
	out << "_cs = (" << stateExpr << ");";
	return out;
}

/* Pass init data to base. */
CTabCodeGen::CTabCodeGen( char *fsmName, ParseData *parseData, 
		RedFsmAp *redFsm, std::ostream &out )
:
	TabCodeGen(fsmName, parseData, redFsm, out)
{ }

std::ostream &CTabCodeGen::CALL( NameInst *name, int targState, bool inFinish )
{
	/* Lookup the target. Always guaranteed to return just one target. */
	RedEntryMapEl *targ = redFsm->entryMap.find( name->id );
	out << "{fsm->_st[fsm->_top++] = _cs; _cs = " << 
			targ->value->position << "; goto again;}";
	return out;
}

std::ostream &CTabCodeGen::RET( bool inFinish )
{
	out << "{_cs = fsm->_st[--fsm->_top]; goto again;}";
	return out;
}

std::ostream &CTabCodeGen::CALLE( char *stateExpr, int targState, bool inFinish )
{
	out << "{fsm->_st[fsm->_top++] = _cs; _cs = (" << stateExpr << "); goto again;}";
	return out;
}

/* Generate the header portion. */
void CTabCodeGen::writeOutHeader()
{
	out << 
		"/* Current state and user data. */\n"
		"struct "; FSM_NAME() << "\n"
		"{\n"
		"	int _cs;\n";
		STRUCT_DATA() <<
		"};\n"
		"\n"
		"/* Init the fsm. */\n"
		"int "; FSM_NAME() << "_init( struct "; FSM_NAME() << " *fsm );\n"
		"\n"
		"/* Execute some chunk of data. */\n"
		"int "; FSM_NAME() << "_execute( struct "; FSM_NAME() << " *fsm, ";
				EL_TYPE() << " *data, int len );\n"
		"\n"
		"/* Indicate to the fsm tha there is no more data. */\n"
		"int "; FSM_NAME() << "_finish( struct "; FSM_NAME() << " *fsm );\n"
		"\n";
}

void CTabCodeGen::writeOutCode()
{
	/* Need the state positions. */
	calcStatePositions();
	
	/* State machine data. */
	out << 
		"#define SPEC_ANY_SINGLE  0x01\n"
		"#define SPEC_ANY_RANGE   0x02\n"
		"#define SPEC_EOF_ACTION  0x04\n"
		"\n";

	/* If there are any transtion functions then output the array. If there
	 * are none, don't bother emitting an empty array that won't be used. */
	if ( anyActions() ) {
		out <<
			"static int "; FSM_NAME() << "_a[] = {\n";
			ACTIONS_ARRAY() <<
			"};\n"
			"\n";
	}

	/* Write the array of keys. */
	out <<
 		"static "; ALPH_TYPE() << " "; FSM_NAME() << "_k[] = {\n";
		KEYS() <<
		"};\n"
		"\n";

	/* Write the array of indicies. */
	out << 
		"static "; INDEX_TYPE() << " "; FSM_NAME() << "_i[] = {\n";
		INDICIES() <<
		"};\n"
		"\n";

	/* Write the array of states. */
	out <<
		"static int "; FSM_NAME() << "_s[] = {\n";
		STATES() <<
		"};\n"
		"\n"
		"static int "; FSM_NAME() << "_t[] = {\n";
		TRANSITIONS() <<
		"};\n"
		"\n"
		"static int "; FSM_NAME() << "_start = " 
				<< redFsm->startState->position << ";\n"
		"\n";

	/* Init routine. */
	out << 
		"int "; FSM_NAME() << "_init( struct "; FSM_NAME() << " *fsm )\n"
		"{\n"
		"	fsm->_cs = "; FSM_NAME() << "_start;\n";
	
	/* If there are any calls, then the stack top needs initialization. */
	if ( anyActionCalls() || anyActionRets() )
		out << "	fsm->_top = 0;\n";

	INIT_CODE() <<
		"	if ( fsm->_cs >= "; FIRST_FINAL_STATE() << " )\n"
		"		return 1;\n"
		"	return 0;\n"
		"}\n"
		"\n";

	/* The binary search for singles. */
	BSEARCH() << "\n";

	/* The binary search for a range. */
	RANGE_BSEARCH() << "\n";

	/* Execution function. */
	out << 
		"int "; FSM_NAME() << "_execute( struct "; FSM_NAME() << " *fsm, "; EL_TYPE() <<
				" *_data, int _len )\n"
		"{\n"
		"	"; EL_TYPE() << " *_p = _data-1;\n"
		"	"; EL_TYPE() << " *_pe = _data+_len;\n"
		"	int _cs = fsm->_cs, *_pcs";

	if ( anyRegCurStateRef() )
		out << ", _ps";
	out << ";\n\tint _specs, *_trans";
		
	if ( anyActions() )
		out << ", *_acts, _nacts";

	out <<
		";\n"
		"	"; ALPH_TYPE() << " *_keys;\n"
		"	"; INDEX_TYPE() << " *_inds;\n"
		"\n";


	out <<
		"again:\n"
		"	if ( ++_p == _pe )\n"
		"		goto out;\n"
		"	_pcs = "; FSM_NAME() << "_s + _cs;\n";

	if ( anyEofActions() ) {
		out <<
			"	if ( _p == 0 )\n"
			"		goto eofActions;\n"
			"\n";
	}
	
	LOCATE_TRANS() <<
		"\n"
		"match:\n";

	if ( anyRegCurStateRef() )
		out << "	_ps = _cs;\n";

	out <<
		"	_cs = *_trans++;\n"
		"\n";

	if ( anyActions() ) {
		out << 
			"	if ( *_trans == 0 )\n"
			"		goto again;\n"
			"	_acts = "; FSM_NAME() << "_a + *_trans;\n"
			"\n";

		/* Only used if there are out actions. */
		if ( anyEofActions() )
			out << "execFuncs:\n";

		out <<
			"	_nacts = *_acts++;\n"
			"	while ( _nacts-- > 0 ) {\n"
			"		switch ( *_acts++ ) {\n";
			ACTION_SWITCH() <<
			"		}\n"
			"	}\n"
			"\n";
	}

	out <<
		"	goto again;\n"
		"\n";

	if ( anyEofActions() ) {
		out <<
			"eofActions:\n"
			"	if ( *_pcs & SPEC_EOF_ACTION ) {\n"
			"		_acts = "; FSM_NAME() << "_a+*(_pcs + (*_pcs>>8)-1);\n"
			"		_p = 0; _pe = (("; EL_TYPE() << "*)0)+1;\n"
			"		goto execFuncs;\n"
			"	}\n"
			"\n";
	}

	out <<
		"out:\n"
		"	fsm->_cs = _cs;\n";

	if ( redFsm->errState != 0 ) {
		out << 
			"	if ( _cs == "; ERROR_STATE() << " )\n"
			"		return -1;\n";
	}

	out << 
		"	if ( _cs >= "; FIRST_FINAL_STATE() << " )\n"
		"		return 1;\n"
		"	return 0;\n"
		"}\n"
		"\n";

	/* Finish routine. */
	out <<
		"int "; FSM_NAME() << "_finish( struct "; FSM_NAME() << " *fsm )\n"
		"{\n";

	if ( anyEofActions() ) {
		/* May need to execute code in the action loop. Must use special
		 * params to execute func. */
		out << "	return "; FSM_NAME() << "_execute( fsm, 0, 1 );\n";
	}
	else {
		if ( redFsm->errState != 0 ) {
			out << 
				"	if ( fsm->_cs == "; ERROR_STATE() << " )\n"
				"		return -1;\n";
		}

		out <<
			"	if ( fsm->_cs >= "; FIRST_FINAL_STATE() << " )\n"
			"		return 1;\n"
			"	return 0;\n";
	}

	out << 
		"}\n"
		"\n";

	/* Cleanup after ourselves. */
	out <<
		"#undef SPEC_ANY_SINGLE\n"
		"#undef SPEC_ANY_RANGE\n"
		"#undef SPEC_EOF_ACTION\n"
		"\n";
}

/* Init base class. */
CCTabCodeGen::CCTabCodeGen( char *fsmName, ParseData *parseData, 
		RedFsmAp *redFsm, std::ostream &out ) 
: 
	TabCodeGen(fsmName, parseData, redFsm, out) 
{ }

std::ostream &CCTabCodeGen::CALL( NameInst *name, int targState, bool inFinish )
{
	/* Lookup the target. Always guaranteed to return just one target. */
	RedEntryMapEl *targ = redFsm->entryMap.find( name->id );
	out << "{this->_st[this->_top++] = _cs; _cs = " << 
			targ->value->position << "; goto again;}";
	return out;
}

std::ostream &CCTabCodeGen::RET( bool inFinish )
{
	out << "{_cs = this->_st[--this->_top]; goto again;}";
	return out;
}

std::ostream &CCTabCodeGen::CALLE( char *stateExpr, int targState, bool inFinish )
{
	/* Lookup the target. Always guaranteed to return just one target. */
	out << "{this->_st[this->_top++] = _cs; _cs = (" << stateExpr << "); goto again;}";
	return out;
}

void CCTabCodeGen::writeOutHeader()
{
	out <<
		"class "; FSM_NAME() << "\n"
		"{\n"
		"public:\n"
		"	int _cs;\n"
		"\n"
		"	/* Initialize the machine for execution. */\n"
		"	int init( );\n"
		"\n"
		"	/* Execute some chunk of data. */\n"
		"	int execute( "; EL_TYPE() << " *data, int len );\n"
		"\n"
		"	/* Indicate to the fsm tha there is no more data. */\n"
		"	int finish( );\n";
		STRUCT_DATA() <<
		"};\n"
		"\n";
}

void CCTabCodeGen::writeOutCode()
{
	/* Need the state positions. */
	calcStatePositions();
	
	/* State machine data. */
	out << 
		"#define SPEC_ANY_SINGLE  0x01\n"
		"#define SPEC_ANY_RANGE   0x02\n"
		"#define SPEC_EOF_ACTION  0x04\n"
		"\n";

	/* If there are any transtion functions then output the array. If there
	 * are none, don't bother emitting an empty array that won't be used. */
	if ( anyActions() ) {
		out <<
			"static int "; FSM_NAME() << "_a[] = {\n";
			ACTIONS_ARRAY() <<
			"};\n"
			"\n";
	}

	/* Write the array of keys. */
	out <<
		"static "; ALPH_TYPE() << " "; FSM_NAME() << "_k[] = {\n";
		KEYS() <<
		"};\n"
		"\n";

	/* Write the array of indicies. */
	out << 
		"static "; INDEX_TYPE() << " "; FSM_NAME() << "_i[] = {\n";
		INDICIES() <<
		"};\n"
		"\n";

	/* Write the array of states. */
	out <<
		"static int "; FSM_NAME() << "_s[] = {\n";
		STATES() <<
		"};\n"
		"\n"
		"static int "; FSM_NAME() << "_t[] = {\n";
		TRANSITIONS() <<
		"};\n"
		"\n"
		"static int "; FSM_NAME() << "_start = " 
				<< redFsm->startState->position << ";\n"
		"\n";

	/* Init routine. */
	out << 
		"int "; FSM_NAME() << "::init( )\n"
		"{\n"
		"	this->_cs = "; FSM_NAME() << "_start;\n";
	
	/* If there are any calls, then the stack top needs initialization. */
	if ( anyActionCalls() || anyActionRets() )
		out << "	this->_top = 0;\n";

	INIT_CODE() <<
		"	if ( this->_cs >= "; FIRST_FINAL_STATE() << " )\n"
		"		return 1;\n"
		"	return 0;\n"
		"}\n"
		"\n";

	/* The binary search for singles. */
	BSEARCH() << "\n";

	/* The binary search for a range. */
	RANGE_BSEARCH() << "\n";

	/* Execution function. */
	out << 
		"int "; FSM_NAME() << "::execute( "; EL_TYPE() << " *_data, int _len )\n"
		"{\n"
		"	"; EL_TYPE() << " *_p = _data-1;\n"
		"	"; EL_TYPE() << " *_pe = _data+_len;\n"
		"	int _cs = this->_cs, *_pcs";

	if ( anyRegCurStateRef() )
		out << ", _ps";
	out << ";\n\tint _specs, *_trans";

	if ( anyActions() )
		out << ", *_acts, _nacts";

	out <<
		";\n"
		"	"; ALPH_TYPE() << " *_keys;\n"
		"	"; INDEX_TYPE() << " *_inds;\n"
		"\n";
	

	out << 
		"again:\n"
		"	if ( ++_p == _pe )\n"
		"		goto out;\n"
		"	_pcs = "; FSM_NAME() << "_s + _cs;\n";

	if ( anyEofActions() ) {
		out << 
			"	if ( _p == 0 )\n"
			"		goto eofActions;\n"
			"\n";
	}

	LOCATE_TRANS() <<
		"\n"
		"match:\n";

	if ( anyRegCurStateRef() )
		out << "	_ps = _cs;\n";

	out <<
		"	_cs = *_trans++;\n"
		"\n";

	if ( anyActions() ) {
		out <<
			"	if ( *_trans == 0 )\n"
			"		goto again;\n"
			"	_acts = "; FSM_NAME() << "_a + *_trans;\n"
			"\n";

		/* Only used if there are out actions. */
		if ( anyEofActions() )
			out << "execFuncs:\n";

		out <<
			"	_nacts = *_acts++;\n"
			"	while ( _nacts-- > 0 ) {\n"
			"		switch ( *_acts++ ) {\n";
			ACTION_SWITCH() <<
			"		}\n"
			"	}\n"
			"\n";
	}

	out <<
		"	goto again;\n"
		"\n";
	
	if ( anyEofActions() ) {
		out << 
			"eofActions:\n"
			"	if ( *_pcs & SPEC_EOF_ACTION ) {\n"
			"		_acts = "; FSM_NAME() << "_a+*(_pcs + (*_pcs>>8)-1);\n"
			"		_p = 0; _pe = (("; EL_TYPE() << "*)0)+1;\n"
			"		goto execFuncs;\n"
			"	}\n"
			"\n";
	}

	out <<
		"out:\n"
		"	this->_cs = _cs;\n";

	if ( redFsm->errState != 0 ) {
		out << 
			"	if ( _cs == "; ERROR_STATE() << " )\n"
			"		return -1;\n";
	}

	out <<
		"	if ( _cs >= "; FIRST_FINAL_STATE() << " )\n"
		"		return 1;\n"
		"	return 0;\n"
		"}\n"
		"\n";

	/* Finish routine. */
	out <<
		"int "; FSM_NAME() << "::finish( )\n"
		"{\n";

	if ( anyEofActions() ) {
		/* May need to execute code in the action loop. Must use special
		 * params to execute func. */
		out << "	return execute( 0, 1 );\n";
	}
	else {
		if ( redFsm->errState != 0 ) {
			out << 
				"	if ( this->_cs == "; ERROR_STATE() << " )\n"
				"		return -1;\n";
		}

		out <<
			"	if ( this->_cs >= "; FIRST_FINAL_STATE() << " )\n"
			"		return 1;\n"
			"	return 0;\n";
	}

	out << 
		"}\n"
		"\n";

	/* Cleanup after ourselves. */
	out <<
		"#undef SPEC_ANY_SINGLE\n"
		"#undef SPEC_ANY_RANGE\n"
		"#undef SPEC_EOF_ACTION\n"
		"\n";
}
