/*
 *  Copyright 2004 Adrian Thurston <thurston@cs.queensu.ca>
 */

/*  This file is part of Ragel.
 *
 *  Ragel is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 * 
 *  Ragel is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *  You should have received a copy of the GNU General Public License
 *  along with Ragel; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 */

#include "ragel.h"
#include "flatcodegen.h"
#include "parsetree.h"
#include "redfsm.h"

std::ostream &FlatCodeGen::STATE_OUT_ACTION( RedStateAp *state )
{
	/* This function is only called if there are any out actions, so need
	 * not guard against there being none. */
	int act = 0;
	if ( state->eofAction != 0 )
		act = state->eofAction->location+1;
	out << act;
	return out;
}

std::ostream &FlatCodeGen::TRANS_ACTION( RedTransAp *trans )
{
	/* If there are actions, emit them. Otherwise emit zero. */
	int act = 0;
	if ( trans->action != 0 )
		act = trans->action->location+1;
	out << act;
	return out;
}

std::ostream &FlatCodeGen::FIRST_FINAL_STATE()
{
	if ( redFsm->firstFinState != 0 )
		out << redFsm->firstFinState->id;
	else
		out << redFsm->nextStateId;
	return out;
}

std::ostream &FlatCodeGen::ERROR_STATE()
{
	out << redFsm->errState->id;
	return out;
}

std::ostream &FlatCodeGen::ACTION_SWITCH()
{
	/* Walk the list of functions, printing the cases. */
	for ( ActionList::Iter act = parseData->actionList; act.lte(); act++ ) {
		/* Write out referenced actions. */
		if ( act->numTransRefs > 0 || act->numEofRefs > 0 ) {
			/* Write the case label, the action and the case break */
			out << "\tcase " << act->actionId << ":\n";
			ACTION( act, 0, false );
			out << "\tbreak;\n";
		}
	}

	/* Write the directive for going back into the output file. The line
	 * number is for the next line, so add one. */
	out << "#line " << outFilter->line + 1 << " \""; LDIR_PATH(outputFile) << "\"\n";
	return out;
}


std::ostream &FlatCodeGen::FLAT_INDEX_OFFSET()
{
	out << "\t";
	int totalStateNum = 0, curIndOffset = 0;
	for ( RedStateList::Iter st = redFsm->stateList; st.lte(); st++ ) {
		/* Write the index offset. */
		out << curIndOffset;
		if ( !st.last() ) {
			out << ", ";
			if ( ++totalStateNum % IALL == 0 )
				out << "\n\t";
		}
		
		/* Move the index offset ahead. */
		if ( st->transList != 0 )
			curIndOffset += redFsm->keyOps->span( st->lowKey, st->highKey );
		curIndOffset += 1;
	}
	out << "\n";
	return out;
}

std::ostream &FlatCodeGen::KEY_SPANS()
{
	out << "\t";
	int totalStateNum = 0;
	for ( RedStateList::Iter st = redFsm->stateList; st.lte(); st++ ) {
		/* Write singles length. */
		unsigned long long span = 0;
		if ( st->transList != 0 )
			span = redFsm->keyOps->span( st->lowKey, st->highKey );
		out << span;
		if ( !st.last() ) {
			out << ", ";
			if ( ++totalStateNum % IALL == 0 )
				out << "\n\t";
		}
	}
	out << "\n";
	return out;
}

std::ostream &FlatCodeGen::EOF_ACTIONS()
{
	out << "\t";
	int totalStateNum = 0;
	for ( RedStateList::Iter st = redFsm->stateList; st.lte(); st++ ) {
		/* Write any eof action. */
		STATE_OUT_ACTION(st);
		if ( !st.last() ) {
			out << ", ";
			if ( ++totalStateNum % IALL == 0 )
				out << "\n\t";
		}
	}
	out << "\n";
	return out;
}

std::ostream &FlatCodeGen::KEYS()
{
	out << '\t';
	int totalTrans = 0;
	for ( RedStateList::Iter st = redFsm->stateList; st.lte(); st++ ) {
		/* Emit just low key and high key. */
		KEY( st->lowKey ) << ", ";
		KEY( st->highKey ) << ", ";
		if ( ++totalTrans % IALL == 0 )
			out << "\n\t";
	}

	/* Output one last number so we don't have to figure out when the last
	 * entry is and avoid writing a comma. */
	out << 0 << "\n";
	return out;
}

std::ostream &FlatCodeGen::INDICIES()
{
	int totalTrans = 0;
	out << '\t';
	for ( RedStateList::Iter st = redFsm->stateList; st.lte(); st++ ) {
		if ( st->transList != 0 ) {
			/* Walk the singles. */
			unsigned long long span = redFsm->keyOps->span( st->lowKey, st->highKey );
			for ( unsigned long long pos = 0; pos < span; pos++ ) {
				out << st->transList[pos]->id << ", ";
				if ( ++totalTrans % IALL == 0 )
					out << "\n\t";
			}
		}

		/* The state's default index goes next. */
		out << st->defTrans->id << ", ";
		if ( ++totalTrans % IALL == 0 )
			out << "\n\t";
	}

	/* Output one last number so we don't have to figure out when the last
	 * entry is and avoid writing a comma. */
	out << 0 << "\n";
	return out;
}

std::ostream &FlatCodeGen::TRANS_TARGS()
{
	/* Transitions must be written ordered by their id. */
	RedTransAp **transPtrs = new RedTransAp*[redFsm->transSet.length()];
	for ( TransApSet::Iter trans = redFsm->transSet; trans.lte(); trans++ )
		transPtrs[trans->id] = trans;

	/* Keep a count of the num of items in the array written. */
	out << '\t';
	int totalStates = 0;
	for ( int t = 0; t < redFsm->transSet.length(); t++ ) {
		/* Write out the target state. */
		RedTransAp *trans = transPtrs[t];
		out << trans->targ->id;
		if ( t < redFsm->transSet.length()-1 ) {
			out << ", ";
			if ( ++totalStates % IALL == 0 )
				out << "\n\t";
		}
	}
	out << "\n";
	delete[] transPtrs;
	return out;
}


std::ostream &FlatCodeGen::TRANS_ACTIONS()
{
	/* Transitions must be written ordered by their id. */
	RedTransAp **transPtrs = new RedTransAp*[redFsm->transSet.length()];
	for ( TransApSet::Iter trans = redFsm->transSet; trans.lte(); trans++ )
		transPtrs[trans->id] = trans;

	/* Keep a count of the num of items in the array written. */
	out << '\t';
	int totalAct = 0;
	for ( int t = 0; t < redFsm->transSet.length(); t++ ) {
		/* Write the function for the transition. */
		RedTransAp *trans = transPtrs[t];
		TRANS_ACTION( trans );
		if ( t < redFsm->transSet.length()-1 ) {
			out << ", ";
			if ( ++totalAct % IALL == 0 )
				out << "\n\t";
		}
	}
	out << "\n";
	delete[] transPtrs;
	return out;
}

std::ostream &FlatCodeGen::LOCATE_TRANS()
{
	out <<
		"	_keys = "; FSM_NAME() << "_k + (_cs<<1);\n"
		"	_inds = "; FSM_NAME() << "_i + "; FSM_NAME() << "_io[_cs];\n"
		"\n"
		"	_slen = "; FSM_NAME() << "_sp[_cs];\n"
		"	if ( _slen > 0 && _keys[0] <="; GET_KEY() << " && "; GET_KEY() << " <= _keys[1] ) {\n"
		"		_trans = _inds["; GET_KEY() << " - _keys[0]];\n"
		"	}\n"
		"	else {\n"
		"		_trans = _inds[_slen];\n"
		"	}\n"
		"\n";

	return out;
}

std::ostream &FlatCodeGen::GOTO( NameInst *name, bool inFinish )
{
	/* Lookup the target. Always guaranteed to return just one target. */
	RedEntryMapEl *targ = redFsm->entryMap.find( name->id );
	out << "{_cs = " << targ->value->id << "; goto again;}";
	return out;
}

std::ostream &FlatCodeGen::GOTOE( InlineItem *ilItem, bool inFinish )
{
	out << "{_cs = (";
	INLINE_LIST( ilItem->children, 0, inFinish );
	out << "); goto again;}";
	return out;
}

std::ostream &FlatCodeGen::CURS( bool inFinish )
{
	out << "(_ps)";
	return out;
}

std::ostream &FlatCodeGen::TARGS( bool inFinish, int targState )
{
	out << "(_cs)";
	return out;
}

std::ostream &FlatCodeGen::GET_ENTRY( NameInst *name, bool inFinish )
{
	RedEntryMapEl *targ = redFsm->entryMap.find( name->id );
	out << targ->value->id;
	return out;
}

std::ostream &FlatCodeGen::NEXT( NameInst *name, bool inFinish )
{
	RedEntryMapEl *targ = redFsm->entryMap.find( name->id );
	out << "_cs = " << targ->value->id << ";";
	return out;
}

std::ostream &FlatCodeGen::NEXTE( InlineItem *ilItem, bool inFinish )
{
	out << "_cs = (";
	INLINE_LIST( ilItem->children, 0, inFinish );
	out << ");";
	return out;
}

std::ostream &FlatCodeGen::CALL( NameInst *name, int targState, bool inFinish )
{
	/* Lookup the target. Always guaranteed to return just one target. */
	RedEntryMapEl *targ = redFsm->entryMap.find( name->id );
	out << "{"; SELF() << "->stack["; SELF() << "->top++] = _cs; _cs = " << 
			targ->value->id << "; goto again;}";
	return out;
}


std::ostream &FlatCodeGen::CALLE( InlineItem *ilItem, int targState, bool inFinish )
{
	out << "{"; SELF() << "->stack["; SELF() << "->top++] = _cs; _cs = (";
	INLINE_LIST( ilItem->children, targState, inFinish );
	out << "); goto again;}";
	return out;
}


std::ostream &FlatCodeGen::RET( bool inFinish )
{
	out << "{_cs = "; SELF() << "->stack[--"; SELF() << "->top]; goto again;}";
	return out;
}

std::ostream &FlatCodeGen::CONTEXT( int contextId )
{
	/* Make a vect to fill with context indexed by state. */
	int numVals = (redFsm->nextStateId>>3)+1;
	unsigned char *vals = new unsigned char[numVals];
	memset( vals, 0, sizeof(unsigned char)*numVals );

	/* For each state where the context is present, set the bit. */
	for ( RedStateList::Iter st = redFsm->stateList; st.lte(); st++ ) {
		if ( st->contextSet.find( contextId ) )
			vals[st->id>>3] |= 1 << (st->id&0x7);
	}

	/* Write out context info. */
	for ( int i = 0; i < numVals; ) {
		out << (unsigned int)vals[i];

		i += 1;
		if ( i < numVals ) {
			out << ", ";
			if ( i % IALL == 0 )
				out << "\n\t";
		}
	}
	out << "\n";

	delete[] vals;
	return out;
}

std::ostream &FlatCodeGen::STATE_DATA()
{
	/* If there are any transtion functions then output the array. If there
	 * are none, don't bother emitting an empty array that won't be used. */
	if ( anyActions() ) {
		out <<
			"static "; ARRAY_TYPE(maxActArrItem) << " "; FSM_NAME() << "_a[] = {\n";
			ACTIONS_ARRAY() <<
			"};\n"
			"\n";
	}


	out <<
 		"static "; ALPH_TYPE() << " "; FSM_NAME() << "_k[] = {\n";
		KEYS() <<
		"};\n"
		"\n";

	out << 
		"static "; ARRAY_TYPE(maxSpan) << " "; FSM_NAME() << "_sp[] = {\n";
		KEY_SPANS() <<
		"};\n"
		"\n";

	out << 
		"static "; ARRAY_TYPE(maxFlatIndexOffset) << " "; FSM_NAME() << "_io[] = {\n";
		FLAT_INDEX_OFFSET() <<
		"};\n"
		"\n";

	out << 
		"static "; ARRAY_TYPE(maxIndex) << " "; FSM_NAME() << "_i[] = {\n";
		INDICIES() <<
		"};\n"
		"\n";

	out << 
		"static "; ARRAY_TYPE(maxState) << " "; FSM_NAME() << "_tt[] = {\n";
		TRANS_TARGS() <<
		"};\n"
		"\n";

	if ( anyActions() ) {
		out << 
			"static "; ARRAY_TYPE(maxActionLoc) << " "; FSM_NAME() << "_ta[] = {\n";
			TRANS_ACTIONS() <<
			"};\n"
			"\n";
	}

	if ( anyEofActions() ) {
		out <<
			"static "; ARRAY_TYPE(maxActionLoc) << " "; FSM_NAME() << "_ea[] = {\n";
			EOF_ACTIONS() <<
			"};\n"
			"\n";
	}

	out <<
		"static int "; FSM_NAME() << "_start = " 
				<< redFsm->startState->id << ";\n"
		"\n";

	return out;
}


/* Pass init data to base. */
CFlatCodeGen::CFlatCodeGen( char *fsmName, ParseData *parseData, 
		RedFsmAp *redFsm, std::ostream &out )
:
	FsmCodeGen(fsmName, parseData, redFsm, out)
{ }

void CFlatCodeGen::writeOutCode()
{
	/* Write the state and transition arrays. */
	STATE_DATA();

	/* Init routine. */
	out << 
		"int "; FSM_NAME() << "_init( struct "; FSM_NAME() << " *fsm )\n"
		"{\n"
		"	fsm->curs = "; FSM_NAME() << "_start;\n";
	
	/* If there are any calls, then the stack top needs initialization. */
	if ( anyActionCalls() || anyActionRets() )
		out << "	fsm->top = 0;\n";

	INIT_CODE() <<
		"	if ( fsm->curs >= "; FIRST_FINAL_STATE() << " )\n"
		"		return 1;\n"
		"	return 0;\n"
		"}\n"
		"\n";

	/* Execution function. */
	out << 
		"int "; FSM_NAME() << "_execute( struct "; FSM_NAME() << " *fsm, "; EL_TYPE() <<
				" *_data, int _len )\n"
		"{\n"
		"	"; EL_TYPE() << " *_p = _data-1;\n"
		"	"; EL_TYPE() << " *_pe = _data+_len;\n"
		"	int _slen, _cs = fsm->curs";

	if ( anyRegCurStateRef() )
		out << ", _ps";
	out << ";\n	int _trans;\n";
		
	if ( anyActions() ) {
		out << "\t";
		ARRAY_TYPE(maxActArrItem) << " *_acts, _nacts;\n";
	}

	out <<
		"	"; ALPH_TYPE() << " *_keys;\n"
		"	"; ARRAY_TYPE(maxIndex) << " *_inds;\n"
		"	fsm->curs = -1;\n"
		"\n";


	out <<
		"again:\n"
		"	if ( ++_p == _pe )\n"
		"		goto out;\n";

	if ( anyEofActions() ) {
		out <<
			"	if ( _p == 0 )\n"
			"		goto eofActions;\n"
			"\n";
	}
	
	LOCATE_TRANS();

	if ( anyRegCurStateRef() )
		out << "	_ps = _cs;\n";

	out <<
		"	_cs = "; FSM_NAME() << "_tt[_trans];\n"
		"\n";

	if ( anyActions() ) {
		out <<
			"	if ( "; FSM_NAME() << "_ta[_trans] == 0 )\n"
			"		goto again;\n"
			"	_acts = "; FSM_NAME() << "_a + "; FSM_NAME() << "_ta[_trans];\n"
			"\n";

		/* Only used if there are out actions. */
		if ( anyEofActions() )
			out << "execFuncs:\n";

		out <<
			"	_nacts = *_acts++;\n"
			"	while ( _nacts-- > 0 ) {\n"
			"		switch ( *_acts++ ) {\n";
			ACTION_SWITCH() <<
			"		}\n"
			"	}\n"
			"\n";
	}

	out <<
		"	goto again;\n"
		"\n";

	if ( anyEofActions() ) {
		out <<
			"eofActions:\n"
			"	if ( "; FSM_NAME() << "_ea[_cs] != 0 ) {\n"
			"		_acts = "; FSM_NAME() << "_a + "; FSM_NAME() << "_ea[_cs];\n"
			"		_p = 0; _pe = (("; EL_TYPE() << "*)0)+1;\n"
			"		goto execFuncs;\n"
			"	}\n"
			"\n";
	}

	out <<
		"out:\n"
		"	fsm->curs = _cs;\n";

	if ( redFsm->errState != 0 ) {
		out << 
			"	if ( _cs == "; ERROR_STATE() << " )\n"
			"		return -1;\n";
	}

	out << 
		"	if ( _cs >= "; FIRST_FINAL_STATE() << " )\n"
		"		return 1;\n"
		"	return 0;\n"
		"}\n"
		"\n";

	/* Finish routine. */
	out <<
		"int "; FSM_NAME() << "_finish( struct "; FSM_NAME() << " *fsm )\n"
		"{\n";

	if ( anyEofActions() ) {
		/* May need to execute code in the action loop. Must use special
		 * params to execute func. */
		out << "	return "; FSM_NAME() << "_execute( fsm, 0, 1 );\n";
	}
	else {
		if ( redFsm->errState != 0 ) {
			out << 
				"	if ( fsm->curs == "; ERROR_STATE() << " )\n"
				"		return -1;\n";
		}

		out <<
			"	if ( fsm->curs >= "; FIRST_FINAL_STATE() << " )\n"
			"		return 1;\n"
			"	return 0;\n";
	}

	out << 
		"}\n"
		"\n";

	for ( ContextMap::Iter ctx = parseData->contextMap; ctx.lte(); ctx++ ) {
		out << "unsigned char _"; FSM_NAME() << "_ctxdata_" << ctx->key << "[] = {\n\t";
		CONTEXT( ctx->value->id );
		out << "};\n\n";
	}
}


/* Init base class. */
CppFlatCodeGen::CppFlatCodeGen( char *fsmName, ParseData *parseData, 
		RedFsmAp *redFsm, std::ostream &out ) 
: 
	FsmCodeGen(fsmName, parseData, redFsm, out) 
{ }

void CppFlatCodeGen::writeOutCode()
{
	/* Write the state and transition arrays. */
	STATE_DATA();

	/* Init routine. */
	out << 
		"int "; FSM_NAME() << "::init( )\n"
		"{\n"
		"	this->curs = "; FSM_NAME() << "_start;\n";
	
	/* If there are any calls, then the stack top needs initialization. */
	if ( anyActionCalls() || anyActionRets() )
		out << "	this->top = 0;\n";

	INIT_CODE() <<
		"	if ( this->curs >= "; FIRST_FINAL_STATE() << " )\n"
		"		return 1;\n"
		"	return 0;\n"
		"}\n"
		"\n";

	/* Execution function. */
	out << 
		"int "; FSM_NAME() << "::execute( "; EL_TYPE() << " *_data, int _len )\n"
		"{\n"
		"	"; EL_TYPE() << " *_p = _data-1;\n"
		"	"; EL_TYPE() << " *_pe = _data+_len;\n"
		"	int _slen, _cs = this->curs";

	if ( anyRegCurStateRef() )
		out << ", _ps";

	out << ";\n	int _trans;\n";

	if ( anyActions() ) {
		out << "\t";
		ARRAY_TYPE(maxActArrItem) << " *_acts, _nacts;\n";
	}

	out <<
		"	"; ALPH_TYPE() << " *_keys;\n"
		"	"; ARRAY_TYPE(maxIndex) << " *_inds;\n"
		"	this->curs = -1;\n"
		"\n";
	

	out << 
		"again:\n"
		"	if ( ++_p == _pe )\n"
		"		goto out;\n";

	if ( anyEofActions() ) {
		out << 
			"	if ( _p == 0 )\n"
			"		goto eofActions;\n"
			"\n";
	}

	LOCATE_TRANS();

	if ( anyRegCurStateRef() )
		out << "	_ps = _cs;\n";

	out <<
		"	_cs = "; FSM_NAME() << "_tt[_trans];\n"
		"\n";

	if ( anyActions() ) {
		out <<
			"	if ( "; FSM_NAME() << "_ta[_trans] == 0 )\n"
			"		goto again;\n"
			"	_acts = "; FSM_NAME() << "_a + "; FSM_NAME() << "_ta[_trans];\n"
			"\n";

		/* Only used if there are out actions. */
		if ( anyEofActions() )
			out << "execFuncs:\n";

		out <<
			"	_nacts = *_acts++;\n"
			"	while ( _nacts-- > 0 ) {\n"
			"		switch ( *_acts++ ) {\n";
			ACTION_SWITCH() <<
			"		}\n"
			"	}\n"
			"\n";
	}

	out <<
		"	goto again;\n"
		"\n";
	
	if ( anyEofActions() ) {
		out << 
			"eofActions:\n"
			"	if ( "; FSM_NAME() << "_ea[_cs] != 0 ) {\n"
			"		_acts = "; FSM_NAME() << "_a + "; FSM_NAME() << "_ea[_cs];\n"
			"		_p = 0; _pe = (("; EL_TYPE() << "*)0)+1;\n"
			"		goto execFuncs;\n"
			"	}\n"
			"\n";
	}

	out <<
		"out:\n"
		"	this->curs = _cs;\n";

	if ( redFsm->errState != 0 ) {
		out << 
			"	if ( _cs == "; ERROR_STATE() << " )\n"
			"		return -1;\n";
	}

	out <<
		"	if ( _cs >= "; FIRST_FINAL_STATE() << " )\n"
		"		return 1;\n"
		"	return 0;\n"
		"}\n"
		"\n";

	/* Finish routine. */
	out <<
		"int "; FSM_NAME() << "::finish( )\n"
		"{\n";

	if ( anyEofActions() ) {
		/* May need to execute code in the action loop. Must use special
		 * params to execute func. */
		out << "	return execute( 0, 1 );\n";
	}
	else {
		if ( redFsm->errState != 0 ) {
			out << 
				"	if ( this->curs == "; ERROR_STATE() << " )\n"
				"		return -1;\n";
		}

		out <<
			"	if ( this->curs >= "; FIRST_FINAL_STATE() << " )\n"
			"		return 1;\n"
			"	return 0;\n";
	}

	out << 
		"}\n"
		"\n";

	for ( ContextMap::Iter ctx = parseData->contextMap; ctx.lte(); ctx++ ) {
		out << "unsigned char "; FSM_NAME() << "::_ctxdata_" << ctx->key << "[] = {\n\t";
		CONTEXT( ctx->value->id );
		out << "};\n\n";
	}
}


/* Init base class. */
ObjCFlatCodeGen::ObjCFlatCodeGen( char *fsmName, ParseData *parseData, 
		RedFsmAp *redFsm, std::ostream &out ) 
: 
	FsmCodeGen(fsmName, parseData, redFsm, out) 
{ }

void ObjCFlatCodeGen::writeOutCode()
{
	/* Write the state and transition arrays. */
	STATE_DATA();

	out <<
		"- (int)initFsm;\n"
		"{\n"
		"\tself->curs = "; FSM_NAME() << "_start;\n";

	/* If there are any calls, then the stack top needs initialization. */
	if ( anyActionCalls() || anyActionRets() )
		out << "\tself->top = 0;\n\n";

	INIT_CODE() <<
		"\treturn ( self->curs >= "; FIRST_FINAL_STATE() << " ) ? 1 : 0;\n"
		"}\n"
		"\n";

	/* Execution function. */
	out <<
		"- (int) executeWithData:("; EL_TYPE() << " *)_data len:(int)_len;\n"
		"{\n"
		"\t"; EL_TYPE() << " *_p = _data - 1;\n"
		"\t"; EL_TYPE() << " *_pe = _data + _len;\n"
		"\tint _slen, _cs = self->curs";

	if ( anyRegCurStateRef() )
		out << ", _ps";
	out << ";\n	int _trans;\n"; // ??? this is declared as unsigned int in other writeOutCode() implementations

	if ( anyActions() ) {
		out << "\t";
		ARRAY_TYPE(maxActArrItem) << " *_acts, _nacts;\n";
	}

	out <<
		"\t"; ALPH_TYPE() << " *_keys;\n"
		"\t"; ARRAY_TYPE(maxIndex) << " *_inds;\n"
		"\tself->curs = -1;\n"
		"\n";
	
	out << 
		"again:\n"
			"\tif ( ++_p == _pe )\n"
				"\t\tgoto out;\n";

	if ( anyEofActions() ) {
		out << 
			"\tif ( _p == 0 )\n"
				"\t\tgoto eofActions;\n"
			"\n";
	}

	LOCATE_TRANS();

	if ( anyRegCurStateRef() ) out << "\t_ps = _cs;\n";

	out <<
		"\t_cs = "; FSM_NAME() << "_tt[_trans];\n"
		"\n";

	if ( anyActions() ) {
		out <<
			"\tif ( "; FSM_NAME() << "_ta[_trans] == 0 )\n"
				"\t\tgoto again;\n"
			"\t_acts = "; FSM_NAME() << "_a + "; FSM_NAME() << "_ta[_trans];\n"
			"\n";

		/* Only used if there are out actions. */
		if ( anyEofActions() )
			out << "execFuncs:\n";

		out <<
			"\t_nacts = *_acts++;\n"
			"\twhile ( _nacts-- > 0 ) {\n"
				"\t\tswitch ( *_acts++ ) {\n";
				ACTION_SWITCH() <<
				"\t\t}\n"
			"\t}\n"
			"\n";
	}

	out <<
		"\tgoto again;\n"
		"\n";
	
	if ( anyEofActions() ) {
		out << 
			"eofActions:\n"
				"\tif ( "; FSM_NAME() << "_ea[_cs] != 0 ) {\n"
					"\t\t_acts = "; FSM_NAME() << "_a + "; FSM_NAME() << "_ea[_cs];\n"
					"\t\t_p = 0; _pe = (("; EL_TYPE() << "*)0) + 1;\n"
					"\t\tgoto execFuncs;\n"
				"\t}\n"
			"\n";
	}

	out <<
		"out:\n"
			"\tself->curs = _cs;\n";

	if ( redFsm->errState != 0 ) {
		out << 
			"\tif ( _cs == "; ERROR_STATE() << " ) return -1;\n";
	}


	out <<
			"\treturn ( _cs >= "; FIRST_FINAL_STATE() << " ) ? 1 : 0;\n"
		"}\n"
		"\n";

	out <<
		"- (int)finish;\n"
		"{\n";
	
	if ( anyEofActions() ) {
		/* May need to execute code in the action loop. Must use special
		 * params to execute func. */
		out <<
			"\treturn [self executeWithData:0 len:1];\n";
	}
	else {
		if ( redFsm->errState != 0 ) {
			out << 
			"\tif ( self->curs == "; ERROR_STATE() << " ) return -1;\n";
		}

		out <<
			"\treturn ( self->curs >= "; FIRST_FINAL_STATE() << " ) ? 1 : 0;\n";
	}

	out << 
		"}\n"
		"\n";

	for ( ContextMap::Iter ctx = parseData->contextMap; ctx.lte(); ctx++ ) {
		out << "unsigned char "; FSM_NAME() << "::_ctxdata_" << ctx->key << "[] = {\n\t";
		CONTEXT( ctx->value->id );
		out << "};\n\n";
	}
}

