/*
 *  Copyright 2001-2005 Adrian Thurston <thurston@cs.queensu.ca>
 */

/*  This file is part of Ragel.
 *
 *  Ragel is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 * 
 *  Ragel is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *  You should have received a copy of the GNU General Public License
 *  along with Ragel; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 */

%{

#include <iostream>
#include <stdlib.h>
#include <limits.h>
#include <errno.h>
#include "ragel.h"
#include "parsetree.h"
#include "rlparse.h"

using std::endl;

/* Bison 1.875 tries to prevent a warning, but results ends up producing a 
 * This is an ugly hack. But it works for this case. */
#ifdef BISON_V1875
#define __attribute__(arg)
#endif

InputData *id = 0;
int includeDepth = 0;

typedef AvlMap<char*, ParseData *, CmpStr> SectionMap;
typedef AvlMapEl<char*, ParseData *> SectionMapEl;
SectionMap sectionMap;

extern bool inlineWhitespace;

/* These come from the scanner and point back into the parser. We will borrow
 * them for error reporting. */
extern YYSTYPE *yylval;
extern YYLTYPE *yylloc;

/* The include stack pointer from the scanner. Used to determine if we are
 * currently processing an included file. */
extern int inc_stack_ptr;

/* Try to do a definition, common to assignment and instantiation. */
void tryMachineDef( const BISON_YYLTYPE &loc, char *name, 
		Join *join, bool isInstance );
void beginOutsideCode();
bool getParseData( const BISON_YYLTYPE &loc );
void endSection( const InputLoc &loc );
void doInclude( const InputLoc &loc, char *sectionName, char *inputFile );
int yylex( YYSTYPE *yylval, YYLTYPE *yylloc );

%}

%pure-parser

%union {
	/* General data types. */
	char c;
	char *data;
	int integer;
	Literal *literal;

	/* Tree nodes. */
	Term *term;
	FactorWithAug *factorWithAug;
	FactorWithRep *factorWithRep;
	FactorWithNeg *factorWithNeg;
	Factor *factor;
	Expression *expression;
	Join *join;
	LongestMatchList *longestMatchList;
	LongestMatchPart *longestMatchPart;

	/* Priorities and actions. */
	AugType augType;
	Action *action;
	PriorDesc *priorDesc;

	/* Regular expression items. */
	RegExpr *regExp;
	ReItem *reItem;
	ReOrBlock *reOrBlock;
	ReOrItem *reOrItem;

	/* Inline parse tree items. */
	InlineItem *ilitem;
	InlineList *illist;
}

/* Outside Code. */
%token OC_Struct
%token OC_Open
%token OC_Close
%token OC_Semi
%token OC_Interface
%token OC_Implementation
%token OC_Protocol
%token OC_End
%token <data> OC_FsmName

%token TK_Section

/* General tokens. */
%token <data> TK_UInt
%token <data> TK_Hex
%token <data> TK_Word
%token <data> TK_Literal
%token <data> TK_BaseClause
%token TK_DoubleDot
%token TK_Arrow
%token TK_StarStar
%token TK_ColonEquals
%token TK_NameSep
%token TK_BarStar
%token TK_RepOpOpen

/* Global error actions. */
%token TK_StartGblError
%token TK_AllGblError
%token TK_FinishGblError
%token TK_LeaveGblError
%token TK_StartFinishGblError
%token TK_FinishLeaveGblError

/* Local error actions. */
%token TK_StartLocalError
%token TK_AllLocalError
%token TK_FinishLocalError
%token TK_LeaveLocalError
%token TK_StartFinishLocalError
%token TK_FinishLeaveLocalError

/* Context embedding. */
%token TK_StartContext
%token TK_AllContext
%token TK_FinishContext
%token TK_LeaveContext
%token TK_StartFinishContext
%token TK_FinishLeaveContext

/* EOF Action embedding. */
%token TK_StartEOF
%token TK_AllEOF
%token TK_FinishEOF
%token TK_LeaveEOF
%token TK_StartFinishEOF
%token TK_FinishLeaveEOF

/* Regular expression tokens. */
%token RE_Slash
%token RE_SqOpen
%token RE_SqOpenNeg
%token RE_SqClose
%token RE_Dot
%token RE_Star
%token RE_Dash
%token <data> RE_Char

/* Tokens specific to inline code. */
%token <data> IL_WhiteSpace
%token <data> IL_Comment
%token <data> IL_Literal
%token <data> IL_Symbol

/* Keywords. */
%token KW_Action
%token KW_Init
%token KW_AlphType
%token KW_Range
%token KW_Element
%token KW_GetKey
%token KW_Context
%token KW_Interface
%token KW_Include

/* Specials in code blocks. */
%token KW_Hold
%token KW_PChar
%token KW_Char
%token KW_Goto
%token KW_Call
%token KW_Ret
%token KW_CurState
%token KW_TargState
%token KW_Entry
%token KW_Next
%token KW_Exec
%token KW_Buf
%token KW_BLen

/* Special token for terminating semi-terminated code blocks. Needed because
 * semi is sent as a token in the code block rather than as a generic symbol. */
%token TK_Semi

/* Symbols. In ragel lexical space, the scanner does not pass 
 * any data along with the symbols, in inline code lexical 
 * space it does. */
%token '*' '?' '+' '!' '^' '(' ')' ';' ',' '=' 
%token ':' '@' '%' '$' '-' '|' '&' '.' '>'

/* Precedence information. Lower is a higher precedence. We need only two
 * precedence groups. Shifting the minus sign in front of a literal number
 * conflicts with the reduction of Expression and the subsequent shifting of a
 * subtraction operator when a '-' is seen. Since we want subtraction to take
 * precedence, we give EXPR_MINUS the higher priority. */
%nonassoc '-'
%nonassoc EXPR_MINUS

%type <augType> AugTypeBase
%type <augType> AugTypeError
%type <augType> AugTypeLocalError
%type <augType> AugTypeContext
%type <augType> AugTypeEOF
%type <integer> PriorityAug
%type <data> PriorityAugNum
%type <action> ActionEmbed
%type <action> OptInlineAction
%type <longestMatchList> LongestMatchList
%type <longestMatchPart> LongestMatchPart
%type <join> Join
%type <expression> Expression
%type <term> Term
%type <factorWithAug> FactorWithLabel
%type <factorWithAug> FactorWithEp
%type <factorWithAug> FactorWithAug
%type <factorWithRep> FactorWithRep
%type <integer> FactorRepNum
%type <factorWithNeg> FactorWithNeg
%type <factor> Factor
%type <literal> RangeLit
%type <data> AlphabetNum
%type <data> MachineName
%type <integer> PriorityName
%type <integer> LocalErrName
%type <integer> ContextName
%type <data> SectionName
%type <data> OptSection
%type <data> OptFileName

%type <illist> InlineBlock
%type <ilitem> InlineBlockItem
%type <ilitem> InlineBlockInterpret
%type <data> InlineBlockAny
%type <data> InlineBlockSymbol

%type <illist> InlineExpr
%type <ilitem> InlineExprItem
%type <ilitem> InlineExprInterpret
%type <illist> InlineExprCommaSepExpr
%type <data> InlineExprSymbol
%type <data> InlineExprAny

%type <regExp> RegularExpr
%type <reItem> RegularExprItem
%type <reItem> RegularExprChar
%type <reOrBlock> RegularExprOrData
%type <reOrItem> RegularExprOrChar

%%

/* Input is any number of input sections. An empty file is accepted. */
input: FsmSpecList;
FsmSpecList: 		
	FsmSpecList FsmSpec |
	/* Nothing */;

FsmSpec: 
	OC_Struct OptStructName OC_Open FsmSpecList OC_Close OC_Semi {
		/* The name is pushed in OptStructName, now done so pop it. */
		id->structStack.remove( -1 );
	} |
	OC_Struct OptStructName OC_Semi {
		/* The name is pushed in OptStructName, now done so pop it. */
		id->structStack.remove( -1 );
	} |
	OC_Interface ObjCSectName FsmSpecList OC_End {
		/* The name is pushed in InterfaceName, now done so pop it. */
		id->structStack.remove( -1 );
	} |
	OC_Implementation ObjCSectName FsmSpecList OC_End {
		/* The name is pushed in ImplementationName, now done so pop it. */
		id->structStack.remove( -1 );
	} |
	OC_Protocol ObjCSectName FsmSpecList OC_End {
		/* The name is pushed in ImplementationName, now done so pop it. */
		id->structStack.remove( -1 );
	} |
	OC_Open FsmSpecList OC_Close |
	OC_Semi;

OptStructName: OC_FsmName { id->structStack.append( $1 ); };
OptStructName: { id->structStack.append( 0 ); };
ObjCSectName: OC_FsmName { id->structStack.append( $1 ); };

/* Fsm Specification. Fsms are begin with '%%' and may be a {} delimited
 * list of Fsm statements or may be a single statement. If no name is
 * given the last name given in a machine is used. */
FsmSpec: 
	StartSection SectionName StmtOrStmtList {
		if ( id->active ) 
			endSection( id->sectionEndLoc );
	} |
	StartSection StmtOrStmtList {
		if ( id->active )
			endSection( id->sectionEndLoc );
	};

StartSection: 
	TK_Section { 
		/* Clear the last section name. */
		id->active = true;
		id->sectionName = 0;
		id->lookedUpSection = false;
		id->sawInterface = false;
		id->sawMain = false;
		id->sectionLoc = InputLoc(@1);
	};

SectionName: 
	TK_Word {
		/* Used by the parse data structure lookup. */
		id->sectionName = $1;
	};

StmtOrStmtList: 
	'{' StatementList '}' {
		/* Put the lexer back in the outside code state. */
		beginOutsideCode();
		id->sectionEndLoc = @3;
	} |
	Statement { 
		beginOutsideCode();
	};

/* A NonEmpty list of statements in a fsm. */
StatementList:
	StatementList Statement |
	/* Nothing */;

/* The differnt types of statements in a fsm spec. */
Statement:	
	Assignment |
	Instantiation |
	ActionSpec |
	InitSpec |
	AlphSpec |
	ElementSpec |
	GetKeySpec |
	Context |
	RangeSpec |
	Interface |
	Include;

/* Garble up to the next ; */
Statement: error ';' { yyerrok; };

/* Allow the user to give some code to insert into the init function. */
InitSpec:
	KW_Init '{' InlineBlock '}' {
		if ( getParseData( @1 ) ) {
			/* Put the code on the init code list. */
			id->pd->initCodeList.append( new InlineBlock( InputLoc(@2), $3, NameRefList() ) );
		}
		id->sectionEndLoc = @4;
	};

/* Allow the user to create a named fsm action that can be referenced when
 * building a machine. */
ActionSpec:	
	KW_Action TK_Word '{' InlineBlock '}' {
		if ( getParseData( @1 ) ) {
			if ( id->pd->actionDict.find( $2 ) ) {
				/* Recover by just ignoring the duplicate. */
				error(@2) << "action \"" << $2 << "\" already defined" << endl;
			}
			else {
				/* Add the action to the list of actions. */
				Action *newAction = new Action( InputLoc(@3), $2, $4, id->nameRefList );

				/* Insert to list and dict. */
				id->pd->actionList.append( newAction );
				id->pd->actionDict.insert( newAction );
			}
		}
		id->sectionEndLoc = @5;
	};

/* Specifies the data type of the input alphabet. One or two words 
 * followed by a semi-colon. */
AlphSpec:
	KW_AlphType TK_Word TK_Word TK_Semi {
		if ( getParseData( @1 ) ) {
			if ( ! id->pd->setAlphType( $2, $3 ) ) {
				// Recover by ignoring the alphtype statement.
				error(@2) << "\"" << $2 << 
					" " << $3 << "\" is not an allowed alphabet type" << endl;
			}
		}
		id->sectionEndLoc = @4;
	} |
	KW_AlphType TK_Word TK_Semi {
		if ( getParseData( @1 ) ) {
			if ( ! id->pd->setAlphType( $2 ) ) {
				// Recover by ignoring the alphtype statement.
				error(@2) << "\"" << $2 << "\" is not an allowed alphabet type" << endl;
			}
		}
		id->sectionEndLoc = @3;
	};

ElementSpec:
	KW_Element InlineBlock TK_Semi {
		if ( getParseData( @1 ) )
			id->pd->elementType = $2;
		id->sectionEndLoc = @3;
	};

GetKeySpec:
	KW_GetKey InlineBlock TK_Semi {
		if ( getParseData( @1 ) )
			id->pd->getKeyExpr = $2;
		id->sectionEndLoc = @3;
	};

/* Declaration of context. */
Context:
	KW_Context TK_Word ';' {
		if ( getParseData( @1 ) ) {
			/* Get a context on word. */
			ContextMapEl *inMap = 0;
			if ( id->pd->contextMap.insert( $2, &inMap ) )
				inMap->value = new Context( $2, id->pd->nextContextId++ );
			inMap->value->declared = false;
			id->pd->contextList.append( inMap->value );
		}
		id->sectionEndLoc = @3;
	};

/* Specifies a range to assume that the input characters will fall into. */
RangeSpec:
	KW_Range AlphabetNum AlphabetNum ';' {
		if ( getParseData( @1 ) ) {
			// Save the upper and lower ends of the range and emit the line number.
			id->pd->lowerNum = $2;
			id->pd->upperNum = $3;
			id->pd->rangeLowLoc = InputLoc(@2);
			id->pd->rangeHighLoc = InputLoc(@3);
		}
		id->sectionEndLoc = @4;
	};

Interface:
	KW_Interface ';' {
		if ( getParseData( @1 ) ) {
			id->sawInterface = true;
		}
		id->sectionEndLoc = @2;
	}

/* Include statements are processed by both the scanner and the parser. */
Include:
	IncludeKeyword OptSection OptFileName ';' {
		if ( id->active )
			doInclude( @1, $2, $3 );
	}

IncludeKeyword: 
	KW_Include {
		/* Do this immediately so that the scanner has a correct sense of the
		 * value in id->active when it reaches the end of the statement before
		 * the above action executes. */
		getParseData( @1 );
	};

OptSection: TK_Word { $$ = $1; } | { $$ = 0; };
OptFileName: TK_Literal { $$ = $1; } | { $$ = 0; };

/* An assignement statement. Assigns the definition of a machine to a variable name. */
Assignment:
	MachineName '=' Join ';' {
		if ( id->active ) {
			/* Main machine must be an instance. */
			bool isInstance = false;
			if ( strcmp($1, machineMain) == 0 ) {
				warning(@1) << "main machine will be implicitly instantiated" << endl;
				isInstance = true;
			}

			/* Generic creation of machine for instantiation and assignment. */
			tryMachineDef( @1, $1, $3, isInstance );

			/* Main machine will trigger code generation. */
			if ( strcmp($1, machineMain) == 0 )
				id->sawMain = true;
		}
		id->sectionEndLoc = @4;
	};

/* An instantiation statement. Instantiates a machine and assigns it to a
 * variable name. */
Instantiation:
	MachineName TK_ColonEquals Join ';' {
		/* Generic creation of machine for instantiation and assignment. */
		if ( id->active ) {
			tryMachineDef( @1, $1, $3, true );

			/* Main machine will trigger code generation. */
			if ( strcmp($1, machineMain) == 0 )
				id->sawMain = true;
		}
		id->sectionEndLoc = @4;
	};

/* Capture the machine name for making the machine's priority name. */
MachineName:
	TK_Word {
		if ( getParseData( @1 ) ) {
			/* Make/get the priority key. The name may have already been referenced
			 * and therefore exist. */
			PriorDictEl *priorDictEl;
			if ( id->pd->priorDict.insert( $1, id->pd->nextPriorKey, &priorDictEl ) )
				id->pd->nextPriorKey += 1;
			id->pd->curDefPriorKey = priorDictEl->value;

			/* Make/get the local error key. */
			LocalErrDictEl *localErrDictEl;
			if ( id->pd->localErrDict.insert( $1, id->pd->nextLocalErrKey, &localErrDictEl ) )
				id->pd->nextLocalErrKey += 1;
			id->pd->curDefLocalErrKey = localErrDictEl->value;
		}
	};

Join: 
	Join ',' Expression {
		/* Append the expression to the list and return it. */
		$1->exprList.append( $3 );
		$$ = $1;
	} |
	Expression {
		/* Create the expression list with the intial expression. */
		$$ = new Join( InputLoc(@1), $1 );
	};

/* Top level production in the parse of a fsm. The lowest precedence
 * is the '|' (or), '&' (intersection), and '-' (subtraction) operators. */
Expression:
	Expression '|' Term {
		$$ = new Expression( $1, $3, Expression::OrType );
	} %prec EXPR_MINUS |
	Expression '&' Term {
		$$ = new Expression( $1, $3, Expression::IntersectType );
	} %prec EXPR_MINUS |
	Expression '-' Term {
		$$ = new Expression( $1, $3, Expression::SubtractType );
	} %prec EXPR_MINUS |
	Term {
		$$ = new Expression( $1 );
	} %prec EXPR_MINUS;

Term:
	Term FactorWithLabel {
		$$ = new Term( $1, $2 );
	} |
	Term '.' FactorWithLabel {
		$$ = new Term( $1, $3 );
	} |
	FactorWithLabel {
		$$ = new Term( $1 );
	};

FactorWithLabel:
	TK_Word ':' FactorWithLabel { 
		/* Add the label to the list and pass the factor up. */
		$3->labels.prepend( Label(@1, $1) );
		$$ = $3; 
	} |
	FactorWithEp;

FactorWithEp:
	FactorWithEp TK_Arrow LocalStateRef { 
		/* Add the target to the list and return the factor object. */
		$1->epsilonLinks.append( EpsilonLink( InputLoc(@2), id->nameRef ) );
		$$ = $1; 
	} |
	FactorWithAug;

/* A local state reference. Qualified name witout :: prefix. */
LocalStateRef:
	NoNameSep StateRefNames;

/* Clear the name ref structure. */
NoNameSep:
	/* Nothing */ {
		id->nameRef.empty();
	}

/* A qualified state reference. */
StateRef:
	OptNameSep StateRefNames;

/* Optional leading name separator. */
OptNameSep:
	TK_NameSep {
		/* Insert an inition null pointer val to indicate the existence of the
		 * initial name seperator. */
		id->nameRef.setAs( 0 );
	} |
	/* Nothing. */ {
		id->nameRef.empty();
	}

/* List of names separated by :: */
StateRefNames:
	StateRefNames TK_NameSep TK_Word {
		id->nameRef.append( $3 );
	} |
	TK_Word {
		id->nameRef.append( $1 );
	};

/* Third group up in precedence. Allow users to embed actions and to change the
 * priorities of transitions. */
FactorWithAug:
	FactorWithAug AugTypeBase ActionEmbed {
		/* Append the action to the factorWithAug, record the refernce from 
		 * factorWithAug to the action and pass up the factorWithAug. */
		$1->actions.append( ParserAction( $2, 0, $3 ) );
		$$ = $1;
	} |
	FactorWithAug AugTypeError ActionEmbed {
		if ( id->active ) {
			/* Append the action to the factorWithAug, record the refernce from 
			 * factorWithAug to the action and pass up the factorWithAug. */
			$1->actions.append( ParserAction( $2, id->pd->curDefLocalErrKey, $3 ) );
		}
		$$ = $1;
	} |
	FactorWithAug AugTypeLocalError '(' LocalErrName ',' ActionEmbed ')' {
		/* Append the action to the factorWithAug, record the refernce from
		 * factorWithAug to the action and pass up the factorWithAug. */
		$1->actions.append( ParserAction( $2, $4, $6 ) );
		$$ = $1;
	} |
	FactorWithAug AugTypeBase '(' PriorityName ',' PriorityAug ')' {
		/* Append the priority using a default name. */
		$1->priorityAugs.append( PriorityAug( $2, $4, $6 ) );
		$$ = $1;
	} |
	FactorWithAug AugTypeBase PriorityAug {
		if ( id->active ) {
			/* Append the named priority to the factorWithAug and pass it up. */
			$1->priorityAugs.append( PriorityAug( $2, id->pd->curDefPriorKey, $3 ) );
		}
		$$ = $1;
	} |
	FactorWithAug AugTypeContext ContextName {
		/* Append the context naming to the factorWithAug and pass it up. */
		$1->contexts.append( ContextEmbed( $2, $3 ) );
		$$ = $1;
	} |
	FactorWithAug AugTypeEOF ActionEmbed {
		/* Apend the eof action to the factorWithAg and pass it up. There is no
		 * local error key. */
		$1->actions.append( ParserAction( $2, 0, $3 ) );
		$$ = $1;
	} |
	FactorWithRep {
		$$ = new FactorWithAug( $1 );
	};

/* A specified priority name. Looks up the name in the current priority
 * dictionary. */
PriorityName:
	TK_Word {
		if ( id->active ) {
			// Lookup/create the priority key.
			PriorDictEl *priorDictEl;
			if ( id->pd->priorDict.insert( $1, id->pd->nextPriorKey, &priorDictEl ) )
				id->pd->nextPriorKey += 1;

			// Use the inserted/found priority key.
			$$ = priorDictEl->value;
		}
	};

LocalErrName:
	TK_Word {
		if ( id->active ) {
			/* Lookup/create the priority key. */
			LocalErrDictEl *localErrDictEl;
			if ( id->pd->localErrDict.insert( $1, id->pd->nextLocalErrKey, &localErrDictEl ) )
				id->pd->nextLocalErrKey += 1;

			/* Use the inserted/found priority key. */
			$$ = localErrDictEl->value;
		}
	};

ContextName:
	TK_Word {
		if ( id->active ) {
			/* Get a context on word. */
			ContextMapEl *inMap = 0;
			if ( id->pd->contextMap.insert( $1, &inMap ) )
				inMap->value = new Context( $1, id->pd->nextContextId++ );
			$$ = inMap->value->id;
		}
	};

/* Priority change specs. */
PriorityAug: 
	PriorityAugNum {
		// Convert the priority number to a long. Check for overflow.
		errno = 0;
		int aug = strtol( $1, 0, 10 );
		if ( errno == ERANGE && aug == LONG_MAX ) {
			// Priority number too large. Recover by setting the priority to 0.
			error(@1) << "priority number " << $1 << " overflows" << endl;
			$$ = 0;
		}
		else if ( errno == ERANGE && aug == LONG_MIN ) {
			// Priority number too large in the neg. Recover by using 0.
			error(@1) << "priority number " << $1 << " underflows" << endl;
			$$ = 0;
		}
		else {
			// No overflow or underflow.
			$$ = aug;
 		}
	};

PriorityAugNum:
	TK_UInt |
	'+' TK_UInt {
		$$ = $2;
	} |
	'-' TK_UInt {
		$$ = (char*)malloc( strlen($2)+2 ); 
		$$[0] = '-'; 
		strcpy($$+1, $2);
	};

/* Differnt places to change priorities. */
AugTypeBase:
	'@' { $$ = at_finish; } |
	'%' { $$ = at_leave; } |
	'$' { $$ = at_all; } |
	'>' { $$ = at_start; };
		
/* Global error actions. */
AugTypeError:
	AugTypeLocalError { $$ = $1; } |
	TK_StartGblError { $$ = at_start_gbl_error; } |
	TK_AllGblError { $$ = at_all_gbl_error; } |
	TK_FinishGblError { $$ = at_finish_gbl_error; } |
	TK_LeaveGblError { $$ = at_leave_gbl_error; } |
	TK_StartFinishGblError { $$ = at_start_finish_gbl_error; } |
	TK_FinishLeaveGblError { $$ = at_finish_leave_gbl_error; };

/* Local error actions. */
AugTypeLocalError:
	TK_StartLocalError { $$ = at_start_local_error; } |
	TK_AllLocalError { $$ = at_all_local_error; } |
	TK_FinishLocalError { $$ = at_finish_local_error; } |
	TK_LeaveLocalError { $$ = at_leave_local_error; } |
	TK_StartFinishLocalError { $$ = at_start_finish_local_error; } |
	TK_FinishLeaveLocalError { $$ = at_finish_leave_local_error; };

/* Context Embedding. */
AugTypeContext:
	TK_StartContext { $$ = at_start_context; } |
	TK_AllContext { $$ = at_all_context; } |
	TK_FinishContext { $$ = at_finish_context; } |
	TK_LeaveContext { $$ = at_leave_context; } |
	TK_StartFinishContext { $$ = at_start_finish_context; } |
	TK_FinishLeaveContext { $$ = at_finish_leave_context; };

/* EOF actions. */
AugTypeEOF:
	TK_StartEOF { $$ = at_start_eof; } |
	TK_AllEOF { $$ = at_all_eof; } |
	TK_FinishEOF { $$ = at_finish_eof; } |
	TK_LeaveEOF { $$ = at_leave_eof; } |
	TK_StartFinishEOF { $$ = at_start_finish_eof; } |
	TK_FinishLeaveEOF { $$ = at_finish_leave_eof; };


/* Different ways to embed actions. A TK_Word is reference to an action given by
 * the user as a statement in the fsm specification. An action can also be
 * specified immediately. */
ActionEmbed:
	TK_Word {
		if ( id->active ) {
			/* Set the name in the actionDict. */
			Action *action = id->pd->actionDict.find( $1 );
			if ( action != 0 ) {
				/* Pass up the action element */
				$$ = action;
			}
			else {
				/* Will recover by returning null as the action. */
				error(@1) << "action lookup of \"" << $1 << "\" failed" << endl;
				$$ = 0;
			}
		}
	} |
	'{' InlineBlock '}' {
		if ( id->active ) {
			/* Create the action, add it to the list and pass up. */
			Action *newAction = new Action( InputLoc(@1), 0, $2, id->nameRefList );
			id->pd->actionList.append( newAction );
			$$ = newAction;
		}
	};

/* The fourth level of precedence. These are the trailing unary operators that
 * allow for repetition. */
FactorWithRep:
	FactorWithRep '*' {
		$$ = new FactorWithRep( InputLoc(@2), $1, 0, 0,
				FactorWithRep::StarType );
	} |
	FactorWithRep TK_StarStar {
		$$ = new FactorWithRep( InputLoc(@2), $1, 0, 0,
				FactorWithRep::StarStarType );
	} |
	FactorWithRep '?' {
		$$ = new FactorWithRep( InputLoc(@2), $1, 0, 0,
				FactorWithRep::OptionalType );
	} |
	FactorWithRep '+' {
		$$ = new FactorWithRep( InputLoc(@2), $1, 0, 0,
				FactorWithRep::PlusType );
	} |
	FactorWithRep TK_RepOpOpen FactorRepNum '}' {
		$$ = new FactorWithRep( InputLoc(@2), $1, $3, 0,
				FactorWithRep::ExactType );
	} |
	FactorWithRep TK_RepOpOpen ',' FactorRepNum '}' {
		$$ = new FactorWithRep( InputLoc(@2), $1, 0, $4,
				FactorWithRep::MaxType );
	} |
	FactorWithRep TK_RepOpOpen FactorRepNum ',' '}' {
		$$ = new FactorWithRep( InputLoc(@2), $1, $3, 0,
				FactorWithRep::MinType );
	} |
	FactorWithRep TK_RepOpOpen FactorRepNum ',' FactorRepNum '}' {
		$$ = new FactorWithRep( InputLoc(@2), $1, $3, $5,
				FactorWithRep::RangeType );
	} |
	FactorWithNeg {
		$$ = new FactorWithRep( InputLoc(@1), $1 );
	};

FactorRepNum:
	TK_UInt {
		// Convert the priority number to a long. Check for overflow.
		errno = 0;
		int rep = strtol( $1, 0, 10 );
		if ( errno == ERANGE && rep == LONG_MAX ) {
			// Repetition too large. Recover by returing repetition 1. */
			error(@1) << "repetition number " << $1 << " overflows" << endl;
			$$ = 1;
		}
		else {
			// Cannot be negative, so no overflow.
			$$ = rep;
 		}
	};

/* The fifth level up in precedence. Negation. */
FactorWithNeg:
	'!' FactorWithNeg {
		$$ = new FactorWithNeg( InputLoc(@1), $2, FactorWithNeg::NegateType );
	} |
	'^' FactorWithNeg {
		$$ = new FactorWithNeg( InputLoc(@1), $2, FactorWithNeg::CharNegateType );
	} |
	Factor {
		$$ = new FactorWithNeg( InputLoc(@1), $1 );
	};

/* The highest level in precedence. Atomic machines such as references to other
 * machines, literal machines, regular expressions or Expressions in side of
 * parenthesis. */
Factor:
	TK_Literal {
		// Create a new factor node going to a concat literal. */
		$$ = new Factor( new Literal( InputLoc(@1), $1, Literal::LitString ) );
	} |
	AlphabetNum {
		// Create a new factor node going to a literal number. */
		$$ = new Factor( new Literal( InputLoc(@1), $1, Literal::Number ) );
	} |
	TK_Word {
		if ( id->active ) {
			// Find the named graph.
			GraphDictEl *gdNode = id->pd->graphDict.find( $1 );
			if ( gdNode == 0 ) {
				// Recover by returning null as the factor node.
				error(@1) << "graph lookup of \"" << $1 << "\" failed" << endl;
				$$ = 0;
			}
			else if ( gdNode->isInstance ) {
				// Recover by retuning null as the factor node.
				error(@1) << "references to graph instantiations not allowed "
						"in expressions" << endl;
				$$ = 0;
			}
			else {
				// Create a factor node that is a lookup of an expression.
				$$ = new Factor( InputLoc(@1), gdNode->value );
			}
		}
	} |
	RE_SqOpen RegularExprOrData RE_SqClose {
		// Create a new factor node going to an OR expression. */
		$$ = new Factor( new ReItem( InputLoc(@1), $2, ReItem::OrBlock ) );
	} |
	RE_SqOpenNeg RegularExprOrData RE_SqClose {
		// Create a new factor node going to a negated OR expression. */
		$$ = new Factor( new ReItem( InputLoc(@1), $2, ReItem::NegOrBlock ) );
	} |
	RE_Slash RegularExpr RE_Slash {
		// Create a new factor node going to a regular exp.
		$$ = new Factor( $2 );
	} |
	RangeLit TK_DoubleDot RangeLit {
		// Create a new factor node going to a range. */
		$$ = new Factor( new Range( $1, $3 ) );
	} |
	'(' Join ')' {
		/* Create a new factor going to a parenthesized join. */
		$$ = new Factor( $2 );
	} |
	TK_BarStar LongestMatchList '*' '|' {
		/* Create a new factor going to a longest match structure. */
		$$ = new Factor( new LongestMatch( @1, $2 ) );
	};

/* Garble up to the closing brace of a parenthesized expression. */
Factor: '(' error ')' { $$ = 0; yyerrok; };

LongestMatchList:
	LongestMatchList LongestMatchPart {
		$1->append( $2 );
		$$ = $1;
	} |
	LongestMatchPart {
		/* Create a new list with the part. */
		$$ = new LongestMatchList;
		$$->append( $1 );
	};

LongestMatchPart: 
	Join OptInlineAction ';' {
		if ( id->active ) {
			$$ = new LongestMatchPart( $1, $2, id->pd->nextLongestMatchId++ );
		}
	};

OptInlineAction:
	'{' InlineBlock '}' {
		if ( id->active ) {
			/* Create the action, add it to the list and pass up. */
			Action *newAction = new Action( InputLoc(@1), 0, $2, id->nameRefList );
			id->pd->actionList.append( newAction );
			$$ = newAction;
		}
	} |
	/* Nothing */ {
		$$ = 0;
	};


/* Any form of a number that can be used as a basic machine. */
AlphabetNum:
	TK_UInt |
	'-' TK_UInt { 
		$$ = (char*)malloc( strlen($2)+2 ); 
		$$[0] = '-'; 
		strcpy($$+1, $2);
	} | 
	TK_Hex ;

InlineBlock:
	InlineBlock InlineBlockItem {
		/* Append the item to the list, return the list. */
		$1->append( $2 );
		$$ = $1;
	} |
	/* Empty */ {
		/* Start with empty list. */
		$$ = new InlineList;
	};

/* Items in a struct block. */
InlineBlockItem:
	InlineBlockAny {
		/* Add a text segment. */
		$$ = new InlineItem( @1, $1, InlineItem::Text );
	} |
	InlineBlockSymbol {
		/* Add a text segment, need string on heap. */
		$$ = new InlineItem( @1, strdup($1), InlineItem::Text );
	} |
	InlineBlockInterpret {
		/* Pass the inline item up. */
		$$ = $1;
	};

/* Uninteresting tokens in a struct block. Data allocated by scanner. */
InlineBlockAny:
	IL_WhiteSpace | IL_Comment | IL_Literal | IL_Symbol |
	TK_UInt | TK_Hex | TK_Word;

/* Symbols in a struct block, no data allocated. */
InlineBlockSymbol:
	'[' { $$ = "["; } | ']' { $$ = "]"; } | 
	',' { $$ = ","; } | ';' { $$ = ";"; } |
	'(' { $$ = "("; } | ')' { $$ = ")"; } |
	'*' { $$ = "*"; } | TK_NameSep { $$ = "::"; };

/* Interpreted statements in a struct block. */
InlineBlockInterpret:
	InlineExprInterpret {
		/* Pass up interpreted items of inline expressions. */
		$$ = $1;
	} |
	KW_Hold SetNoWs ';' SetWs {
		$$ = new InlineItem( @1, InlineItem::Hold );
	} |
	KW_Goto SetNoWs StateRef ';' SetWs { 
		$$ = new InlineItem( @1, new NameRef(id->nameRef), InlineItem::Goto );
	} | 
	KW_Goto SetNoWs '*' SetWs InlineExpr ';' {
		$$ = new InlineItem( @1, InlineItem::GotoE );
		$$->children = $5;
	} |
	KW_Next SetNoWs StateRef ';' SetWs { 
		$$ = new InlineItem( @1, new NameRef(id->nameRef), InlineItem::Next );
	} |
	KW_Next SetNoWs '*' SetWs InlineExpr ';' {
		$$ = new InlineItem( @1, InlineItem::NextE );
		$$->children = $5;
	} |
	KW_Call SetNoWs StateRef ';' SetWs {
		$$ = new InlineItem( @1, new NameRef(id->nameRef), InlineItem::Call );
	} | 
	KW_Call SetNoWs '*' SetWs InlineExpr ';' {
		$$ = new InlineItem( @1, InlineItem::CallE );
		$$->children = $5;
	} |
	KW_Ret SetNoWs ';' SetWs {
		$$ = new InlineItem( @1, InlineItem::Ret );
	} |
	KW_Exec SetNoWs '(' InlineExpr ',' InlineExpr ')' ';' SetWs {
		$$ = new InlineItem( @1, InlineItem::Exec );

		/* Make a child node for each of the expressions. */
		InlineItem *left = new InlineItem( InputLoc(), InlineItem::Node );
		left->children = $4;
		InlineItem *right = new InlineItem( InputLoc(), InlineItem::Node );
		right->children = $6;

		/* Append the child lists. */
		$$->children = new InlineList;
		$$->children->append( left );
		$$->children->append( right );
	};

/* Turn off whitspace collecting when scanning inline blocks. */
SetNoWs: { inlineWhitespace = false; };

/* Turn on whitespace collecting when scanning inline blocks. */
SetWs: { inlineWhitespace = true; };

InlineExpr:
	InlineExpr InlineExprItem {
		$1->append( $2 );
		$$ = $1;
	} |
	/* Empty */ {
		/* Init the list used for this expr. */
		$$ = new InlineList;
	};

InlineExprItem:
	InlineExprAny {
		/* Return a text segment. */
		$$ = new InlineItem( @1, $1, InlineItem::Text );
	} |
	InlineExprSymbol {
		/* Return a text segment, must heap alloc the text. */
		$$ = new InlineItem( @1, strdup($1), InlineItem::Text );
	} |
	InlineExprInterpret {
		/* Pass the inline item up. */
		$$ = $1;
	} |
	'(' InlineExprCommaSepExpr ')' {
		/* Return a list of items. */
		$$ = new InlineItem( InputLoc(), InlineItem::Node );
		$$->children = new InlineList;
		$$->children->append( new InlineItem( @1, strdup("("), InlineItem::Text ) );
		$$->children->append( *($2) );
		$$->children->append( new InlineItem( @3, strdup(")"), InlineItem::Text ) );
		delete $2;
	};

InlineExprInterpret:
	KW_PChar {
		$$ = new InlineItem( @1, InlineItem::PChar );
	} |
	KW_Char {
		$$ = new InlineItem( @1, InlineItem::Char );
	} |
	KW_CurState {
		$$ = new InlineItem( @1, InlineItem::Curs );
	} |
	KW_TargState {
		$$ = new InlineItem( @1, InlineItem::Targs );
	} |
	KW_Buf {
		$$ = new InlineItem( @1, InlineItem::Buf );
	} |
	KW_BLen {
		$$ = new InlineItem( @1, InlineItem::BufLen );
	} |
	KW_Entry SetNoWs '(' StateRef ')' SetWs {
		$$ = new InlineItem( @1, new NameRef(id->nameRef), InlineItem::Entry );
	};

/* Comma separated list fo inline expressions. Passes an InlineList up. */
InlineExprCommaSepExpr:
	InlineExprCommaSepExpr ',' InlineExpr {
		$1->append( new InlineItem( @2, strdup(","), InlineItem::Text ) );
		$1->append( *($3) );
		$$ = $1;
		delete $3;
	} |
	InlineExpr {
		/* Pass the expression list up. */
		$$ = $1;
	};

InlineExprAny:
	IL_WhiteSpace | IL_Comment | IL_Literal | IL_Symbol |
	TK_UInt | TK_Hex | TK_Word;

/* Anything in a ExecValExpr that is not dynamically allocated. This includes
 * all special symbols caught in inline code except the semi. */
InlineExprSymbol:
	'[' { $$ = "["; } | ']' { $$ = "]"; } | 
	'*' { $$ = "*"; } | TK_NameSep { $$ = "::"; };


/* Parser for regular expression fsms. Any number of expression items which
 * generally gives a machine one character long or one character long stared. */
RegularExpr:
	RegularExpr RegularExprItem {
		// An optimization to lessen the tree size. If a non-starred char is directly
		// under the left side on the right and the right side is another non-starred
		// char then paste them together and return the left side. Otherwise
		// just put the two under a new reg exp node.
		if ( $2->type == ReItem::Data && !$2->star &&
			$1->type == RegExpr::RecurseItem &&
			$1->item->type == ReItem::Data && !$1->item->star )
		{
			// Append the right side to the right side of the left and toss 
			// the right side.
			$1->item->str += $2->str;
			delete $2;
			$$ = $1;
		}
		else {
			$$ = new RegExpr( $1, $2 );
		}
	} |
	/* Nothing */ {
		// Can't optimize the tree.
		$$ = new RegExpr();
	};

/* RegularExprItems can be a character spec with an optional staring of the char. */
RegularExprItem:
	RegularExprChar RE_Star {
		$1->star = true;
		$$ = $1;
	} |
	RegularExprChar {
		$$ = $1;
	};

/* A character spec can be a set of characters inside of square parenthesis,
 * a dot specifying any character or some explicitly stated character. */
RegularExprChar:
	RE_SqOpen RegularExprOrData RE_SqClose {
		$$ = new ReItem( InputLoc(@1), $2, ReItem::OrBlock );
	} |
	RE_SqOpenNeg RegularExprOrData RE_SqClose {
		$$ = new ReItem( InputLoc(@1), $2, ReItem::NegOrBlock );
	} |
	RE_Dot {
		$$ = new ReItem( InputLoc(@1), ReItem::Dot );
	} |
	RE_Char {
		$$ = new ReItem( InputLoc(@1), $1[0] );
	};

/* The data inside of a [] expression in a regular expression. Accepts any
 * number of characters or ranges. */
RegularExprOrData:
	RegularExprOrData RegularExprOrChar {
		// An optimization to lessen the tree size. If an or char is directly
		// under the left side on the right and the right side is another or
		// char then paste them together and return the left side. Otherwise
		// just put the two under a new or data node.
		if ( $2->type == ReOrItem::Data &&
				$1->type == ReOrBlock::RecurseItem &&
				$1->item->type == ReOrItem::Data )
		{
			// Append the right side to right side of the left and toss
			// the right side.
			$1->item->str += $2->str;
			delete $2;
			$$ = $1;
		}
		else {
			// Can't optimize, put the left and right under a new node.
			$$ = new ReOrBlock( $1, $2 );
		}
	} | 
	/* Nothing */ {
		$$ = new ReOrBlock();
	};


/* A single character inside of an or expression. Can either be a character
 * or a set of characters. */
RegularExprOrChar:
	RE_Char {
		$$ = new ReOrItem( InputLoc(@1), $1[0] );
	} |
	RE_Char RE_Dash RE_Char {
		$$ = new ReOrItem( InputLoc(@2), $1[0], $3[0] );
	};

RangeLit:
	TK_Literal {
		// Range literas must have only one char.
		if ( strlen($1) != 1 ) {
			// Recover by using the literal anyways.
			error(@1) << "literal used in range must be of length 1" << endl;
		}
		$$ = new Literal( InputLoc(@1), $1, Literal::LitString );
	} |
	AlphabetNum {
		// Create a new literal number.
		$$ = new Literal( InputLoc(@1), $1, Literal::Number );
	};

%%

/* Try to do a definition, common to assignment and instantiation. Warns about 
 * instances other than main not being implemented yet. */
void tryMachineDef( const BISON_YYLTYPE &loc, char *name, Join *join, bool isInstance )
{
	GraphDictEl *newEl = id->pd->graphDict.insert( name );
	if ( newEl != 0 ) {
		/* New element in the dict, all good. */
		newEl->value = new VarDef( name, join );
		newEl->isInstance = isInstance;
		newEl->loc = loc;

		/* It it is an instance, put on the instance list. */
		if ( isInstance )
			id->pd->instanceList.append( newEl );
	}
	else {
		// Recover by ignoring the duplicate.
		error(loc) << "fsm \"" << name << "\" previously defined" << endl;
	}
}

void doInclude( const InputLoc &loc, char *sectionName, char *inputFile )
{
	/* Bail if we hit the max include depth. */
	if ( includeDepth == INCLUDE_STACK_SIZE ) {
		error(loc) << "hit maximum include depth of " << INCLUDE_STACK_SIZE << endl;
	}
	else {
		char *includeTo = id->pd->fsmName;

		/* Implement defaults for the input file and section name. */
		if ( inputFile == 0 ) 
			inputFile = id->fileName;
		if ( sectionName == 0 )
			sectionName = id->pd->fsmName;

		/* Parse the included file. */
		InputData *oldId = id;
		id = new InputData( inputFile, sectionName, includeTo );
		includeDepth += 1;
		yyparse();
		includeDepth -= 1;
		delete id;
		id = oldId;
	}
}

bool getParseData( const BISON_YYLTYPE &loc )
{
	/* Lookup only once. */
	if ( id->lookedUpSection )
		return id->active;

	/* If we don't have a section name then try to get one from the structure
	 * that we are in. */
	if ( id->sectionName == 0 ) {
		/* If we don't yet have a section name, try to get the fsm spec name from
		 * any structure we are contained in. */
		if ( id->structStack.length() > 0 ) {
			char *head = id->structStack[id->structStack.length()-1];
			if ( head != 0 )
				id->sectionName = head;
		}
	}

	/* If we have a structure name then use that to get a 
	 * parse data structure. */
	if ( id->sectionName != 0 ) {
		if ( id->includeSpec != 0 ) {
			if ( strcmp( id->sectionName, id->includeSpec ) == 0 )
				id->sectionName = id->includeTo;
			else
				id->active = false;
		}

		/* Lookup the parse data, if it is not there then create it. */
		SectionMapEl *sectionMapEl = sectionMap.find( id->sectionName );
		if ( sectionMapEl == 0 ) {
			ParseData *newPd = new ParseData( id->fileName, id->sectionName, 
					id->sectionLoc );
			sectionMapEl = sectionMap.insert( id->sectionName, newPd );
		}
		id->pd = sectionMapEl->value;
	}
	else { 
		/* We don't have a name to use, only option is to use the last parse
		 * data. If there was no last named parse data then this is an error. */
		if ( id->pd == 0 ) {
			error(loc) << "don't have a name for this FSM specification" << endl;
			id->active = false;
		}
	}

	id->lookedUpSection = true;
	return id->active;
}

/* Invoked at the end of a section. Decided what to output depending on the
 * mode that ragel is invoked in (graphviz, code) and the state of the parse. */
void endSection( const InputLoc &loc )
{
	/* In order to generate anything we must be in the top level file and the
	 * current spec must be active and there must not have been any parse
	 * errors. */
	if ( inc_stack_ptr > 0 || !id->active || id->pd->errorCount > 0 )
		return;

	switch ( outputFormat ) {
	case OutCCode: case OutCppCode: case OutObjCCode: {
		/* Generating C, C++ or Objective C code. */
		bool generatedOutput = false;
		if ( id->sawInterface ) {
			id->pd->generateInterface();
			generatedOutput = true;
		}
		if ( id->sawMain ) {
			id->pd->generateCode();
			generatedOutput = true;
		}
		if ( generatedOutput )
			id->pd->endCodeGen( id->sectionEndLoc );
		break;
	}
	case OutGraphvizDot:
		/* Generating a file that will be fed to graphviz. If we saw main or we
		 * were instructed to generate get some other name then do the
		 * generation. */
		if ( id->sawMain || machineName != 0 )
			id->pd->generateGraphviz();
		break;
	case OutPrint:
		/* Printing a human readable version the machine, currently
		 * unimplemented. Only proceed if we saw main or are after some other
		 * machine. */
		if ( id->sawMain || machineSpec != 0 )
			id->pd->printFsm();
		break;
	case OutDump:
		/* Printing a raw dump of the machine. Only proceed if we saw main or
		 * are after some other machine. */
		if ( id->sawMain )
			id->pd->dumpFsm();
		break;
	}
}

void yyerror( char *err )
{
	/* Bison won't give us the location, but in the last call to the scanner we
	 * saved a pointer to the locationn variable. Use that. instead. */
	error(::yylloc->first_line, ::yylloc->first_column) << err << endl;
}

