/*
 *  Copyright 2001-2003, 2005 Adrian Thurston <thurston@cs.queensu.ca>
 */

/*  This file is part of Ragel.
 *
 *  Ragel is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 * 
 *  Ragel is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *  You should have received a copy of the GNU General Public License
 *  along with Ragel; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 */

#include "ragel.h"
#include "tabcodegen.h"
#include "parsetree.h"
#include "redfsm.h"

/* Determine if we should use indicies or not. */
void TabCodeGen::calcIndexSize()
{
	int sizeWithInds = 0, sizeWithoutInds = 0;

	/* Calculate cost of using with indicies. */
	for ( RedStateList::Iter st = redFsm->stateList; st.lte(); st++ ) {
		int totalIndex = st->outSingle.length() + st->outRange.length() + 
				(st->defTrans == 0 ? 0 : 1);
		sizeWithInds += arrayTypeSize(maxIndex) * totalIndex;
	}
	sizeWithInds += arrayTypeSize(maxState) * redFsm->transSet.length();
	if ( anyActions() )
		sizeWithInds += arrayTypeSize(maxActionLoc) * redFsm->transSet.length();

	/* Calculate the cost of not using indicies. */
	for ( RedStateList::Iter st = redFsm->stateList; st.lte(); st++ ) {
		int totalIndex = st->outSingle.length() + st->outRange.length() + 
				(st->defTrans == 0 ? 0 : 1);
		sizeWithoutInds += arrayTypeSize(maxState) * totalIndex;
		if ( anyActions() )
			sizeWithoutInds += arrayTypeSize(maxActionLoc) * totalIndex;
	}

	/* If using indicies reduces the size, use them. */
	useIndicies = sizeWithInds < sizeWithoutInds;
}


std::ostream &TabCodeGen::STATE_OUT_ACTION( RedStateAp *state )
{
	/* This function is only called if there are any out actions, so need
	 * not guard against there being none. */
	int act = 0;
	if ( state->eofAction != 0 )
		act = state->eofAction->location+1;
	out << act;
	return out;
}

std::ostream &TabCodeGen::TRANS_ACTION( RedTransAp *trans )
{
	/* If there are actions, emit them. Otherwise emit zero. */
	int act = 0;
	if ( trans->action != 0 )
		act = trans->action->location+1;
	out << act;
	return out;
}

std::ostream &TabCodeGen::FIRST_FINAL_STATE()
{
	if ( redFsm->firstFinState != 0 )
		out << redFsm->firstFinState->id;
	else
		out << redFsm->nextStateId;
	return out;
}

std::ostream &TabCodeGen::ERROR_STATE()
{
	out << redFsm->errState->id;
	return out;
}

std::ostream &TabCodeGen::ACTION_SWITCH()
{
	/* Walk the list of functions, printing the cases. */
	for ( ActionList::Iter act = parseData->actionList; act.lte(); act++ ) {
		/* Write out referenced actions. */
		if ( act->numTransRefs > 0 || act->numEofRefs > 0 ) {
			/* Write the case label, the action and the case break. */
			out << "\tcase " << act->actionId << ":\n";
			ACTION( act, 0, false );
			out << "\tbreak;\n";
		}
	}

	/* Write the directive for going back into the output file. The line
	 * number is for the next line, so add one. */
	out << "#line " << outFilter->line + 1 << " \""; LDIR_PATH(outputFile) << "\"\n";
	return out;
}

std::ostream &TabCodeGen::KEY_OFFSETS()
{
	out << "\t";
	int totalStateNum = 0, curKeyOffset = 0;
	for ( RedStateList::Iter st = redFsm->stateList; st.lte(); st++ ) {
		/* Write the key offset. */
		out << curKeyOffset;
		if ( !st.last() ) {
			out << ", ";
			if ( ++totalStateNum % IALL == 0 )
				out << "\n\t";
		}

		/* Move the key offset ahead. */
		curKeyOffset += st->outSingle.length() + st->outRange.length()*2;
	}
	out << "\n";
	return out;
}


std::ostream &TabCodeGen::INDEX_OFFSETS()
{
	out << "\t";
	int totalStateNum = 0, curIndOffset = 0;
	for ( RedStateList::Iter st = redFsm->stateList; st.lte(); st++ ) {
		/* Write the index offset. */
		out << curIndOffset;
		if ( !st.last() ) {
			out << ", ";
			if ( ++totalStateNum % IALL == 0 )
				out << "\n\t";
		}

		/* Move the index offset ahead. */
		curIndOffset += st->outSingle.length() + st->outRange.length() + 1;
	}
	out << "\n";
	return out;
}

std::ostream &TabCodeGen::SINGLE_LENS()
{
	out << "\t";
	int totalStateNum = 0;
	for ( RedStateList::Iter st = redFsm->stateList; st.lte(); st++ ) {
		/* Write singles length. */
		out << st->outSingle.length();
		if ( !st.last() ) {
			out << ", ";
			if ( ++totalStateNum % IALL == 0 )
				out << "\n\t";
		}
	}
	out << "\n";
	return out;
}

std::ostream &TabCodeGen::RANGE_LENS()
{
	out << "\t";
	int totalStateNum = 0;
	for ( RedStateList::Iter st = redFsm->stateList; st.lte(); st++ ) {
		/* Emit length of range index. */
		out << st->outRange.length();
		if ( !st.last() ) {
			out << ", ";
			if ( ++totalStateNum % IALL == 0 )
				out << "\n\t";
		}
	}
	out << "\n";
	return out;
}


std::ostream &TabCodeGen::EOF_ACTIONS()
{
	out << "\t";
	int totalStateNum = 0;
	for ( RedStateList::Iter st = redFsm->stateList; st.lte(); st++ ) {
		/* Write any eof action. */
		STATE_OUT_ACTION(st);
		if ( !st.last() ) {
			out << ", ";
			if ( ++totalStateNum % IALL == 0 )
				out << "\n\t";
		}
	}
	out << "\n";
	return out;
}

std::ostream &TabCodeGen::KEYS()
{
	out << '\t';
	int totalTrans = 0;
	for ( RedStateList::Iter st = redFsm->stateList; st.lte(); st++ ) {
		/* Loop the singles. */
		for ( RedTransList::Iter stel = st->outSingle; stel.lte(); stel++ ) {
			KEY( stel->lowKey ) << ", ";
			if ( ++totalTrans % IALL == 0 )
				out << "\n\t";
		}

		/* Loop the state's transitions. */
		for ( RedTransList::Iter rtel = st->outRange; rtel.lte(); rtel++ ) {
			/* Lower key. */
			KEY( rtel->lowKey ) << ", ";
			if ( ++totalTrans % IALL == 0 )
				out << "\n\t";

			/* Upper key. */
			KEY( rtel->highKey ) << ", ";
			if ( ++totalTrans % IALL == 0 )
				out << "\n\t";
		}
	}

	/* Output one last number so we don't have to figure out when the last
	 * entry is and avoid writing a comma. */
	out << 0 << "\n";
	return out;
}

std::ostream &TabCodeGen::INDICIES()
{
	int totalTrans = 0;
	out << '\t';
	for ( RedStateList::Iter st = redFsm->stateList; st.lte(); st++ ) {
		/* Walk the singles. */
		for ( RedTransList::Iter stel = st->outSingle; stel.lte(); stel++ ) {
			out << stel->value->id << ", ";
			if ( ++totalTrans % IALL == 0 )
				out << "\n\t";
		}

		/* Walk the ranges. */
		for ( RedTransList::Iter rtel = st->outRange; rtel.lte(); rtel++ ) {
			out << rtel->value->id << ", ";
			if ( ++totalTrans % IALL == 0 )
				out << "\n\t";
		}

		/* The state's default index goes next. */
		if ( st->defTrans != 0 ) {
			out << st->defTrans->id << ", ";
			if ( ++totalTrans % IALL == 0 )
				out << "\n\t";
		}
	}

	/* Output one last number so we don't have to figure out when the last
	 * entry is and avoid writing a comma. */
	out << 0 << "\n";
	return out;
}

std::ostream &TabCodeGen::TRANS_TARGS()
{
	int totalTrans = 0;
	out << '\t';
	for ( RedStateList::Iter st = redFsm->stateList; st.lte(); st++ ) {
		/* Walk the singles. */
		for ( RedTransList::Iter stel = st->outSingle; stel.lte(); stel++ ) {
			RedTransAp *trans = stel->value;
			out << trans->targ->id << ", ";
			if ( ++totalTrans % IALL == 0 )
				out << "\n\t";
		}

		/* Walk the ranges. */
		for ( RedTransList::Iter rtel = st->outRange; rtel.lte(); rtel++ ) {
			RedTransAp *trans = rtel->value;
			out << trans->targ->id << ", ";
			if ( ++totalTrans % IALL == 0 )
				out << "\n\t";
		}

		/* The state's default target state. */
		if ( st->defTrans != 0 ) {
			RedTransAp *trans = st->defTrans;
			out << trans->targ->id << ", ";
			if ( ++totalTrans % IALL == 0 )
				out << "\n\t";
		}
	}

	/* Output one last number so we don't have to figure out when the last
	 * entry is and avoid writing a comma. */
	out << 0 << "\n";
	return out;
}


std::ostream &TabCodeGen::TRANS_ACTIONS()
{
	int totalTrans = 0;
	out << '\t';
	for ( RedStateList::Iter st = redFsm->stateList; st.lte(); st++ ) {
		/* Walk the singles. */
		for ( RedTransList::Iter stel = st->outSingle; stel.lte(); stel++ ) {
			RedTransAp *trans = stel->value;
			TRANS_ACTION( trans ) << ", ";
			if ( ++totalTrans % IALL == 0 )
				out << "\n\t";
		}

		/* Walk the ranges. */
		for ( RedTransList::Iter rtel = st->outRange; rtel.lte(); rtel++ ) {
			RedTransAp *trans = rtel->value;
			TRANS_ACTION( trans ) << ", ";
			if ( ++totalTrans % IALL == 0 )
				out << "\n\t";
		}

		/* The state's default index goes next. */
		if ( st->defTrans != 0 ) {
			RedTransAp *trans = st->defTrans;
			TRANS_ACTION( trans ) << ", ";
			if ( ++totalTrans % IALL == 0 )
				out << "\n\t";
		}
	}

	/* Output one last number so we don't have to figure out when the last
	 * entry is and avoid writing a comma. */
	out << 0 << "\n";
	return out;
}

std::ostream &TabCodeGen::TRANS_TARGS_WI()
{
	/* Transitions must be written ordered by their id. */
	RedTransAp **transPtrs = new RedTransAp*[redFsm->transSet.length()];
	for ( TransApSet::Iter trans = redFsm->transSet; trans.lte(); trans++ )
		transPtrs[trans->id] = trans;

	/* Keep a count of the num of items in the array written. */
	out << '\t';
	int totalStates = 0;
	for ( int t = 0; t < redFsm->transSet.length(); t++ ) {
		/* Write out the target state. */
		RedTransAp *trans = transPtrs[t];
		out << trans->targ->id;
		if ( t < redFsm->transSet.length()-1 ) {
			out << ", ";
			if ( ++totalStates % IALL == 0 )
				out << "\n\t";
		}
	}
	out << "\n";
	delete[] transPtrs;
	return out;
}


std::ostream &TabCodeGen::TRANS_ACTIONS_WI()
{
	/* Transitions must be written ordered by their id. */
	RedTransAp **transPtrs = new RedTransAp*[redFsm->transSet.length()];
	for ( TransApSet::Iter trans = redFsm->transSet; trans.lte(); trans++ )
		transPtrs[trans->id] = trans;

	/* Keep a count of the num of items in the array written. */
	out << '\t';
	int totalAct = 0;
	for ( int t = 0; t < redFsm->transSet.length(); t++ ) {
		/* Write the function for the transition. */
		RedTransAp *trans = transPtrs[t];
		TRANS_ACTION( trans );
		if ( t < redFsm->transSet.length()-1 ) {
			out << ", ";
			if ( ++totalAct % IALL == 0 )
				out << "\n\t";
		}
	}
	out << "\n";
	delete[] transPtrs;
	return out;
}



std::ostream &TabCodeGen::BSEARCH()
{
	out <<
		"static "; ALPH_TYPE() << " *"; FSM_NAME() << "_bsearch( "; ALPH_TYPE() << 
				" c, "; ALPH_TYPE() << " *keys, int len )\n"
		"{\n"
		"	"; ALPH_TYPE() << " *lower = keys;\n"
		"	"; ALPH_TYPE() << " *mid;\n"
		"	"; ALPH_TYPE() << " *upper = keys + len - 1;\n"
		"	while (1) {\n"
		"		if ( upper < lower )\n"
		"			return 0;\n"
		"\n"
		"		mid = lower + ((upper-lower) >> 1);\n"
		"		if ( c < *mid )\n"
		"			upper = mid - 1;\n"
		"		else if ( c > *mid )\n"
		"			lower = mid + 1;\n"
		"		else\n"
		"			return mid;\n"
		"	}\n"
		"}\n";

	return out;
}

std::ostream &TabCodeGen::RANGE_BSEARCH()
{
	out <<
		"static "; ALPH_TYPE() << " *"; FSM_NAME() << "_range_bsearch( "; ALPH_TYPE() << 
				" c, "; ALPH_TYPE() << " *keys, int len )\n"
		"{\n"
		"	"; ALPH_TYPE() << " *lower = keys;\n"
		"	"; ALPH_TYPE() << " *mid;\n"
		"	"; ALPH_TYPE() << " *upper = keys + len - 2;\n"
		"	while (1) {\n"
		"		if ( upper < lower )\n"
		"			return 0;\n"
		"\n"
		"		mid = lower + (((upper-lower) >> 1) & ~1);\n"
		"		if ( c < mid[0] )\n"
		"			upper = mid - 2;\n"
		"		else if ( c > mid[1] )\n"
		"			lower = mid + 2;\n"
		"		else\n"
		"			return mid;\n"
		"	}\n"
		"}\n";

	return out;
}

std::ostream &TabCodeGen::LOCATE_TRANS()
{
	out <<
		"	_keys = "; FSM_NAME() << "_k + "; FSM_NAME() << "_ko[_cs];\n"
		"	_trans = "; FSM_NAME() << "_io[_cs];\n"
		"\n"
		"	_slen = "; FSM_NAME() << "_sl[_cs];\n"
		"	if ( _slen > 0 ) {\n"
		"		"; ALPH_TYPE() << " *match = "; FSM_NAME() << 
						"_bsearch( "; GET_KEY() << ", _keys, _slen );\n"
		"		if ( match != 0 ) {\n"
		"			_trans += (match - _keys);\n"
		"			goto match;\n"
		"		}\n"
		"		_keys += _slen;\n"
		"		_trans += _slen;\n"
		"	}\n"
		"\n"
		"	_rlen = "; FSM_NAME() << "_rl[_cs];\n"
		"	if ( _rlen > 0 ) {\n"
		"		"; ALPH_TYPE() << " *match = "; FSM_NAME() << 
						"_range_bsearch( "; GET_KEY() << ", _keys, (_rlen<<1) );\n"
		"		if ( match != 0 ) {\n"
		"			_trans += ((match - _keys)>>1);\n"
		"			goto match;\n"
		"		}\n"
		"		_trans += _rlen;\n"
		"	}\n"
		"\n";

	return out;
}

std::ostream &TabCodeGen::GOTO( NameInst *name, bool inFinish )
{
	/* Lookup the target. Always guaranteed to return just one target. */
	RedEntryMapEl *targ = redFsm->entryMap.find( name->id );
	out << "{_cs = " << targ->value->id << "; goto again;}";
	return out;
}

std::ostream &TabCodeGen::GOTOE( InlineItem *ilItem, bool inFinish )
{
	out << "{_cs = (";
	INLINE_LIST( ilItem->children, 0, inFinish );
	out << "); goto again;}";
	return out;
}

std::ostream &TabCodeGen::CURS( bool inFinish )
{
	out << "(_ps)";
	return out;
}

std::ostream &TabCodeGen::TARGS( bool inFinish, int targState )
{
	out << "(_cs)";
	return out;
}

std::ostream &TabCodeGen::GET_ENTRY( NameInst *name, bool inFinish )
{
	RedEntryMapEl *targ = redFsm->entryMap.find( name->id );
	out << targ->value->id;
	return out;
}

std::ostream &TabCodeGen::NEXT( NameInst *name, bool inFinish )
{
	RedEntryMapEl *targ = redFsm->entryMap.find( name->id );
	out << "_cs = " << targ->value->id << ";";
	return out;
}

std::ostream &TabCodeGen::NEXTE( InlineItem *ilItem, bool inFinish )
{
	out << "_cs = (";
	INLINE_LIST( ilItem->children, 0, inFinish );
	out << ");";
	return out;
}

std::ostream &TabCodeGen::CALL( NameInst *name, int targState, bool inFinish )
{
	/* Lookup the target. Always guaranteed to return just one target. */
	RedEntryMapEl *targ = redFsm->entryMap.find( name->id );
	out << "{"; SELF() << "->stack["; SELF() << "->top++] = _cs; _cs = " << 
			targ->value->id << "; goto again;}";
	return out;
}

std::ostream &TabCodeGen::CALLE( InlineItem *ilItem, int targState, bool inFinish )
{
	out << "{"; SELF() << "->stack["; SELF() << "->top++] = _cs; _cs = (";
	INLINE_LIST( ilItem->children, targState, inFinish );
	out << "); goto again;}";
	return out;
}

std::ostream &TabCodeGen::RET( bool inFinish )
{
	out << "{_cs = "; SELF() << "->stack[--"; SELF() << "->top]; goto again;}";
	return out;
}

std::ostream &TabCodeGen::CONTEXT( int contextId )
{
	/* Make a vect to fill with context indexed by state. */
	int numVals = (redFsm->nextStateId>>3)+1;
	unsigned char *vals = new unsigned char[numVals];
	memset( vals, 0, sizeof(unsigned char)*numVals );

	/* For each state where the context is present, set the bit. */
	for ( RedStateList::Iter st = redFsm->stateList; st.lte(); st++ ) {
		if ( st->contextSet.find( contextId ) )
			vals[st->id>>3] |= 1 << (st->id&0x7);
	}

	/* Write out context info. */
	for ( int i = 0; i < numVals; ) {
		out << (unsigned int)vals[i];

		i += 1;
		if ( i < numVals ) {
			out << ", ";
			if ( i % IALL == 0 )
				out << "\n\t";
		}
	}
	out << "\n";

	delete[] vals;
	return out;
}

std::ostream &TabCodeGen::STATE_DATA()
{
	/* If there are any transtion functions then output the array. If there
	 * are none, don't bother emitting an empty array that won't be used. */
	if ( anyActions() ) {
		out <<
			"static "; ARRAY_TYPE(maxActArrItem) << " "; FSM_NAME() << "_a[] = {\n";
			ACTIONS_ARRAY() <<
			"};\n"
			"\n";
	}

	out << 
		"static "; ARRAY_TYPE(maxKeyOffset) << " "; FSM_NAME() << "_ko[] = {\n";
		KEY_OFFSETS() <<
		"};\n"
		"\n";

	out <<
 		"static "; ALPH_TYPE() << " "; FSM_NAME() << "_k[] = {\n";
		KEYS() <<
		"};\n"
		"\n";

	out << 
		"static "; ARRAY_TYPE(maxSingleLen) << " "; FSM_NAME() << "_sl[] = {\n";
		SINGLE_LENS() <<
		"};\n"
		"\n";

	out << 
		"static "; ARRAY_TYPE(maxRangeLen) << " "; FSM_NAME() << "_rl[] = {\n";
		RANGE_LENS() <<
		"};\n"
		"\n";

	out << 
		"static "; ARRAY_TYPE(maxIndexOffset) << " "; FSM_NAME() << "_io[] = {\n";
		INDEX_OFFSETS() <<
		"};\n"
		"\n";

	if ( useIndicies ) {
		out << 
			"static "; ARRAY_TYPE(maxIndex) << " "; FSM_NAME() << "_i[] = {\n";
			INDICIES() <<
			"};\n"
			"\n";

		out << 
			"static "; ARRAY_TYPE(maxState) << " "; FSM_NAME() << "_tt[] = {\n";
			TRANS_TARGS_WI() <<
			"};\n"
			"\n";

		if ( anyActions() ) {
			out << 
				"static "; ARRAY_TYPE(maxActionLoc) << " "; FSM_NAME() << "_ta[] = {\n";
				TRANS_ACTIONS_WI() <<
				"};\n"
				"\n";
		}
	}
	else {
		out << 
			"static "; ARRAY_TYPE(maxState) << " "; FSM_NAME() << "_tt[] = {\n";
			TRANS_TARGS() <<
			"};\n"
			"\n";

		if ( anyActions() ) {
			out << 
				"static "; ARRAY_TYPE(maxActionLoc) << " "; FSM_NAME() << "_ta[] = {\n";
				TRANS_ACTIONS() <<
				"};\n"
				"\n";
		}
	}

	if ( anyEofActions() ) {
		out <<
			"static "; ARRAY_TYPE(maxActionLoc) << " "; FSM_NAME() << "_ea[] = {\n";
			EOF_ACTIONS() <<
			"};\n"
			"\n";
	}

	out <<
		"static int "; FSM_NAME() << "_start = " 
				<< redFsm->startState->id << ";\n"
		"\n";

	return out;
}

/* Pass init data to base. */
CTabCodeGen::CTabCodeGen( char *fsmName, ParseData *parseData, 
		RedFsmAp *redFsm, std::ostream &out )
:
	FsmCodeGen(fsmName, parseData, redFsm, out)
{ }

void CTabCodeGen::writeOutCode()
{
	/* Write out state array data. */
	STATE_DATA();

	/* Init routine. */
	out << 
		"int "; FSM_NAME() << "_init( struct "; FSM_NAME() << " *fsm )\n"
		"{\n"
		"	fsm->curs = "; FSM_NAME() << "_start;\n";
	
	/* If there are any calls, then the stack top needs initialization. */
	if ( anyActionCalls() || anyActionRets() )
		out << "	fsm->top = 0;\n";

	INIT_CODE() <<
		"	if ( fsm->curs >= "; FIRST_FINAL_STATE() << " )\n"
		"		return 1;\n"
		"	return 0;\n"
		"}\n"
		"\n";

	/* The binary search for singles. */
	BSEARCH() << "\n";

	/* The binary search for a range. */
	RANGE_BSEARCH() << "\n";

	/* Execution function. */
	out << 
		"int "; FSM_NAME() << "_execute( struct "; FSM_NAME() << " *fsm, "; EL_TYPE() <<
				" *_data, int _len )\n"
		"{\n"
		"	"; EL_TYPE() << " *_p = _data-1;\n"
		"	"; EL_TYPE() << " *_pe = _data+_len;\n"
		"	int _slen, _rlen, _cs = fsm->curs";

	if ( anyRegCurStateRef() )
		out << ", _ps";
	out << ";\n	unsigned int _trans;\n";
		
	if ( anyActions() ) {
		out << "\t";
		ARRAY_TYPE(maxActArrItem) << " *_acts, _nacts;\n";
	}

	out <<
		"	"; ALPH_TYPE() << " *_keys;\n"
		"	fsm->curs = -1;\n"
		"\n";


	out <<
		"again:\n"
		"	if ( ++_p == _pe )\n"
		"		goto out;\n";

	if ( anyEofActions() ) {
		out <<
			"	if ( _p == 0 )\n"
			"		goto eofActions;\n"
			"\n";
	}
	
	LOCATE_TRANS() <<
		"\n"
		"match:\n";

	if ( anyRegCurStateRef() )
		out << "	_ps = _cs;\n";

	if ( useIndicies ) {
		out << "	_trans = "; 
		FSM_NAME() << "_i[_trans];\n";
	}

	out <<
		"	_cs = "; FSM_NAME() << "_tt[_trans];\n"
		"\n";

	if ( anyActions() ) {
		out <<
			"	if ( "; FSM_NAME() << "_ta[_trans] == 0 )\n"
			"		goto again;\n"
			"	_acts = "; FSM_NAME() << "_a + "; FSM_NAME() << "_ta[_trans];\n"
			"\n";

		/* Only used if there are out actions. */
		if ( anyEofActions() )
			out << "execFuncs:\n";

		out <<
			"	_nacts = *_acts++;\n"
			"	while ( _nacts-- > 0 ) {\n"
			"		switch ( *_acts++ ) {\n";
			ACTION_SWITCH() <<
			"		}\n"
			"	}\n"
			"\n";
	}

	out <<
		"	goto again;\n"
		"\n";

	if ( anyEofActions() ) {
		out <<
			"eofActions:\n"
			"	if ( "; FSM_NAME() << "_ea[_cs] != 0 ) {\n"
			"		_acts = "; FSM_NAME() << "_a + "; FSM_NAME() << "_ea[_cs];\n"
			"		_p = 0; _pe = (("; EL_TYPE() << "*)0)+1;\n"
			"		goto execFuncs;\n"
			"	}\n"
			"\n";
	}

	out <<
		"out:\n"
		"	fsm->curs = _cs;\n";

	if ( redFsm->errState != 0 ) {
		out << 
			"	if ( _cs == "; ERROR_STATE() << " )\n"
			"		return -1;\n";
	}

	out << 
		"	if ( _cs >= "; FIRST_FINAL_STATE() << " )\n"
		"		return 1;\n"
		"	return 0;\n"
		"}\n"
		"\n";

	/* Finish routine. */
	out <<
		"int "; FSM_NAME() << "_finish( struct "; FSM_NAME() << " *fsm )\n"
		"{\n";

	if ( anyEofActions() ) {
		/* May need to execute code in the action loop. Must use special
		 * params to execute func. */
		out << "	return "; FSM_NAME() << "_execute( fsm, 0, 1 );\n";
	}
	else {
		if ( redFsm->errState != 0 ) {
			out << 
				"	if ( fsm->curs == "; ERROR_STATE() << " )\n"
				"		return -1;\n";
		}

		out <<
			"	if ( fsm->curs >= "; FIRST_FINAL_STATE() << " )\n"
			"		return 1;\n"
			"	return 0;\n";
	}

	out << 
		"}\n"
		"\n";

	for ( ContextMap::Iter ctx = parseData->contextMap; ctx.lte(); ctx++ ) {
		out << "unsigned char _"; FSM_NAME() << "_ctxdata_" << ctx->key << "[] = {\n\t";
		CONTEXT( ctx->value->id );
		out << "};\n\n";
	}
}

/* Init base class. */
CppTabCodeGen::CppTabCodeGen( char *fsmName, ParseData *parseData, 
		RedFsmAp *redFsm, std::ostream &out ) 
: 
	FsmCodeGen(fsmName, parseData, redFsm, out) 
{ }

void CppTabCodeGen::writeOutCode()
{
	/* Write out state array data. */
	STATE_DATA();

	/* Init routine. */
	out << 
		"int "; FSM_NAME() << "::init( )\n"
		"{\n"
		"	this->curs = "; FSM_NAME() << "_start;\n";
	
	/* If there are any calls, then the stack top needs initialization. */
	if ( anyActionCalls() || anyActionRets() )
		out << "	this->top = 0;\n";

	INIT_CODE() <<
		"	if ( this->curs >= "; FIRST_FINAL_STATE() << " )\n"
		"		return 1;\n"
		"	return 0;\n"
		"}\n"
		"\n";

	/* The binary search for singles. */
	BSEARCH() << "\n";

	/* The binary search for a range. */
	RANGE_BSEARCH() << "\n";

	/* Execution function. */
	out << 
		"int "; FSM_NAME() << "::execute( "; EL_TYPE() << " *_data, int _len )\n"
		"{\n"
		"	"; EL_TYPE() << " *_p = _data-1;\n"
		"	"; EL_TYPE() << " *_pe = _data+_len;\n"
		"	int _slen, _rlen, _cs = this->curs";

	if ( anyRegCurStateRef() )
		out << ", _ps";
	out << ";\n	unsigned int _trans;\n";

	if ( anyActions() ) {
		out << "\t";
		ARRAY_TYPE(maxActArrItem) << " *_acts, _nacts;\n";
	}

	out <<
		"	"; ALPH_TYPE() << " *_keys;\n"
		"	this->curs = -1;\n"
		"\n";
	

	out << 
		"again:\n"
		"	if ( ++_p == _pe )\n"
		"		goto out;\n";

	if ( anyEofActions() ) {
		out << 
			"	if ( _p == 0 )\n"
			"		goto eofActions;\n"
			"\n";
	}

	LOCATE_TRANS() <<
		"\n"
		"match:\n";

	if ( anyRegCurStateRef() )
		out << "	_ps = _cs;\n";

	if ( useIndicies ) {
		out << "	_trans = "; 
		FSM_NAME() << "_i[_trans];\n";
	}

	out <<
		"	_cs = "; FSM_NAME() << "_tt[_trans];\n"
		"\n";

	if ( anyActions() ) {
		out <<
			"	if ( "; FSM_NAME() << "_ta[_trans] == 0 )\n"
			"		goto again;\n"
			"	_acts = "; FSM_NAME() << "_a + "; FSM_NAME() << "_ta[_trans];\n"
			"\n";

		/* Only used if there are out actions. */
		if ( anyEofActions() )
			out << "execFuncs:\n";

		out <<
			"	_nacts = *_acts++;\n"
			"	while ( _nacts-- > 0 ) {\n"
			"		switch ( *_acts++ ) {\n";
			ACTION_SWITCH() <<
			"		}\n"
			"	}\n"
			"\n";
	}

	out <<
		"	goto again;\n"
		"\n";
	
	if ( anyEofActions() ) {
		out << 
			"eofActions:\n"
			"	if ( "; FSM_NAME() << "_ea[_cs] != 0 ) {\n"
			"		_acts = "; FSM_NAME() << "_a + "; FSM_NAME() << "_ea[_cs];\n"
			"		_p = 0; _pe = (("; EL_TYPE() << "*)0)+1;\n"
			"		goto execFuncs;\n"
			"	}\n"
			"\n";
	}

	out <<
		"out:\n"
		"	this->curs = _cs;\n";

	if ( redFsm->errState != 0 ) {
		out << 
			"	if ( _cs == "; ERROR_STATE() << " )\n"
			"		return -1;\n";
	}

	out <<
		"	if ( _cs >= "; FIRST_FINAL_STATE() << " )\n"
		"		return 1;\n"
		"	return 0;\n"
		"}\n"
		"\n";

	/* Finish routine. */
	out <<
		"int "; FSM_NAME() << "::finish( )\n"
		"{\n";

	if ( anyEofActions() ) {
		/* May need to execute code in the action loop. Must use special
		 * params to execute func. */
		out << "	return execute( 0, 1 );\n";
	}
	else {
		if ( redFsm->errState != 0 ) {
			out << 
				"	if ( this->curs == "; ERROR_STATE() << " )\n"
				"		return -1;\n";
		}

		out <<
			"	if ( this->curs >= "; FIRST_FINAL_STATE() << " )\n"
			"		return 1;\n"
			"	return 0;\n";
	}

	out << 
		"}\n"
		"\n";

	for ( ContextMap::Iter ctx = parseData->contextMap; ctx.lte(); ctx++ ) {
		out << "unsigned char "; FSM_NAME() << "::_ctxdata_" << ctx->key << "[] = {\n\t";
		CONTEXT( ctx->value->id );
		out << "};\n\n";
	}
}

/* Init base class. */
ObjCTabCodeGen::ObjCTabCodeGen( char *fsmName, ParseData *parseData, 
		RedFsmAp *redFsm, std::ostream &out ) 
: 
	FsmCodeGen(fsmName, parseData, redFsm, out) 
{ }

void ObjCTabCodeGen::writeOutCode()
{
	/* Write out state array data. */
	STATE_DATA();

	out <<
		"- (int) initFsm;\n"
		"{\n"
		"\tself->curs = "; FSM_NAME() << "_start;\n";

	/* If there are any calls, then the stack top needs initialization. */
	if ( anyActionCalls() || anyActionRets() )
		out << "\tself->top = 0;\n\n";

	INIT_CODE() <<
		"\treturn ( self->curs >= "; FIRST_FINAL_STATE() << " ) ? 1 : 0;\n"
		"}\n"
		"\n";

	/* The binary search for singles. */
	BSEARCH() << "\n";

	/* The binary search for a range. */
	RANGE_BSEARCH() << "\n";

	/* Execution function. */
	out <<
		"- (int) executeWithData:("; EL_TYPE() << " *)_data len:(int)_len;\n"
		"{\n"
		"\t"; EL_TYPE() << " *_p = _data - 1;\n"
		"\t"; EL_TYPE() << " *_pe = _data + _len;\n"
		"\tint _slen, _rlen, _cs = self->curs";

	if ( anyRegCurStateRef() )
		out << ", _ps";
	out << ";\n	unsigned int _trans;\n";

	if ( anyActions() ) {
		out << "\t";
		ARRAY_TYPE(maxActArrItem) << " *_acts, _nacts;\n";
	}

	out <<
		"\t"; ALPH_TYPE() << " *_keys;\n"
		"\tself->curs = -1;\n"
		"\n";
	
	out << 
		"again:\n"
			"\tif ( ++_p == _pe )\n"
				"\t\tgoto out;\n";

	if ( anyEofActions() ) {
		out << 
			"\tif ( _p == 0 )\n"
				"\t\tgoto eofActions;\n"
			"\n";
	}

	LOCATE_TRANS() <<
		"\n"
		"match:\n";

	if ( anyRegCurStateRef() )
		out << "\t_ps = _cs;\n";

	if ( useIndicies ) {
		out << "\t_trans = "; 
		FSM_NAME() << "_i[_trans];\n";
	}

	out <<
		"\t_cs = "; FSM_NAME() << "_tt[_trans];\n"
		"\n";

	if ( anyActions() ) {
		out <<
			"\tif ( "; FSM_NAME() << "_ta[_trans] == 0 )\n"
				"\t\tgoto again;\n"
			"\t_acts = "; FSM_NAME() << "_a + "; FSM_NAME() << "_ta[_trans];\n"
			"\n";

		/* Only used if there are out actions. */
		if ( anyEofActions() )
			out << "execFuncs:\n";

		out <<
			"\t_nacts = *_acts++;\n"
			"\twhile ( _nacts-- > 0 ) {\n"
				"\t\tswitch ( *_acts++ ) {\n";
				ACTION_SWITCH() <<
				"\t\t}\n"
			"\t}\n"
			"\n";
	}

	out <<
		"\tgoto again;\n"
		"\n";
	
	if ( anyEofActions() ) {
		out << 
			"eofActions:\n"
				"\tif ( "; FSM_NAME() << "_ea[_cs] != 0 ) {\n"
					"\t\t_acts = "; FSM_NAME() << "_a + "; FSM_NAME() << "_ea[_cs];\n"
					"\t\t_p = 0; _pe = (("; EL_TYPE() << "*)0)+1;\n"
					"\t\tgoto execFuncs;\n"
				"\t}\n"
			"\n";
	}

	out <<
		"out:\n"
			"\tself->curs = _cs;\n";

	if ( redFsm->errState != 0 ) {
		out << 
			"\tif ( _cs == "; ERROR_STATE() << " ) return -1;\n";
	}

	out <<
			"\treturn ( _cs >= "; FIRST_FINAL_STATE() << " ) ? 1 : 0;\n"
		"}\n"
		"\n";

	out <<
		"- (int) finish;\n"
		"{\n";
	
	if ( anyEofActions() ) {
		/* May need to execute code in the action loop. Must use special
		 * params to execute func. */
		out <<
			"\treturn [self executeWithData:("; EL_TYPE() << "*)0 len:1];\n";
	}
	else {
		if ( redFsm->errState != 0 ) {
			out << 
			"\tif ( self->curs == "; ERROR_STATE() << " ) return -1;\n";
		}

		out <<
			"\treturn ( self->curs >= "; FIRST_FINAL_STATE() << " ) ? 1 : 0;\n";
	}

	out << 
		"}\n"
		"\n";

	for ( ContextMap::Iter ctx = parseData->contextMap; ctx.lte(); ctx++ ) {
		out << "unsigned char "; FSM_NAME() << "::_ctxdata_" << ctx->key << "[] = {\n\t";
		CONTEXT( ctx->value->id );
		out << "};\n\n";
	}
}
