/*
 * Parses unix mail boxes into headers and bodies.
 */

#include <stdio.h>
#include "vectsimp.h"

struct MBox
{
	int curs;
	VectSimp<char> headName;
	VectSimp<char> headContent;
	%% interface;
};

%% MBox
{
	# Buffer the header names.
	action bufHeadName { headName.append(fc); }

	# Buffer the header content.
	action bufHeadContent { headContent.append(fc); }

	# Terminate a header. If it is an interesting header then prints it.
	action finBufHeadContent {
		/* Terminate the buffers. */
		headName.append(0);
		headContent.append(0);

		/* Print the header. Interesting headers. */
		printf("%s:%s\n", headName.data, headContent.data);
		
		/* Clear for the next time we use them. */
		headName.empty();
		headContent.empty();
	}

	action msgstart{
		printf("NEW MESSAGE\n");
	}

	# Prints a blank line after the end of the headers of each message.
	action blankLine {
		printf("\n");
	}
	
	# Helpers we will use in matching the date section of the from line.
	day = /[A-Z][a-z][a-z]/;
	month = /[A-Z][a-z][a-z]/;
	year = /[0-9][0-9][0-9][0-9]/;
	time = /[0-9][0-9]:[0-9][0-9]/ . ( /:[0-9][0-9]/ | '' );
	letterZone = /[A-Z][A-Z][A-Z]/;
	numZone = /[+\-][0-9][0-9][0-9][0-9]/;
	zone = letterZone | numZone;
	dayNum = /[0-9 ][0-9]/;

	# These are the different formats of the date minus an obscure
	# type that has a funny string 'remote from xxx' on the end. Taken
	# from c-client in the imap-2000 distribution.
	date = day . ' ' . month . ' ' . dayNum . ' ' . time . ' ' .
		( year | year . ' ' . zone | zone . ' ' . year );

	# Note the priority assignment on the end of the from line. While we
	# matching the body of a message we may enter into this machine. We will
	# not leave the body of the previous message until this entire from line is
	# matched. 
	fromLine = 'From ' . /[^\n]/* . ' ' . date . '\n' @(new_msg,1) @msgstart;

	# The types of characters that can be used as a header name.
	hchar = print - [ :];

	header =
		# The name of the header.
		hchar+ $bufHeadName . ':' 
		# The content of the header. Look out for continuations.
		. ( (extend - '\n') $bufHeadContent | '\n'. [ \t] @bufHeadContent )*
		# Buffer must end with a newline that does not continue.
		. '\n' %finBufHeadContent;

	messageLine = ( extend - '\n' )* . '\n' @(new_msg, 0);

	# When we get to the last newline we are still matching messageLine
	# so on the last newline it will think we are still in the message.
	# We need this because we can't assume that every newline means
	# the end of the current message, whereas at the same time we requre
	# that there be a newline before the fromLine of the next message.
	message = ( fromLine .  header* .  '\n' @blankLine .  messageLine* . '\n' );

	# Its important that the priority in the fromLine gets bumped up
	# so that we are able to move to new messages. Otherwise we
	# will always stay in the message body of the first message.
	main := message*;

}

#include <stdio.h>
#define BUFSIZE 2048

MBox mbox;
char buf[BUFSIZE];

int main()
{
	mbox.init();
	while ( 1 ) {
		int len = fread( buf, 1, BUFSIZE, stdin );
		mbox.execute( buf, len );
		if ( len != BUFSIZE )
			break;
	}
	if ( mbox.finish() > 0 )
		printf("ACCEPT\n");
	else
		printf("FAIL\n");
	return 0;
}

