/*
 *  Copyright 2001-2006 Adrian Thurston <thurston@cs.queensu.ca>
 */

/*  This file is part of Ragel.
 *
 *  Ragel is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 * 
 *  Ragel is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *  You should have received a copy of the GNU General Public License
 *  along with Ragel; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 */

#ifndef _COMMON_H
#define _COMMON_H

#include <climits>

/* Types of alphabet supported by Ragel. */
enum AlphType
{
	AT_Char,
	AT_UnsignedChar,
	AT_Short,
	AT_UnsignedShort,
	AT_Int,
	AT_UnsignedInt,
	AT_Long,
	AT_UnsignedLong
};

/* Each is casted to an long size because the numbers read in are this size. */
#define RL_CHAR_MIN    ((long)((char)CHAR_MIN))
#define RL_CHAR_MAX    ((long)((char)CHAR_MAX))
#define RL_UCHAR_MIN   ((unsigned long)((unsigned char)0))
#define RL_UCHAR_MAX   ((unsigned long)((unsigned char)UCHAR_MAX))
#define RL_SHORT_MIN   ((long)((short)SHRT_MIN))
#define RL_SHORT_MAX   ((long)((short)SHRT_MAX))
#define RL_USHORT_MIN  ((unsigned long)((unsigned short)0))
#define RL_USHORT_MAX  ((unsigned long)((unsigned short)USHRT_MAX))
#define RL_INT_MIN     ((long)((int)INT_MIN))
#define RL_INT_MAX     ((long)((int)INT_MAX))
#define RL_UINT_MIN    ((unsigned long)((unsigned int)0))
#define RL_UINT_MAX    ((unsigned long)((unsigned int)UINT_MAX))
#define RL_LONG_MIN    ((long)LONG_MIN)
#define RL_LONG_MAX    ((long)LONG_MAX)
#define RL_ULONG_MIN   ((unsigned long)0)
#define RL_ULONG_MAX   ((unsigned long)ULONG_MAX)

/* An abstraction of the key operators that manages key operations such as
 * comparison and increment according the signedness of the key. */
struct KeyOps
{
	/* Default to signed alphabet. */
	KeyOps() 
		:isAlphSigned(true) {}

	/* Default to signed alphabet. */
	KeyOps( bool isSigned ) 
		:isAlphSigned(isSigned) {}

	/* Less than. */
	bool lt( long key1, long key2 ) {
		return isAlphSigned ?  key1 < key2 : 
			(unsigned long)key1 < (unsigned long)key2;
	}

	/* Less than or equal to. */
	bool lteq( long key1, long key2 ) {
		return isAlphSigned ?  key1 <= key2 : 
			(unsigned long)key1 <= (unsigned long)key2;
	}

	/* Greater than. */
	bool gt( long key1, long key2 ) {
		return isAlphSigned ? key1 > key2 : 
			(unsigned long)key1 > (unsigned long)key2;
	}

	/* Greater than or equal to. */
	bool gteq( long key1, long key2 ) {
		return isAlphSigned ? key1 >= key2 : 
			(unsigned long)key1 >= (unsigned long)key2;
	}

	/* Equality. */
	bool eq( long key1, long key2 ) {
		return key1 == key2;
	}

	/* Decrement. Needed only for ranges. */
	void dec( long &key ) {
		key = isAlphSigned ? key - 1 : ((unsigned long)key)-1;
	}

	/* Increment. Needed only for ranges. */
	void inc( long &key ) {
		key = isAlphSigned ? key+1 : ((unsigned long)key)+1;
	}

	/* Compute the distance between two keys. */
	unsigned long long span( long key1, long key2 ) {
		return isAlphSigned ? 
			(long long)key2 - (long long)key1 + 1 : 
			(unsigned long long)((unsigned long)key2) - 
				(unsigned long long)((unsigned long)key1) + 1;
	}

	bool isAlphSigned;
	long lowKey, highKey;
};

#endif /* _COMMON_H */
