/*
 * Show off concurrent abilities.
 */

#include <iostream>
#include <stdlib.h>
#include <stdio.h>

using namespace std;

#define BUFSIZE 2048

struct Concurrent
{
	int cur_char;
	int start_word;
	int start_comment;
	int start_literal;

	int cs;

	// Initialize the machine. Invokes any init statement blocks. Returns 0
	// if the machine begins in a non-accepting state and 1 if the machine
	// begins in an accepting state.
	void init( );

	// Execute the machine on a block of data. Returns -1 if after processing
	// the data, the machine is in the error state and can never accept, 0 if
	// the machine is in a non-accepting state and 1 if the machine is in an
	// accepting state.
	void execute( const char *data, int len );

	// Indicate that there is no more data. Returns -1 if the machine finishes
	// in the error state and does not accept, 0 if the machine finishes
	// in any other non-accepting state and 1 if the machine finishes in an
	// accepting state.
	int finish( );
};

%%{
	machine Concurrent;

	action next_char {
		cur_char += 1;
	}

	action start_word {
		start_word = cur_char;
	}
	action end_word {
		cout << "word: " << start_word << 
				" " << cur_char-1 << endl;
	}

	action start_comment {
		start_comment = cur_char;
	}
	action end_comment {
		cout << "comment: " << start_comment <<
				" " << cur_char-1 << endl;
	}

	action start_literal {
		start_literal = cur_char;
	}
	action end_literal {
		cout << "literal: " << start_literal <<
				" " << cur_char-1 << endl;
	}

	# Count characters.
	chars = ( any @next_char )*;

	# Words are non-whitespace. 
	word = ( any-space )+ >start_word %end_word;
	words = ( ( word | space ) $1 %0 )*;

	# Finds C style comments. 
	comment = ( '/*' any* $0 '*/'@1 ) >start_comment %end_comment;
	comments = ( ( comment | any ) $1 %0 )*;

	# Finds single quoted strings. 
	literalChar = ( any - ['\\] ) | ( '\\' . any );
	literal = ('\'' literalChar* '\'' ) >start_literal %end_literal;
	literals = ( ( literal | (any-'\'') ) $1 %0 )*;

	main := chars | words | comments | literals;
}%%

%% write data;

void Concurrent::init( )
{
	cur_char = 0;
	%% write init;
}

void Concurrent::execute( const char *data, int len )
{
	const char *p = data;
	const char *pe = data + len;

	%% write exec;
}

int Concurrent::finish( )
{
	%% write eof;
	if ( cs == Concurrent_error )
		return -1;
	if ( cs >= Concurrent_first_final )
		return 1;
	return 0;
}


int main()
{
	Concurrent concurrent;
	char buf[BUFSIZE];

	concurrent.init();
	while ( 1 ) {
		int len = fread( buf, 1, BUFSIZE, stdin );
		concurrent.execute( buf, len );
		if ( len != BUFSIZE )
			break;
	}

	if ( concurrent.finish() > 0 )
		cout << "ACCEPT" << endl;
	else
		cout << "FAIL" << endl;

	return 0;
}
