/*
   Name: $RCSfile: rapple.c,v $
   Author: Alan Moran
   $Date: 2005/11/26 15:21:09 $
   $Revision: 1.29 $
   $Id: rapple.c,v 1.29 2005/11/26 15:21:09 a_j_moran Exp $

   Legal Notice:

   This program is free software; you can redistribute it and/or
   modify it under the terms of the license contained in the
   COPYING file that comes with this distribution.

 */

/**
   @file

   @brief rapple command line client
*/

#include <unistd.h>
#include <sys/stat.h>
#include <dirent.h>

#include "globals.h"

int         version_flag = -1, del_flag = 0, check_config_flag = 0, help_flag = 0;
int         loglevel;
rpl_str_t   config = NULL, host = NULL, srcdir = NULL, webdir = NULL, dsdir = NULL, catalog_loc = NULL, local_user;

static void usage();
static void version(int verbose);
static void check_config();
static void rpl_src_dir_error(void);
static void rpl_output_dir_error(void);
static void rpl_ds_dir_error(void);
static void rpl_overlap_dir_error(void);

/**
   Prints usage message (e.g., in response to -h option).
 */
static void
usage() 
{
    printf("Usage: %s [-chvVx] [-f configfile] [-H host] [-d srcdir] [-o webdir]\n", PACKAGE_NAME);
    printf("Options:\n");
    printf("   -c              : check configuration\n");
    printf("   -d srcdir       : specify an alternative sources dir (defaults to config file settings)\n");
    printf("   -f configfile   : specify an alternative config file\n");
    printf("   -h              : show help (this message)\n");
    printf("   -H host         : specify an alternative host (domain)\n");
    printf("   -o webdir       : specify an alternative output dir (defaults to config file settings)\n");
    printf("   -v              : show version information\n");
    printf("   -V              : show verbose version information\n");
    printf("   -x              : delete datastore contents\n");
	printf("source directory defaults to the <srcdir> entry in the configuration file but can\n");
	printf("be overriden by the value given by the -d option.\n");
    exit(0);
}

/**
    Prints a message indicating a problem with the configuration file.
 */
static void
rpl_cfg_parse_error() 
{
    printf("There appears to be a problem with your configuration file!\n");
    printf("Please note that the configuration file is located in the following order of priority:\n");
    printf("   1. as specified by the -f option;\n");
    printf("   2. as determined by the value of the RAPPLE_CONF environment variable;\n");
    printf("   3. by searching the current working directory for the rapple.conf file.\n");
    printf("You must ensure that your configuration file is customized for your needs, is well formed and is"
           "accurate.\n");
}

/**
    Prints a message indicating a problem with the sources directory.
 */
static void
rpl_src_dir_error() 
{
	printf("You must specify a sources directory! Either:\n");
	printf("   1. provided a default value in the <srcdir> element of the configuration file; or\n");
	printf("   2. specify an alternative location using the -d option.\n");
	exit(1);
}

/**
    Prints a message indicating a problem with the output directory.
 */
static void
rpl_output_dir_error() 
{
	printf("You must specify an output directory! Either:\n");
	printf("   1. provided a default value in the <webdir> element of the configuration file; or\n");
	printf("   2. specify an alternative location using the -o option.\n");
	exit(1);
}

#if SUPPORT_XSLT_PROC == 2
/**
    Prints a message indicating a problem with the catalog location.
 */
static void
rpl_catalog_error()
{
	printf("You must specify the catalog location for libxslt.  Either:\n");
	printf("   1. provide a valid location in the <catalog> element of the configuration file; or\n");
	printf("   2. set the XML_CATALOG_FILES environment variable.\n");
	exit(1);
}
#endif

/**
    Prints a message indicating a problem with the datastore.
 */
static void
rpl_ds_dir_error() 
{
	printf("No datastore has been defined.  Please check your configuration file.\n");
	exit(1);
}

/**
    Prints a message indicating a problem with overlapping source and output directories.
 */
static void
rpl_overlap_dir_error()
{
	printf("The following conditions constrain the source, web and datastore directories:\n");
	printf("   1. source and output directories may neither coincide nor be nested!\n");
	printf("   2. the datastore may not coincide or be contained in the source directory\n");
	exit(1);
}

/**
   Prints version information (i.e., in response to -v option).
 */
static void
version(int verbose) 
{
    printf("%s\n", PACKAGE_STRING);
    if(verbose) {
        printf("Copyright 2005 Alan Moran <a_j_moran@users.sourceforge.net>\n");
        printf("project homepage http://rapple.sourceforge.net/\n");
    }
    exit(0);
}

/**
   Prints configuration information.
 */
static void
check_config() 
{
	rpl_list            *dg_names = NULL;
	rpl_str_list_node   *dg_node;

    /* configure rapple (if config is NULL then default file is searched for) */
    rpl_cfg_parse(config);

    printf("\nexamining your configuration.\n\n");
    printf("configuration file is well-formed!\n");
	printf("source dir..................%s\n", srcdir);
	printf("web dir.....................%s\n", webdir);
    printf("config file.................%s\n", rpl_cfg_get_config_filename());
    printf("logfile.....................%s\n", rpl_cfg_get_logfile());
    loglevel = rpl_cfg_get_loglevel();
    printf("loglevel....................%d (%s)\n", loglevel, rpl_log_get_level_desc(loglevel));
    printf("datastore...................%s\n", dsdir);
    printf("tidy conf...................%s\n", (rpl_cfg_get_trf_tidy_config()) ? rpl_cfg_get_trf_tidy_config()
           : "None defined! Using system defaults.");
    printf("domain......................%s\n", host);
    printf("xslt........................%s\n", rpl_cfg_get_trf_tpl_xslt());
	if(rpl_cfg_get_linkchecker_xml_report() != NULL)
	{
		printf("link checker XSLT...........%s\n", rpl_cfg_get_linkchecker_xslt());
		printf("link checker report (XML)...%s\n", rpl_cfg_get_linkchecker_xml_report());
		printf("link checker report (HTML)..%s\n", rpl_cfg_get_linkchecker_html_report());
	}
#if SUPPORT_XSLT_PROC == 2
    printf("catalog.....................%s\n", catalog_loc);
#endif
	dg_names = rpl_cfg_get_dg_dir_names();
	if(rpl_list_count(dg_names) > 0)
	{
   		printf("the following directories will be indexed:\n");
		dg_node = rpl_list_first(dg_names);
		while(dg_node != NULL)
		{
			if(dg_node->str)
   				printf("............................%s\n", dg_node->str);
			dg_node = rpl_list_next(&dg_node->node);
		}
	}
#ifdef HAVE_MYSQL
	printf("MySQL support...............enabled (ask administrator for credentials)\n");
    printf("name........................%s\n", rpl_cfg_get_db_name());
    printf("host........................%s\n", rpl_cfg_get_db_host());
    printf("port........................%d\n", rpl_cfg_get_db_port());
#else
	printf("MySQL support...............disabled\n");
#endif
    printf("\ndone! please check that the above values are what you expect.\n\n");
    exit(0);
}

int
main(int argc, char** argv) 
{
    int rc, result;

	/* init resources */
    rpl_message_init();

    while(optind < argc) {
        result = getopt(argc, argv, "cd:hH:o:uvVf:x");
        /* reached end of the list */
        if(result == -1)
            break;
        switch(result) {
            /* unrecognised option */
        case '?':
            break;
            /* missing parameter for option */
        case ':':
            break;
            /* check configuration */
        case 'c':
            check_config_flag = 1;
            break;
			/* specify alternative sources directory */
        case 'd':
            srcdir = rpl_me_malloc(strlen(optarg) + 1);
            sprintf(srcdir, "%s", optarg);
            break;
            /* alternate config filename */
        case 'f':
            config = rpl_me_malloc(strlen(optarg) + 1);
            sprintf(config, "%s", optarg);
            break;
            /* display help */
        case 'h':
            help_flag = 1;
            break;
            /* alternate config filename */
        case 'H':
            host = rpl_me_malloc(strlen(optarg) + 1);
            sprintf(host, "%s", optarg);
            break;
			/* specify alternative output directory */
        case 'o':
            webdir = rpl_me_malloc(strlen(optarg) + 1);
            sprintf(webdir, "%s", optarg);
            break;
            /* display version information */
        case 'v':
            version_flag = 0;
            break;
            /* display verbose version information */
        case 'V':
            version_flag = 1;
            break;
            /* delete datastore contents */
        case 'x':
            del_flag = 1;
            break;
        default:
            usage();
            break;
        }
    }

    /* configure rapple (if config is NULL then default file is searched for) */
    rc = rpl_cfg_parse(config);
    if((rc == RPL_CFG_FILE_NOT_FOUND) || (rc == RPL_CFG_PARSE_ERROR)) {
        rpl_cfg_parse_error();
        rpl_message_done();
        exit(EXIT_FAILURE);
    }

	/* deletion can be issued independently */
	if(del_flag > 0)
	{
		rpl_fs_delete_ds();
    	return(EXIT_SUCCESS);
	}

	/* acquire datastore directory */
	if((dsdir = rpl_cfg_get_ds_basedir()) == NULL)
		rpl_ds_dir_error();

	/* set default sources and output directories */
	if(srcdir == NULL)
	{
		if((srcdir = rpl_cfg_get_srcdir()) == NULL)
			rpl_src_dir_error();
	}
	if(webdir == NULL)
	{
		if((webdir = rpl_cfg_get_webdir()) == NULL)
			rpl_output_dir_error();
	}

#if SUPPORT_XSLT_PROC == 2
	/* set catalog location */
	if((catalog_loc = rpl_cfg_get_trf_catalog()) == NULL)
		catalog_loc = getenv("XML_CATALOG_FILES");
	if(catalog_loc == NULL)
		rpl_catalog_error();
#endif

	/* set default host (domain) */
	if(host == NULL)
	{
		if((host = rpl_cfg_get_trf_tpl_domain()) == NULL)
			rpl_output_dir_error();
	}
	
	/* check overlapping of sources, output and datastore directories */
	if((strcmp(srcdir,webdir) == 0) 
			|| (strstr(srcdir, webdir) != NULL) 
				|| (strstr(webdir, srcdir) != NULL)
					|| (strcmp(srcdir, dsdir) == 0)
						|| strstr(srcdir, dsdir) != NULL)
		rpl_overlap_dir_error();

	/* process flags */
    if(help_flag)
        usage();

    if(version_flag >= 0) 
        version(version_flag);

    if(check_config_flag > 0) 
        check_config();

	/* invoke workflow */
	rpl_wk_process(host, srcdir, webdir);

	/* free resources */
    rpl_cfg_cleanup(); 
	/* TODO: there appears to be an intermittent free issue here - perhaps 
	   a resource is being freed in mod_link code ? */
    /* rpl_message_done(); */
    rpl_log_close_logfile();

    return(EXIT_SUCCESS);
}

