/*
   Name: $RCSfile: webasset.c,v $
   Author: Alan Moran
   $Date: 2005/11/14 21:30:44 $
   $Revision: 1.20 $
   $Id: webasset.c,v 1.20 2005/11/14 21:30:44 a_j_moran Exp $

   Legal Notice:

   This program is free software; you can redistribute it and/or
   modify it under the terms of the license contained in the
   COPYING file that comes with this distribution.

 */

/**
   @file

   @brief Web asset data structure manipulation functions.

   The webasset module defines an abstract data type (ADT) for web assets
   (e.g., HTML webpages, PDF documents, images etc.) and furnishes an interface
   with which to query the specifics of an asset (its name, relative directory,
   name of remote user in authenticated HTTP context etc.)  It is the purpose
   of the client code to invoke this interface to populate and retrieve values
   from the type.  However, in doing so the client is insulated from
   implementation changes that may occur behind the interface (e.g., the
   algorithm used to determine under what conditions and asset is to be
   considered transformable etc.)
*/

#include <pwd.h>
#include <sys/types.h>

#include "globals.h"
#include "parser.h"
#include "mime.h"

const rpl_wa_trf_status_description trf_status[] = {
    { RPL_WA_TRF_ST_SUCCESS, "Success" },
    { RPL_WA_TRF_ST_FAILURE, "Failure" },
    { RPL_WA_TRF_ST_WARNING, "Warning" },
    { RPL_WA_TRF_ST_UNDEF, "Undefined" },
};
int trf_size= sizeof(trf_status) / sizeof(trf_status[0]);

static void     rpl_wa_copy_str(rpl_str_t src, rpl_str_t *dest);

/**
   Copies the contents of src into dest. Convenience function used throughout this file.

   @param src
   @param dest
 */
static
void rpl_wa_copy_str(rpl_str_t src, rpl_str_t *dest)
{
    size_t length;

    assert(src != NULL);

    length = strlen(src) + 1;
    *dest = (rpl_str_t)rpl_me_malloc(length);
    snprintf(*dest, length, "%s", src);
}

/**
   Initialises the rpl_web_asset structure pointed to by wa.

   @param wa
 */
void
rpl_wa_init(rpl_web_asset *wa)
{
	struct passwd *uidpwent;

    wa->dts = time(NULL);
	wa->key = RPL_STR_NUL;
	/* set the contributer to the name of the process owner */
	uidpwent = getpwuid(getuid());
	wa->contrib = rpl_me_malloc(strlen(uidpwent->pw_name) + 1);
	sprintf(wa->contrib, uidpwent->pw_name);
    wa->filename = RPL_STR_NUL;
    wa->rel_dir = RPL_STR_NUL;
    wa->mime_type = RPL_STR_NUL;
    wa->tidy_stat_desc = RPL_STR_NUL;
    wa->tmpl_stat_desc = RPL_STR_NUL;
}

/**
   Frees resources being held by rpl_web_asset structure pointed to by wa.

   @param wa
 */
void
rpl_wa_destroy(rpl_web_asset *wa)
{
    assert (wa != NULL);

    if (wa->key != RPL_STR_NUL)
        rpl_me_free(wa->key);
    if (wa->contrib != RPL_STR_NUL)
        rpl_me_free(wa->contrib);
    if (wa->filename != RPL_STR_NUL)
        rpl_me_free(wa->filename);
    if (wa->rel_dir != RPL_STR_NUL)
        rpl_me_free(wa->rel_dir);
    if (wa->mime_type != RPL_STR_NUL)
        rpl_me_free(wa->mime_type);
    if (wa->tidy_stat_desc != RPL_STR_NUL)
        rpl_me_free(wa->tidy_stat_desc);
    if (wa->tmpl_stat_desc != RPL_STR_NUL)
        rpl_me_free(wa->tmpl_stat_desc);
}

/**
   Sets key.  This implementation requires that the filename
   and relative directory already be set.

   @param wa
 */
void
rpl_wa_set_key(rpl_web_asset *wa)
{
	assert(wa != NULL);

	/* key requires that both filename and rel_dir be set */
	assert(rpl_wa_get_filename(*wa) && rpl_wa_get_rel_dir(*wa));

	wa->key = rpl_reg_create_key(wa->rel_dir, wa->filename);
}

/**
   Returns registry key (used for lookup purposes)

   @param wa webasset containing key
   
   @return unique key
*/
rpl_str_t
rpl_wa_get_key(rpl_web_asset wa)
{
	return wa.key;
}

/**
   Sets user.

   @param contrib name of contributor as determined by local process owner or HTTP authentication
   @param wa
 */
void
rpl_wa_set_contributor(rpl_str_t contrib, rpl_web_asset *wa)
{
    assert((contrib != NULL) && (wa != NULL));

    rpl_wa_copy_str(contrib, &wa->contrib);
}

/**
   Returns remote user.

   @param wa

   @return remote user.
 */
rpl_str_t
rpl_wa_get_contributor(rpl_web_asset wa)
{
    return wa.contrib;
}

/**
   Sets filename.

   @param filename
   @param wa
 */
void
rpl_wa_set_filename(rpl_str_t filename, rpl_web_asset *wa)
{
    assert((filename != NULL) && (wa != NULL));
    rpl_wa_copy_str(filename, &wa->filename);
}

/**
   Returns filename.

   @param wa

   @return filename
 */
rpl_str_t
rpl_wa_get_filename(rpl_web_asset wa)
{
    return wa.filename;
}

/**
   Sets the relative directory directory.

   @param rel_dir
   @param wa
 */
void
rpl_wa_set_rel_dir(rpl_str_t rel_dir, rpl_web_asset *wa)
{
    assert((rel_dir != NULL) && (wa != NULL));
    rpl_wa_copy_str(rel_dir, &wa->rel_dir);
}

/**
   Returns the relative directory.

   @param wa

   @return the relative directory.
 */
rpl_str_t
rpl_wa_get_rel_dir(rpl_web_asset wa)
{
    return wa.rel_dir;
}

/**
   Sets MIME type.

   @param mime_type
   @param wa
 */
void
rpl_wa_set_mime_type(rpl_str_t mime_type, rpl_web_asset *wa)
{
    assert((mime_type != NULL) && (wa != NULL));
    rpl_wa_copy_str(mime_type, &wa->mime_type);
}

/**
   Gets MIME type.

   @param wa
 */
rpl_str_t
rpl_wa_get_mime_type(rpl_web_asset wa)
{
    return wa.mime_type;
}

/**
   Sets human readable tidy status description.

   @param src
   @param wa
 */
void
rpl_wa_set_tidy_stat_desc(rpl_str_t desc, rpl_web_asset *wa)
{
    assert((desc != NULL) && (wa != NULL));
    rpl_wa_copy_str(desc, &wa->tidy_stat_desc);
}

/**
   Returns description of tidy status.

   @param wa

   @return description of tidy status.
 */
rpl_str_t
rpl_wa_get_tidy_stat_desc(rpl_web_asset wa)
{
    return wa.tidy_stat_desc;
}

/**
   Sets human readable xslt status description.

   @param src
   @param wa
 */
void
rpl_wa_set_tmpl_stat_desc(rpl_str_t desc, rpl_web_asset *wa)
{
    assert((desc != NULL) && (wa != NULL));
    rpl_wa_copy_str(desc, &wa->tmpl_stat_desc);
}

/**
   Returns description of XSLT transformation status

   @param wa

   @return description of XSLT transformation status.
 */
rpl_str_t
rpl_wa_get_tmpl_stat_desc(rpl_web_asset wa)
{
    return wa.tmpl_stat_desc;
}

/**
   Determines whether an asset possesses a supported MIME type.

   @param wa
   @return 1 (supported), 0 (unsupported).
 */
int
rpl_wa_is_supported_mime_type(rpl_web_asset wa)
{
    return rpl_mime_is_supported_type(wa.mime_type);
}

/**
   Determines whether an asset is transformable.

   @param wa
   @return 1 (transformable), 0 (not transformable).
 */
int
rpl_wa_is_transformable(rpl_web_asset wa)
{
    return rpl_mime_is_transformable(wa.mime_type);
}

/**
   Returns the human readable description corresponding to code. Returns RPL_STR_NUL is code does not
   corresponding to transformation status code.

   @param code

   @return human readable description corresponding to code or RPL_STR_NUL if code is not found.
 */
rpl_str_t
rpl_wa_get_trf_status_desc(unsigned int code)
{
    int i;

    for(i=0; i<trf_size; i++)
        if(code == trf_status[i].code) break;

    return (i<trf_size) ? trf_status[i].desc : RPL_STR_NUL;
}


