/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU Lesser General Public License as         *
 *  published by the Free Software Foundation; either version 2 of the     *
 *  License, or (at your option) any later version.                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  Lesser General Public License for more details.                        *
 *                                                                         *
 *  You should have received a copy of the GNU Lesser General Public       *
 *  License along with this program; if not, write to the                  *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/
#ifndef RDSDNSZONE_H
#define RDSDNSZONE_H

#include <ClientProxy>
#include <QxtPimpl>
#include <rdsglobal.h>

#include <QStringList>

class RdsDnsZonePrivate;
class RdsDnsRecord;

/**
	@author Chris Vickery <chris@resara.com>
 */
class RDS_CLIENT_EXPORT RdsDnsZone : public QtRpc::ClientProxy
{
	Q_OBJECT;
	QXT_DECLARE_PRIVATE(RdsDnsZone);
	QTRPC_CLIENTPROXY_PIMPL(RdsDnsZone);
public:
	RdsDnsZone(QObject *parent = 0);
	RdsDnsZone(const RdsDnsZone &mgr);
	~RdsDnsZone();

	RdsDnsZone &operator=(const RdsDnsZone &mgr);

	/// IMPORTANT! This is copied on the server side, too. Keep them synced plox!
	enum RecordType
	{
		Invalid,
		A,
		AAAA,
		AFSDB,
		CERT,
		CNAME,
		DHCID,
		DLV,
		DNAME,
		DNSKEY,
		DS,
		HIP,
		IPSECKEY,
		KEY,
		LOC,
		MX,
		NAPTR,
		NS,
		NSEC,
		NSEC3,
		NSEC3PARAM,
		PTR,
		RRSIG,
		SIG,
		SOA,
		SPF,
		SRV,
		SSHFP,
		TA,
		TXT
	};
	static QString typeToString(RecordType type);
	static RecordType stringToType(const QString& type);

signals:
	/**
	 *        Parses a zone file
	 * @param config the zone file to parse
	 * @return true.  Check isValid() and errorString() for errors.
	 */
	ReturnValue parse(const QString &config);
	/**
		 *         Gets the currently loaded zone file.
		 * @return QString on success, error on failure.
	 */
	ReturnValue toString() const;
	/**
		 *        Saves the current zone information to a file
		 * @return true on success, error on failure.
	 */
	ReturnValue save();

	/**
		 *        Gets the domain name for the zone
		 * @return domain name as a QString
	 */
	ReturnValue domainName() const;
	/**
		 *        Sets the Domain Name for the zone
		 * @param domainName the domain name
		 * @return true
	 */
	ReturnValue setDomainName(const QString &domainName);

	/**
		 *        Gets the name of the primary name server
		 * @return returns the server name as a QString
	 */
	ReturnValue primaryNameServer() const;
	/**
		 *        Sets the name of the primary name server for the zone
		 * @param primaryNameServer name of the name server
		 * @return true
	 */
	ReturnValue setPrimaryNameServer(const QString &primaryNameServer); //must also have a host entry if it's not an IP, which it shouldn't be
	/**
		 *        Gets the name of the administrator name server
		 * @return the server name as a QString
	 */
	ReturnValue adminNameServer() const;
	/**
		 *        Sets the name of the administrator name server for the zone
		 * @param ns the name of the admininstrator name server
		 * @return true
	 */
	ReturnValue setAdminNameServer(const QString &ns);
	/**
		 *         Gets a the SOA values for the zone
		 * @return QVariantList of SOA vlaues
	 */
	ReturnValue soaValues() const;
	/**
		 *        Sets the SOA values for the zone.  Must be 5 values in the list to be valid.
		 * @param values a list of the SOA values
		 * @return true
	 */
	ReturnValue setSoaValues(const QStringList &values);

	/**
		 *        Adds a resource record to the zone
		 * @param type the resource record type (NS, A, CNAME,...)
		 * @param key the key to identify the RR
		 * @param value the value of the RR
		 * @return true if a new record was added, false if it replaced an existing record
	 */
	ReturnValue addRecord(int type, const QString &key, const QString &value);
	ReturnValue addRecord(int type, const QString &key, const QString &value, const QString &origin);
	ReturnValue addRecord(int type, const QString &key, const QString &value, const QString &origin, const QString &ttl);
	ReturnValue addRecord(int type, const QString &key, const QStringList &value);
	ReturnValue addRecord(int type, const QString &key, const QStringList &value, const QString &origin);
	ReturnValue addRecord(int type, const QString &key, const QStringList &value, const QString &origin, const QString &ttl);

	/**
		 *        Adds a resource record to the zone (ignores comments, origin, and ttl)
		 * @param record the record to add
		 * @return true if a new record was added, false if it replaced an existing record
	 */
	ReturnValue addRecord(const RdsDnsRecord &record); //return value is if it replaced or not
	/**
		 *        Removes a record from the zone
		 * @param type the type of record being removed
		 * @param key the key of the record to remove
		 * @return true if the record was removed, false if not (record did not exist)
	 */
	ReturnValue removeRecord(int type, const QString &key); //return value is if it existed or not
	/**
		 *        Removes a record from the zone
		 * @param record The record to remove
		 * @return true if the record was removed, false if not (record did not exist)
	 */
	ReturnValue removeRecord(const RdsDnsRecord &record); //return value is if it existed or not
	/**
		 *        Gets a resource record from the zone
		 * @param type the type of record to get
		 * @param key the key of the desired record
		 * @return returns a DnsRecord of the record, if the record was not found the type will be RdsDnsZone::Invalid
	 */
	ReturnValue record(int type, const QString &key) const;
	ReturnValue record(int type, const QString &key, const QString &origin) const;
	ReturnValue records(int type, const QString &key) const;
	ReturnValue records(int type, const QString &key, const QString &origin) const;
	/**
		 *         Gets a list of all the records from the zone
		 * @return DnsRecordList
	 */
	ReturnValue listRecords() const;
	/**
		 *        Gets a list of records with the specified type
		 * @param type the type of records to get
		 * @return DnsRecordList
	 */
	ReturnValue listRecordsByType(int type) const;
	/**
		 *        Gets a list of records with the specified key
		 * @param key the key of the desired records
		 * @return DnsRecordList
	 */
	ReturnValue listRecordsByKey(const QString &key) const;
	/**
		 *        Gets a list of records with the specified origin value
		 * @param origin the origin of the records to list
		 * @return DnsRecordList
	 */
	ReturnValue listRecordsByOrigin(const QString &origin) const;

	/**
		 * 	  TODO fix so it actually returns things...
		 * @param root
		 * @return QStringList
	 */
	ReturnValue listOrigins(const QString &root) const;
	/**
		 *
		 * @return QStringList
	 */
	ReturnValue listOrigins() const;

	/**
		 *        checks the validity of the zone
		 * @return true if valid, false if not
	 */
	ReturnValue isValid() const;
	/**
		 *        Gets the Error String
		 * @return QString
	 */
	ReturnValue errorString() const;

	ReturnValue addOrigin(const QString &origin);
	ReturnValue removeOrigin(const QString &origin);
	ReturnValue renameOrigin(const QString &oldname, const QString &newname);
};

Q_DECLARE_METATYPE(RdsDnsZone);


#endif
