/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  The contents of this file are property of Resara LLC. The viewing or   *
 *  distribution of this file in any form without express permission from  *
 *  Resara LLC is strictly prohibited.                                     *
 *  If you do not have permission from Resara LLC you must immediately     *
 *  delete this file and all copies of it to avoid legal action.           *
 *                                                                         *
 ***************************************************************************/
#include "acemodel.h"
#include <QDebug>

AceModel::AceModel(QObject *parent)
		: QAbstractItemModel(parent)
{
	_advanced = false;
	_allowflags = None;
	_inheritedallowflags = None;
}


AceModel::~AceModel()
{
}


int AceModel::rowCount(const QModelIndex &parent) const
{
	if (!parent.isValid())
	{
		return(_permissions.count());
	}
	else
	{
		Permission *perm = (Permission *)parent.internalPointer();
		return(perm->children.size());
	}
}

int AceModel::columnCount(const QModelIndex &parent) const
{
	Q_UNUSED(parent);
	return(3);
}

QModelIndex AceModel::index(int row, int column, const QModelIndex &parent) const
{
	if (!parent.isValid())
	{
		if (row >= _permissions.size()) return(QModelIndex());

		return(createIndex(row, column, _permissions[row]));
	}
	else
	{
		Permission *perm = (Permission *)parent.internalPointer();
		if (perm == NULL) return(QModelIndex());
		if (row >= perm->children.size()) return(QModelIndex());

		return(createIndex(row, column, perm->children[row]));
	}
}

QModelIndex AceModel::parent(const QModelIndex &index) const
{
	if (!index.isValid()) return(QModelIndex());
	Permission *perm = (Permission *)index.internalPointer();
	if (perm == NULL) return(QModelIndex());
	if (perm->parent == NULL) return(QModelIndex());

	Permission *parent = perm->parent->parent;
	if (parent == NULL)
	{
		int row = _permissions.indexOf(perm->parent);
		if (row == -1) return(QModelIndex());

		return(createIndex(row, 0, perm->parent));
	}
	else
	{
		int row = parent->children.indexOf(perm->parent);
		if (row == -1) return(QModelIndex());

		return(createIndex(row, 0, perm->parent));
	}
}

Qt::ItemFlags AceModel::flags(const QModelIndex &index) const
{
	if (!index.isValid()) return(0);
	Permission *perm = (Permission *)index.internalPointer();
	if (perm == NULL) return(0);



	if (index.column() == 0)
	{
		return(Qt::ItemIsEnabled);
	}
	else if (index.column() == 1)
	{
		Qt::ItemFlags enabled = 0;
		if ((perm->flags & _inheritedallowflags) != perm->flags) enabled = Qt::ItemIsEnabled;

		return(enabled | Qt::ItemIsUserCheckable | Qt::ItemIsTristate);
	}
	else if (index.column() == 2)
	{
		Qt::ItemFlags enabled = 0;
		if ((perm->flags & _inheriteddenyflags) != perm->flags) enabled = Qt::ItemIsEnabled;

		return(enabled | Qt::ItemIsUserCheckable | Qt::ItemIsTristate);
	}

	return(0);
}

QVariant AceModel::data(const QModelIndex &index, int role) const
{
	if (!index.isValid()) return(QVariant());
	Permission *perm = (Permission *)index.internalPointer();
	if (perm == NULL) return(QVariant());

	if (role == Qt::DisplayRole)
	{
		if (index.column() == 0) return(perm->name);
	}
	else if (role == Qt::CheckStateRole)
	{
		if (perm->flags == 0) return(QVariant());
		if (index.column() == 1)
		{
			Flags flags = _allowflags | _inheritedallowflags;
			int state = perm->flags & flags;
			if (state == 0) return(Qt::Unchecked);
			else if (state == perm->flags) return(Qt::Checked);
			else return(Qt::Unchecked);
		}
		else if (index.column() == 2)
		{
			if ((perm->display & AllowOnly) != 0) return(QVariant());
			Flags flags = _denyflags | _inheriteddenyflags;
			int state = perm->flags & flags;
			if (state == 0) return(Qt::Unchecked);
			else if (state == perm->flags) return(Qt::Checked);
			else return(Qt::Unchecked);
		}
	}

	return(QVariant());
}

bool AceModel::setData(const QModelIndex &index, const QVariant &value, int role)
{
	if (!index.isValid()) return(false);
	Permission *perm = (Permission *)index.internalPointer();
	if (perm == NULL) return(false);
	if (role != Qt::CheckStateRole) return(false);

	if(perm->flags == Special) return(false);
	
	if (index.column() == 1)
	{
		if ((perm->flags & _inheritedallowflags) == perm->flags) return(false);

		if (value.toBool())
		{
			//Don't set flags that are already inherited
			Flags flags = perm->flags; // & ~_inheritedallowflags;

			_allowflags |= flags;
			_denyflags &= ~flags;
			
			if ((perm->display & Exclusive) != 0)
			{
				_allowflags &= ~perm->excluded;
			}

			checkExcluded();
			updateAll();
		}
		else
		{
			Flags flags = perm->flags;
			if (perm->flags == FullControl)
			{
				flags &= ~Modify;
			}
			else if (perm->flags == Modify)
			{
				flags &= ~(Read | List | ReadAndExec | Write);
			}
			else if(perm->flags == ReadAndExec)
			{
				flags &= ~(Read | List);
			}
			_allowflags &= ~flags;
			updateAll();
		}
	}
	else if (index.column() == 2)
	{
		if ((perm->flags & _inheriteddenyflags) == perm->flags) return(false);

		if (value.toBool())
		{
			//Don't set flags that are already inherited
			Flags flags = perm->flags; // & ~_inheriteddenyflags;

			_denyflags |= flags;
			_allowflags &= ~flags;
			updateAll();
		}
		else
		{
			Flags flags = perm->flags;
			if (perm->flags == FullControl)
			{
				flags &= ~Modify;
			}
			else if (perm->flags == Modify)
			{
				flags &= ~(Read | List | ReadAndExec | Write);
			}
			else if(perm->flags == ReadAndExec)
			{
				flags &= ~(Read | List);
			}
			_denyflags &= ~flags;
			updateAll();
		}
	}

	

	flagsChanged((RdsAce::AccessFlag)(int)(_allowflags & (FullControl | 1 << 25)),
		     (RdsAce::AccessFlag)(int)(_denyflags & (FullControl | 1 << 25)),
	             (RdsAce::Flag)(int)((_allowflags >> 27) ^ RdsAce::InheritOnly),
		     (RdsAce::Flag)(int)((_lastinheritence >> 27) ^ RdsAce::InheritOnly));

	_lastinheritence = _allowflags;
		     
	return(false);
}

QVariant AceModel::headerData(int section, Qt::Orientation orientation, int role) const
{
	Q_UNUSED(orientation);
	if (role != Qt::DisplayRole) return(QVariant());
	switch (section)
	{
		case 0:
			return("Permission");
		case 1:
			return("Allow");
		case 2:
			return("Deny");
	}

	return(QVariant());
}


AceModel::Permission *AceModel::createPermission(QString name, Flags flags, Permission *parent, DisplayFlags display, Flags excluded)
{
	Permission *perm = new Permission();
	perm->name = name;
	perm->flags = flags;
	perm->parent = parent;
	perm->display = display;
	perm->excluded = excluded;

	if (parent == NULL)
	{
		_permissions << perm;
	}
	else
	{
		parent->children << perm;
	}

	return(perm);
}

void AceModel::setAdvanced(bool advanced)
{
	_advanced = advanced;

	beginResetModel();
	
	QModelIndexList indexes = persistentIndexList();
	QModelIndexList newlist;
	for(int i = 0; i < indexes.count(); i++)
	{
		newlist << QModelIndex();
	}
	
	changePersistentIndexList(indexes,newlist);
	
	QList<Permission *> remove = removePermissions();
	foreach(Permission *perm, remove)
	{
		delete(perm);
	}
	_permissions.clear();

	Permission *read = NULL, *write = NULL, *special = NULL, *full = NULL, *inherit = NULL;
	if (!advanced) full = createPermission("Full Control", FullControl | ReadAndExec);
	else full = createPermission("Full Control", 0);
	if (advanced) createPermission("Delete Subfolders and Files", DeleteTree, full);
	if (advanced) createPermission("Change Permissions", ChangePermissions , full);
	if (advanced) createPermission("Take Ownership", TakeOwnership , full);
	if (!advanced) createPermission("Modify", Modify | ReadAndExec);
	if (!advanced) createPermission("Read & Excecute", ReadAndExec);
	if (!advanced) createPermission("List Folder Contents", List);
	if (!advanced) read = createPermission("Read", Read);
	else read = createPermission("Read", 0);
	if (advanced) createPermission("Enter Folder / Execute File", List, read);
	if (advanced) createPermission("List Folder / Read Data", ReadData, read);
	if (advanced) createPermission("Read Attributes", ReadAttributes, read);
	if (advanced) createPermission("Read Extended Attributes", ReadExtendedAttributes, read);
	if (advanced) createPermission("Read Permissions", ReadPermissions, read);
	if (!advanced) write = createPermission("Write", Write);
	else write = createPermission("Write", 0);
	if (advanced) createPermission("Create Files / Write Data", CreateFile, write);
	if (advanced) createPermission("Create Folder / Append Data", CreateFolder, write);
	if (advanced) createPermission("Write Attributes", WriteAttributes, write);
	if (advanced) createPermission("Write Extended Attributes", WriteExtendedAttributes, write);
	if (advanced) createPermission("Delete", Delete, write);
	if (!advanced) special = createPermission("Special Permissions", Special);
	if (advanced) inherit = createPermission("Inherit To", 0);
	if (advanced) createPermission("Files", FileInherit, inherit, AllowOnly);
	if (advanced) createPermission("Folders", FolderInherit, inherit, AllowOnly);
	if (advanced) createPermission("This Folder", InheritOnly, inherit, AllowOnly);
	if (advanced) createPermission("Don't Inherit", NoPropogate, inherit, AllowOnly  | Exclusive, FileInherit | FolderInherit | InheritOnly);
	RdsAce::AccessFlag

	resetModel();
}

QList<AceModel::Permission *> AceModel::removePermissions(Permission *perm)
{
	QList<Permission *> list;
	if (perm == NULL)
	{
		foreach(Permission *child, _permissions)
		{
			list << child;
			list << removePermissions(child);
		}
	}
	else
	{
		foreach(Permission *child, perm->children)
		{
			list << child;
			list << removePermissions(child);
		}
	}

	return(list);
}

void AceModel::updateAll(const QModelIndex &parent)
{
	for (int i = 0; i < rowCount(parent); i++)
	{
		dataChanged(index(i, 1, parent), index(i, 2, parent));
		updateAll(index(i, 0, parent));
	}
}

void AceModel::checkExcluded(Permission *perm)
{
	QList<Permission *> list;
	if (perm == NULL)
	{
		list = _permissions;
	}
	else
	{
		list = perm->children;
	}

	foreach(Permission *p, list)
	{
		if (((p->display & Exclusive) == Exclusive)
		        && ((p->flags & _allowflags) = p->flags)
		        && ((p->excluded & _allowflags) != 0))
		{
			_allowflags &= ~p->flags;
		}

		checkExcluded(p);
	}
}

void AceModel::setFlags(RdsAce::AccessFlags allow, RdsAce::AccessFlags deny, RdsAce::AccessFlags inheritedallow,
			RdsAce::AccessFlags inheriteddeny, bool allowspecial, bool denyspecial, RdsAce::Flags inheritflags)
{
	_allowflags = (AceModel::Flag)(int)allow;
	_denyflags = (AceModel::Flag)(int)deny;
	_inheritedallowflags = (AceModel::Flag)(int)inheritedallow;
	_inheriteddenyflags = (AceModel::Flag)(int)inheriteddeny;
	inheritflags ^= RdsAce::InheritOnly;
	_allowflags |= Flags((AceModel::Flag)((int)inheritflags << 27)) & InheritFlags;
	
	if(allowspecial) _inheritedallowflags |= Special;
	if(denyspecial) _inheriteddenyflags |= Special;
	
	_lastinheritence = _allowflags;
	
	reset();
}


