/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU General Public License as published by   *
 *  the Free Software Foundation; either version 2 of the License, or (at  *
 *  your option) any later version.                                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  General Public License for more details.                               *
 *                                                                         *
 *  You should have received a copy of the GNU General Public License      *
 *  along with this program; if not, write to the                          *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/


#include "computermanagerwidget.h"
#include <QMessageBox>
#include <RdsClient>
#include <QDebug>
#include <QInputDialog>
#include <RdsOrganizationalUnit>
#include <QMenu>
#include <QSettings>
#include <QShortcut>
#include "computereditwidget.h"
#include "newuserdialog.h"
#include "oueditwidget.h"
#include "groupeditwidget.h"
#include "changepassworddialog.h"
#include "config.h"

extern QStringList __systemids;
extern QStringList __protectedids;


ComputerManagerWidget::ComputerManagerWidget(QWidget* parent, Qt::WFlags fl)
		: QWidget(parent, fl), Ui::ComputerManagerWidget()
{
	setupUi(this);
	pushButton_4->setIcon(QPixmap(findRdsIcon("icons/16x16/clear_left.png")));
	AddOuButton->setIcon(QPixmap(findRdsIcon("icons/32x32/add.png")));
	DeleteButton->setIcon(QPixmap(findRdsIcon("icons/32x32/remove.png")));
	ReloadButton->setIcon(QPixmap(findRdsIcon("icons/32x32/reload.png")));
	_edit = NULL;

	ReturnValue ret = rdsClient()->getService("UserGroupComputerManager");
	if (ret.isError())
	{
		QMessageBox::critical(this, "Failed to get user manager", ret.errString());
		return;
	}

	_manager = new RdsEntityManager(this);
	*_manager = ret;

	_model = new RdsUserGroupComputerModel(_manager, this);
	_sort = new RdsUserGroupComputerSortModel(_model);
	_sort->showUsers(false);
	_sort->showGroups(false);
	_sort->showComputers(true);
	_sort->setSourceModel(_model);
	UserView->setModel(_sort);
	UserView->sortByColumn(0, Qt::AscendingOrder);
	_selection = UserView->selectionModel();
	_model->setupSelection(_sort, UserView);
	UserView->header()->resizeSection(0,200);

	QObject::connect(_model, SIGNAL(inputChanged(QString, QString)), this, SLOT(inputChanged(QString, QString)));
	QObject::connect(_model, SIGNAL(outputsChanged(QStringList)), this, SLOT(outputsChanged(QStringList)));
	QObject::connect(FilterEntry, SIGNAL(textChanged(QString)), _sort, SLOT(setFilterRegExp(QString)));

	ComputerEditWidget *w = new ComputerEditWidget(_manager, EditFrame);
	EditLayout->addWidget(w);
	_edit = w;
	w->setInput("");

	//Actions and Menu stuff
	UserView->setContextMenuPolicy(Qt::CustomContextMenu);
	QObject::connect(UserView, SIGNAL(customContextMenuRequested(const QPoint&)), this, SLOT(contextMenu(const QPoint&)));

	_newouaction = new QAction("New OU", this);
	QObject::connect(_newouaction, SIGNAL(triggered()), this, SLOT(on_AddOuButton_clicked()));
	_removeaction = new QAction("Delete", this);
	QObject::connect(_removeaction, SIGNAL(triggered()), this, SLOT(on_DeleteButton_clicked()));
	_showsystemaction = new QAction("Show System Objects", this);
	QObject::connect(_showsystemaction, SIGNAL(toggled(bool)), this, SLOT(showSystemObjects(bool)));
	_showsystemaction->setCheckable(true);

	QSettings settings;
	if (settings.value("Computers/ShowSystemsObjects").toBool())
	{
		_showsystemaction->setChecked(true);
		_sort->setHiddenEntities(QStringList());
	}
	else
	{
		_showsystemaction->setChecked(false);
		_sort->setHiddenEntities(__systemids);
	}
	
	_removeaction->setShortcut(QKeySequence("Del"));
	new QShortcut(QKeySequence("Del"), this, SLOT(on_DeleteButton_clicked()));
}

ComputerManagerWidget::~ComputerManagerWidget()
{
}


void ComputerManagerWidget::inputChanged(QString id, QString type)
{
	_input = id;
	//qDebug() << "Input Changed:" << id << type;
	if (_edit != NULL)
	{
		if (_edit->unsavedChanges() && _edit->isEnabled())
		{
			if (QMessageBox::question(this, "Unsaved Changes", "The User you were editing has unsaved changes. Would you like to apply them?", QMessageBox::Apply | QMessageBox::Discard, QMessageBox::Discard) == QMessageBox::Apply)
			{
				_edit->apply();
			}
		}

		delete _edit;
		_edit = NULL;
	}

	if (type == "computer")
	{
		ComputerEditWidget *w = new ComputerEditWidget(_manager, EditFrame);
		EditLayout->addWidget(w);
		_edit = w;
		w->setInput(id);
	}
	else if (type == "ou")
	{
		OuEditWidget *w = new OuEditWidget(_manager, EditFrame);
		EditLayout->addWidget(w);
		_edit = w;
		w->setInput(id);
	}
	else
	{
		ComputerEditWidget *w = new ComputerEditWidget(_manager, EditFrame);
		EditLayout->addWidget(w);
		_edit = w;
		w->setInput("");
	}
}

void ComputerManagerWidget::outputsChanged(QStringList outputs)
{
	//qDebug() << "Output Changed" << outputs;
	if (_edit != NULL) _edit->setOutputs(outputs);
}

void ComputerManagerWidget::on_AddOuButton_clicked()
{
	QString ou = "";
	RdsEntityModel::Cache *cache = _model->getCache(_input);
	if (cache != NULL)
	{
		if (cache->type() == "ou") ou = _input;
		else if ((cache->parent() != NULL) && (cache->parent()->type() == "ou")) ou = cache->parent()->id();
	}

	bool ok = false;
	QString name = QInputDialog::getText(this, "Add OU", "Please enter the name for the new organizational unit.", QLineEdit::Normal, "", &ok);

	if (!ok) return;
	if (name == "")
	{
		QMessageBox::critical(this, "Error", "You must specify a name.");
		return;
	}
	
	if (!name.contains(QRegExp("^[0-9a-zA-Z][^+\\;,=]*$")))
	{
		QMessageBox::critical(NULL, "Error", "The name must not contain any of these characters (+ \\ ; , =).");
		return;
	}

	RdsOrganizationalUnit o(ou);
	ReturnValue ret = o.createOu(name);

	if (ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to create OU: " + ret.errString());
		return;
	}

	_manager->addEntity(ret.toString());
}

void ComputerManagerWidget::on_DeleteButton_clicked()
{
	QStringList outputs = _model->outputs();

	if (outputs.size() == 0)
	{
		QMessageBox::information(this,"Error", "You must select something to delete.");
		return;
	}
	else if (outputs.size() == 1)
	{
		RdsEntityModel::Cache *cache = _model->getCache(outputs[0]);
		if (cache == NULL) return;

		int result;

		if (cache->type() == "ou")
		{
			result = QMessageBox::question(this, "Delete OU", QString("Do you want to delete %1").arg(cache->name()),
			                               QMessageBox::Yes | QMessageBox::No, QMessageBox::No);
		}
		else if (cache->type() == "computer")
		{
			result = QMessageBox::question(this, "Delete Computer", QString("Do you want to delete %1").arg(cache->name()),
			                               QMessageBox::Yes | QMessageBox::No, QMessageBox::No);
		}
		else return;

		if (result != QMessageBox::Yes) return;
	}
	else
	{
		int result = QMessageBox::question(this, "Delete Items", QString("Do you want to delete %1 items?").arg(outputs.count()),
		                                   QMessageBox::Yes | QMessageBox::No, QMessageBox::No);

		if (result != QMessageBox::Yes)	return;
	}

	ReturnValue ret, err(true);

	UserView->selectionModel()->clear();
	inputChanged("","");
	
	//If we get here, we delete the outputs
	foreach(QString output, outputs)
	{
		if ((__systemids.contains(output)) || (__protectedids.contains(output)))
		{
			err = ReturnValue(1, QString("The selected item is a system object."));
			continue;
		}

		ret = rdsLdapSession()->remove(output,true);
		if (ret.isError())
			err = ret;
		else
			_manager->removeEntity(output);
	}

	if (err.isError())
	{
		QMessageBox::warning(this, "Warning", "Some items could not be deleted: " + err.errString());
	}
}

void ComputerManagerWidget::contextMenu(const QPoint &)
{
	QMenu menu;
	menu.addAction(_newouaction);
	menu.addAction(_removeaction);
	menu.addSeparator();
	menu.addAction(_showsystemaction);

	menu.exec(QCursor::pos());
}

void ComputerManagerWidget::showSystemObjects(bool show)
{
	QSettings settings;
	settings.setValue("Computers/ShowSystemsObjects", show);
	if (show)
	{
		_sort->setHiddenEntities(QStringList());
	}
	else
	{
		_sort->setHiddenEntities(__systemids);
	}
}

void ComputerManagerWidget::on_ReloadButton_clicked()
{
	_model->reload();
}

