/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU General Public License as published by   *
 *  the Free Software Foundation; either version 2 of the License, or (at  *
 *  your option) any later version.                                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  General Public License for more details.                               *
 *                                                                         *
 *  You should have received a copy of the GNU General Public License      *
 *  along with this program; if not, write to the                          *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/

#include "datetimewidget.h"
#include <QTimer>
#include <QMessageBox>
#include <RdsClient>

DateTimeWidget::DateTimeWidget(QWidget* parent, Qt::WFlags fl)
		: QWidget(parent, fl), Ui::DateTimeWidget()
{
	setupUi(this);
	
	update();
	
	QTimer *timer = new QTimer(this);
	QObject::connect(timer, SIGNAL(timeout()), this, SLOT(updateTime()));
	timer->start(1000);
}

DateTimeWidget::~DateTimeWidget()
{
}

void DateTimeWidget::updateTime()
{
	if((_datetext != DateEdit->text()) || (_timetext != TimeEdit->text())) return;
	_datetime = _datetime.addSecs(1);
	TimeEdit->setDateTime(_datetime);
	DateEdit->setDateTime(_datetime);
	_datetext = DateEdit->text();
	_timetext = TimeEdit->text();
}

void DateTimeWidget::on_DateEdit_editingFinished()
{
	if(_datetext == DateEdit->text()) return;
	ApplyButton->setEnabled(true);
	DiscardButton->setEnabled(true);
	_datetime.setDate(DateEdit->date());
	_datetime.setTime(TimeEdit->time());
	_datetext = DateEdit->text();
	_timetext = TimeEdit->text();
	_datetimeedited = true;
}

void DateTimeWidget::on_TimeEdit_editingFinished()
{
	if(_timetext == TimeEdit->text()) return;
	ApplyButton->setEnabled(true);
	DiscardButton->setEnabled(true);
	_datetime.setTime(TimeEdit->time());
	_datetext = DateEdit->text();
	_timetext = TimeEdit->text();
	_datetimeedited = true;
}

void DateTimeWidget::on_ApplyButton_clicked()
{
	if(_timezone != TimeZoneChooser->currentText())
	{
		ReturnValue ret = rdsClient()->setTimeZone(this, SLOT(setTimeZoneReturned(uint, ReturnValue)),  TimeZoneChooser->currentText().replace(" ","_"));
		if(ret.isError())
		{
			QMessageBox::critical(this,"Error","Failed to set the time zone: " + ret.errString());
			update();
			return;
		}
	}
	else
	{
		setTimeZoneReturned(1,true);
	}
}

void DateTimeWidget::setDateReturned(uint id, ReturnValue ret)
{
	Q_UNUSED(id);
	if(ret.isError())
	{
		QMessageBox::critical(this,"Error","Failed to set date and time: " + ret.errString());
	}
	update();
}

void DateTimeWidget::setTimeZoneReturned(uint id, ReturnValue ret)
{
	Q_UNUSED(id);
	if(ret.isError())
	{
		QMessageBox::critical(this,"Error","Failed to set the time zone: " + ret.errString());
		update();
		return;
	}
	
	if(_datetimeedited)
	{
		ReturnValue ret = rdsClient()->setDateTime(this, SLOT(setDateReturned(uint, ReturnValue)),_datetime);
		if(ret.isError())
		{
			QMessageBox::critical(this,"Error","Failed to set date and time: " + ret.errString());
			update();
			return;
		}
	}
	else
	{
		update();
	}
}

void DateTimeWidget::on_DiscardButton_clicked()
{
	update();
}

void DateTimeWidget::update()
{
	ReturnValue ret = rdsClient()->dateTime();
	if(ret.isError())
	{
		setEnabled(false);
		QMessageBox::critical(this,"Error","Failed to get date and time: " + ret.errString());
		return;
	}
	
	_datetime = ret.toDateTime();
	TimeEdit->setDateTime(_datetime);
	DateEdit->setDateTime(_datetime);
	
	ApplyButton->setEnabled(false);
	DiscardButton->setEnabled(false);
	_datetext = DateEdit->text();
	_timetext = TimeEdit->text();
	_datetimeedited = false;
	
	ret = rdsClient()->timeZone();
	if(ret.isError())
	{
		setEnabled(false);
		QMessageBox::critical(this,"Error","Failed to get time zone: " + ret.errString());
		return;
	}
	
	_timezone = ret.toString().replace("_"," ");
	
	ret = rdsClient()->listTimeZones();
	if(ret.isError())
	{
		setEnabled(false);
		QMessageBox::critical(this,"Error","Failed to list time zones: " + ret.errString());
		return;
	}
	
	QStringList tmp = ret.toStringList();
	
	QStringList zones;
	foreach(QString zone, tmp)
	{
		zones << zone.replace("_", " ");
	}
	if(!zones.contains(_timezone)) zones << _timezone;
	zones.sort();
	TimeZoneChooser->clear();
	TimeZoneChooser->addItems(zones);
	TimeZoneChooser->setCurrentIndex(TimeZoneChooser->findText(_timezone));
}

void DateTimeWidget::on_TimeZoneChooser_activated(QString timezone)
{
	if(_timezone != timezone)
	{
		ApplyButton->setEnabled(true);
		DiscardButton->setEnabled(true);
	}
}