/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU General Public License as published by   *
 *  the Free Software Foundation; either version 2 of the License, or (at  *
 *  your option) any later version.                                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  General Public License for more details.                               *
 *                                                                         *
 *  You should have received a copy of the GNU General Public License      *
 *  along with this program; if not, write to the                          *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/


#include "dhcpsubneteditwidget.h"
#include <RdsEntityManager>
#include <QMessageBox>
#include <QDebug>
#include <RdsUtils>
#include <RdsDaemonManager>
#include <RdsFileManager>
#include <RdsClient>
#include <RdsDhcpSubnet>

#define IP_VALIDATOR QRegExp("^([0-9]{0,3}\\.){0,3}[0-9]{0,3}$")
#define IP_VALIDATOR_STRONG QRegExp("^(([0-9][0-9]?)|(1[0-9][0-9]?)|(2[0-4][0-9])|(25[0-5]))\\.(([0-9][0-9]?)|(1[0-9][0-9]?)|(2[0-4][0-9])|(25[0-5]))\\.(([0-9][0-9]?)|(1[0-9][0-9]?)|(2[0-4][0-9])|(25[0-5]))\\.(([0-9][0-9]?)|(1[0-9][0-9]?)|(2[0-4][0-9])|(25[0-5]))$")


DhcpSubnetEditWidget::DhcpSubnetEditWidget(RdsDhcpManager *manager, QWidget* parent)
		: RdsEntityWidget(manager, parent), Ui::DhcpSubnetEditWidget()
{
	setupUi(this);
	setApplyButton(ApplyButton);
	setDiscardButton(DiscardButton);

	_manager = manager;
	_configured = true;

	addWidget("name", Name, NoMulti, "^[^/]+$", "The name is not valid.");
	addWidget("network", Network, NoMulti, IP_VALIDATOR.pattern(), "The network address is not a valid IP address.");
	addWidget("netmask", Netmask, None, IP_VALIDATOR.pattern(), "The netmask is not a valid IP address.");
	addWidget("start", Start, NoMulti, IP_VALIDATOR.pattern(), "The start address is not a valid IP address.");
	addWidget("end", End, NoMulti, IP_VALIDATOR.pattern(), "The end address is not a valid IP address.");
	addWidget("domainname", DomainName, None, "^[A-Za-z][A-Za-z0-9.\\-]*$", "The domain name is not valid.");
	addWidget("gateway", Gateway, None, IP_VALIDATOR.pattern(), "The gateway must be a valid IP address.");
	addWidget("dnsservers", DnsServers, "listString", SIGNAL(changed()));
	DnsServers->setSeperator(QRegExp(",[ ]*"), ", ");
	DnsServers->setErrorText("You must specify a valid IP address.");
	DnsServers->setWeakValidator(IP_VALIDATOR);
	DnsServers->setStrongValidator(IP_VALIDATOR_STRONG);

	QObject::connect(manager, SIGNAL(entityRenamed(QString, QString)), this, SLOT(entityRenamed(QString, QString)));
}

DhcpSubnetEditWidget::~DhcpSubnetEditWidget()
{
}

void DhcpSubnetEditWidget::entityRenamed(QString oldid, QString newid)
{
	if (oldid != input()) return;

	QStringList outlist = outputs();
	for (int i = 0; i < outlist.size(); i++)
	{
		if (outlist[i] == input()) outlist[i] = newid;
	}

	setInput(newid);
	setOutputs(outlist);

	//setField("name", newid);
}

ReturnValue DhcpSubnetEditWidget::getData()
{
	QVariantMap fields;

	if (input() == "") return(ReturnValue(1, "Invalid Folder"));

	ReturnValue err = true;

	ReturnValue ret = _manager->values(input());
	if (ret.isError()) err = ret;
	else
	{
		RdsDhcpSubnet values = ret;

		ret = values.address();
		if (ret.isError()) err = ret;
		fields["network"] = ret.value<QHostAddress>().toString();

		ret = values.netmask();
		if (ret.isError()) err = ret;
		fields["netmask"] = ret.value<QHostAddress>().toString();

		ret = values.ranges();
		if (ret.isError()) err = ret;
		else
		{
			RdsDhcpSubnet::RangeList ranges = ret.value<RdsDhcpSubnet::RangeList>();
			if (ranges.size() > 0)
			{
				fields["start"] = ranges[0].first.toString();
				fields["end"] = ranges[0].second.toString();
			}
		}

		ret = values.option("domain-name");
		if (ret.isError()) err = ret;
		fields["domainname"] = ret.toString().replace("\"", "");

		ret = values.option("domain-name-servers");
		if (ret.isError()) err = ret;
		fields["dnsservers"] = ret;

		ret = values.option("routers");
		if (ret.isError()) err = ret;
		fields["gateway"] = ret;

		ret = values.name();
		if (ret.isError()) err = ret;
		fields["name"] = ret;
	}

	if (err.isError())
	{
		QMessageBox msg(QMessageBox::Warning, "Error", "There was an error reading settings: " + ret.errString(), QMessageBox::Ok);
		QAbstractButton *editbutton = msg.addButton("Edit Anyway", QMessageBox::RejectRole);
		msg.exec();
		if (msg.clickedButton() == editbutton)
		{
			if (QMessageBox::warning(this, "Warning", "Editing in this state may be dangerous. Continue?", QMessageBox::Yes | QMessageBox::No, QMessageBox::No) == QMessageBox::Yes)
			{
				return(fields);
			}
		}
		return(err);
	}

	return(fields);
}

#define SET_FIELD(field,function) if (name == field) \
	{ \
		ret = function; \
		if (ret.isError()) err = ret; \
	}

#define SET_FIELD(field,function) if (name == field) \
{ \
ret = function; \
if (ret.isError()) err = ret; \
	}
	#define SET_VALUE_STRING(field,key) \
	if(value.toString() == "") \
	{\
	ret = values.removeValue(key); \
	if (ret.isError()) err = ret; \
		}\
		else SET_FIELD(field,values.setValue(key, value.toString()));
		#define SET_VALUE_STRING_QUOTED(key,function) \
		SET_FIELD(field,values.setValue(key, "\"" + value.toString() + "\""));
	#define SET_VALUE_BOOL(key,function) \
	SET_FIELD(field,values.setValue(key, value.toBool()));
	#define SET_OPTION_STRING(field,key) \
	if(value.toString() == "") \
	{\
	ret = values.removeValue(key); \
	if (ret.isError()) err = ret; \
		}\
		else SET_FIELD(field,values.setOption(key, value.toString()));
		#define SET_OPTION_STRING_QUOTED(key,function) \
		SET_FIELD(field,values.setOption(key, "\"" + value.toString() + "\""));
	#define SET_OPTION_BOOL(key,function) \
	SET_FIELD(field,values.setOption(key, value.toBool()));

bool DhcpSubnetEditWidget::setData(QString id, QVariantMap fields)
{
	ReturnValue err = true;
	ReturnValue ret;

	ret = _manager->values(id);
	if (ret.isError()) err = ret;
	RdsDhcpSubnet values = ret;
	bool rangesaved = false;

	foreach(QString name, fields.keys())
	{
		QVariant value = fields[name];

		if (name == "network")
		{
			ret = values.setAddress(QHostAddress(value.toString()));
			if (ret.isError()) err = ret;
		}
		else if (name == "netmask")
		{
			ret = values.setNetmask(QHostAddress(value.toString()));
			if (ret.isError()) err = ret;
		}
		else if (((name == "start") || (name == "end")) && (!rangesaved))
		{
			RdsDhcpSubnet::RangeList list;
			list << RdsDhcpSubnet::Range(QHostAddress(Start->text()), QHostAddress(End->text()));

			ret = values.setRanges(list);
			if (ret.isError()) err = ret;

			rangesaved = true;
		}
		else if (name == "domainname")
		{
			value = "\"" + value.toString() + "\"";
		}

		SET_OPTION_STRING("domainname", "domain-name");
		SET_OPTION_STRING("gateway", "routers");
		SET_OPTION_STRING("dnsservers", "domain-name-servers");
		SET_VALUE_STRING("ip", "fixed-address");
		SET_VALUE_STRING("mac", "hardware ethernet");

		if (name == "name")
		{
			//qDebug() << "setting name" << value;
			ret = values.setName(value.toString());
			if (ret.isError())
			{
				err = ret;
			}
			else
			{
				QString name = input();
				name = name.left(name.lastIndexOf("/") + 2);
				name += value.toString();

				QStringList outlist = outputs();
				for (int i = 0; i < outlist.size(); i++)
				{
					if (outlist[i] == input()) outlist[i] = name;
				}

				setInput(name);
				setOutputs(outlist);

				manager()->renameEntity(id, name);
			}
		}
	}

	if (err.isError())
	{
		qWarning() << "Failed to save:" << id << err.errString();
		return(false);
	}
	else return(true);
}

void DhcpSubnetEditWidget::entityUpdated(QString id)
{
	if (id != input()) return;

	if (unsavedChanges())
	{
		if (QMessageBox::question(this, "Group Changed",
		                          "Another user has made changes to the item you are editing. Would you like to overwrite your changes?",
		                          QMessageBox::Yes | QMessageBox::No, QMessageBox::No) == QMessageBox::Yes)
		{
			RdsEntityWidget::entityUpdated(id);
		}
	}
	else
	{
		RdsEntityWidget::entityUpdated(id);
	}
}

bool DhcpSubnetEditWidget::endSetData()
{
	ReturnValue ret = _manager->save();
	if (ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to save changes: " + ret.errString());
		return(false);
	}

	RdsDaemonManager mgr;
	ret = mgr.init();

	if (ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to get RdsDaemonManager service: " + ret.errString());
		return(false);
	}

	ret = mgr.restartService("Dhcp");

	if (ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to restart the DHCP server: " + ret.errString());
		return(false);
	}

	return(true);
}

bool DhcpSubnetEditWidget::validate()
{
	if(!QRegExp(IP_VALIDATOR_STRONG).exactMatch(Network->text()))
	{
		QMessageBox::critical(this, "Error", "The network address must be a valid IP address");
		return(false);
	}
	
	if(!QRegExp(IP_VALIDATOR_STRONG).exactMatch(Netmask->text()))
	{
		QMessageBox::critical(this, "Error", "The netmask must be a valid IP address");
		return(false);
	}
	
	if(!QRegExp(IP_VALIDATOR_STRONG).exactMatch(Start->text()))
	{
		QMessageBox::critical(this, "Error", "The start address must be a valid IP address");
		return(false);
	}
	
	if(!QRegExp(IP_VALIDATOR_STRONG).exactMatch(End->text()))
	{
		QMessageBox::critical(this, "Error", "The end address must be a valid IP address");
		return(false);
	}
	
	if((Gateway->text() != "") && !QRegExp(IP_VALIDATOR_STRONG).exactMatch(Gateway->text()))
	{
		QMessageBox::critical(this, "Error", "The gateway must be a valid IP address");
		return(false);
	}
	
	QHostAddress network(Network->text());
	QHostAddress netmask(Netmask->text());
	QHostAddress start(Start->text());
	QHostAddress end(End->text());
	
	if((network.toIPv4Address() & netmask.toIPv4Address()) != network.toIPv4Address())
	{
		QMessageBox::critical(this, "Error", "The network address and netmask are not consistent. Please make sure you enter a valid network address for the netmask.");
		return(false);
	}
	
	if((start.toIPv4Address() & netmask.toIPv4Address()) != network.toIPv4Address())
	{
		QMessageBox::critical(this, "Error", "The start address is not in the range defined by the network address and netmask.");
		return(false);
	}
	
	if((end.toIPv4Address() & netmask.toIPv4Address()) != network.toIPv4Address())
	{
		QMessageBox::critical(this, "Error", "The end address is not in the range defined by the network address and netmask.");
		return(false);
	}
	
	if(start.toIPv4Address() > end.toIPv4Address())
	{
		QMessageBox::critical(this, "Error", "The start address must be lower than the end address.");
		return(false);
	}
	
	ReturnValue ret = _manager->checkRange(input(), start, end);
	if(ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to check start and end addresses: " + ret.errString());
		return(false);
	}
	
	if(!ret.toBool())
	{
		QMessageBox::critical(this, "Error", "The range of this subnet overlaps the range of another subnet. Please modify the start and end addresses and try again.");
		return(false);
	}
	
	return(true);
}

