/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU General Public License as published by   *
 *  the Free Software Foundation; either version 2 of the License, or (at  *
 *  your option) any later version.                                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  General Public License for more details.                               *
 *                                                                         *
 *  You should have received a copy of the GNU General Public License      *
 *  along with this program; if not, write to the                          *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/
#include "rdsdnsmodel.h"
#include <QMessageBox>
#include <QDebug>
#include <RdsDnsManager>
#include <RdsDnsZone>
#include <RdsDnsRecord>
#include <RdsDaemonManager>
#include "config.h"

QString typeToString(RdsDnsZone::RecordType type);

RdsDnsModel::RdsDnsModel(RdsDnsManager *manager, QObject *parent)
		: RdsEntityModel(manager, 2, parent)
{
	_manager = manager;
	setCachedPixmap("manager", QPixmap(findRdsIcon("./icons/16x16/server.png")));
	setCachedPixmap("zone", QPixmap(findRdsIcon("./icons/16x16/network.png")));
	setCachedPixmap("slavezone", QPixmap(findRdsIcon("./icons/16x16/network.png")));
	setCachedPixmap("origin", QPixmap(findRdsIcon("./icons/16x16/folder.png")));
	setCachedPixmap("record", QPixmap(findRdsIcon("./icons/16x16/template_source.png")));
}

RdsDnsModel::~RdsDnsModel()
{

}

QVariant RdsDnsModel::userData(const QModelIndex &index, int role, RdsEntityModel::Cache *cache) const
{
	Q_UNUSED(index);


	if ((role == Qt::DecorationRole) && (index.column() == 0))
	{
		return(cachedPixmap(cache->type()));
	}
	if ((role == Qt::DisplayRole) && (index.column() == 1))
	{
		return(cache->metadata()["description"]);
	}

	return(QVariant());
}

QVariant RdsDnsModel::userHeaderData(int section, int role) const
{
	if (role == Qt::DisplayRole)
	{
		if (section == 0)
		{
			return("Shares");
		}
		else 	if (section == 1)
		{
			return("Description");
		}
	}

	return(QVariant());
}

Qt::ItemFlags RdsDnsModel::flags(const QModelIndex &index, Cache *cache) const
{
	Q_UNUSED(index);
	Q_UNUSED(cache);
	if ((index.column() == 0) && (cache->type() != "manager")) return(Qt::ItemIsEditable);
	else return(0);
}

bool RdsDnsModel::setData(const QModelIndex &index, const QVariant &value, int role, Cache *cache)
{
	Q_UNUSED(index);
	Q_UNUSED(role);
	if (cache->name() == value.toString()) return(false); //if the new name is the same, we didn't change anything
	if (value.toString() == "") return(false);
	if (cache->type() == "manager") return(false);
	if (cache->type() == "root") return(false);

	QString zone = getZoneFromId(cache->id());
	QString origin = getOriginFromId(cache->id());

	ReturnValue ret = _manager->zone(zone);
	if (ret.isError())
	{
		QMessageBox::critical(NULL, "Error", "Failed to get zone " + zone + ": " + ret.errString());
		return(false);
	}

	RdsDnsZone z = ret;

	if (cache->type() == "zone")
	{
		if (!QRegExp("[0-9A-Za-z._-]*").exactMatch(value.toString()))
		{
			QMessageBox::critical(NULL, "Error", "The name must contain only numbers, letters, periods (.), underscores (_), and hyphen (-).");
			return(false);
		}

		QString newdomain = value.toString() + ".";

		z.setDomainName(newdomain);
		if (ret.isError())
		{
			QMessageBox::critical(NULL, "Error", "Failed to rename zone " + zone + ": " + ret.errString());
			return(false);
		}

		ret = z.save();
		ReturnValue ret = _manager->zone(zone);
		if (ret.isError())
		{
			QMessageBox::critical(NULL, "Error", "Failed to save zone " + zone + ": " + ret.errString());
			return(false);
		}

		ret = _manager->removeZone(zone);
		if (ret.isError())
		{
			QMessageBox::critical(NULL, "Error", "Failed to remove zone " + zone + ": " + ret.errString());
			return(false);
		}

		ret = _manager->save();
		if (ret.isError())
		{
			QMessageBox::critical(NULL, "Error", "Failed to save settings:" + ret.errString());
			return(false);
		}

		_manager->renameEntity(zone, newdomain);

		save();
		return(true);
	}
	else if (cache->type() == "origin")
	{
		if (!QRegExp("[0-9A-Za-z_-]*").exactMatch(value.toString()))
		{
			QMessageBox::critical(NULL, "Error", "The name must contain only numbers, letters, underscores (_), and the hyphen (_).");
			return(false);
		}

		QString newid = cache->id().left(cache->id().lastIndexOf(":")) + ":" +  value.toString();
		QString neworigin = origin.mid(origin.indexOf(".")).prepend(value.toString());

		ret = z.renameOrigin(origin, neworigin);
		if (ret.isError())
		{
			QMessageBox::critical(NULL, "Error", "Failed to move origin " + origin + ": " + ret.errString());
			return(false);
		}

		ret = z.save();
		if (ret.isError())
		{
			QMessageBox::critical(NULL, "Error", "Failed to save zone " + zone + ": " + ret.errString());
			return(false);
		}

		_manager->renameEntity(cache->id(), newid);
		save();
		return(true);
	}
	else if (cache->type() == "record")
	{
		if (value.toString() != zone && !QRegExp("[0-9A-Za-z_-]*").exactMatch(value.toString()))
		{
			QMessageBox::critical(NULL, "Error", "The name must contain only numbers, letters, underscores (_), and the hyphen (_).");
			return(false);
		}

		RdsDnsRecord record(cache->id());
		ret = z.records(record.type(), record.key(), record.origin());
		if (ret.isError())
		{
			QMessageBox::critical(NULL, "Error", "Failed to get record " + record.key() + ": " + ret.errString());
			return(false);
		}
		RdsDnsRecordList records;
		records = ret.value<RdsDnsRecordList>();
		if (!records.count())
		{
			QMessageBox::critical(NULL, "Error", "Failed to get record " + record.key() + ": No records found, but no errors reported");
			return(false);
		}

		QStringList values;
		foreach(RdsDnsRecord record, records)
		{
			z.removeRecord(record);
			values << record.value();
		}

		record.setKey(value.toString());

		ret = z.addRecord(record.type(), record.key(), values, record.origin(), record.ttl());
		if (ret.isError())
		{
			QMessageBox::critical(NULL, "Error", "Failed to add record " + record.key() + ": " + ret.errString());
			return(false);
		}

		ret = z.save();
		if (ret.isError())
		{
			QMessageBox::critical(NULL, "Error", "Failed to save zone " + zone + ": " + ret.errString());
			return(false);
		}

		_manager->renameEntity(cache->id(), record.id(zone));
		save();
		return(true);
	}

	return(false);
}

void RdsDnsModel::move(Cache *entity, Cache *parent)
{
	if (entity->type() == "manager") return;
	if (entity->type() == "zone") return;
	if (parent->type() == "manager") return;
	if (parent->type() == "root") return;

	if (getZoneFromId(entity->id()) != getZoneFromId(parent->id()))
	{
		QMessageBox::warning(NULL, "Error", "You may not drag objects between domains.");
		return;
	}

	QString zone = getZoneFromId(parent->id());

	ReturnValue ret = _manager->zone(zone);
	if (ret.isError())
	{
		QMessageBox::critical(NULL, "Error", "Failed to get zone " + zone + ": " + ret.errString());
		return;
	}

	RdsDnsZone z = ret;

	if (entity->type() == "origin")
	{
		QString oldorigin = getOriginFromId(entity->id());
		QString neworigin = getOriginFromId(parent->id());

		ret = z.renameOrigin(oldorigin, entity->name() + "."  + neworigin);
		if (ret.isError())
		{
			QMessageBox::critical(NULL, "Error", "Failed to move origin " + oldorigin + ": " + ret.errString());
			return;
		}

		ret = z.save();
		if (ret.isError())
		{
			QMessageBox::critical(NULL, "Error", "Failed to save zone " + zone + ": " + ret.errString());
			return;
		}

		int index = parent->id().indexOf("#");
		QString id = parent->id();
		if (index > 0) id = id.left(index);

		id += ":" + entity->name();

		save();

		_manager->removeEntity(entity->id());
		_manager->addEntity(id);
	}
	else if (entity->type() == "record")
	{
		QString neworigin = getOriginFromId(parent->id());

		RdsDnsRecord record(entity->id());
		ret = z.records(record.type(), record.key(), record.origin());
		if (ret.isError())
		{
			QMessageBox::critical(NULL, "Error", "Failed to get record " + record.key() + ": " + ret.errString());
			return;
		}
		RdsDnsRecordList records;
		records = ret.value<RdsDnsRecordList>();
		if (!records.count())
		{
			QMessageBox::critical(NULL, "Error", "Failed to get record " + record.key() + ": No records found, but no errors reported");
			return;
		}

		if (record.key() == zone)
		{
			QMessageBox::critical(NULL, "Error", "You may not drag domain records between origins.");
			return;
		}

		QStringList values;
		foreach(RdsDnsRecord record, records)
		{
			z.removeRecord(record);
			values << record.value();
		}

		record.setOrigin(neworigin);

		ret = z.addRecord(record.type(), record.key(), values, record.origin(), record.ttl());
		if (ret.isError())
		{
			QMessageBox::critical(NULL, "Error", "Failed to add record " + record.key() + ": " + ret.errString());
			return;
		}

		ret = z.save();
		if (ret.isError())
		{
			QMessageBox::critical(NULL, "Error", "Failed to save zone " + zone + ": " + ret.errString());
			return;
		}

		_manager->removeEntity(entity->id());
		_manager->addEntity(record.id(zone));
		save();
	}
}

QString RdsDnsModel::getZoneFromId(const QString& id)
{
	QString zone = id;

	zone = zone.left(zone.indexOf('#'));
	zone = zone.left(zone.indexOf(':'));

	return(zone);
}

QString RdsDnsModel::getOriginFromId(const QString& id)
{
	QString origin = id;
	origin = origin.left(origin.indexOf('#'));
	QStringList originPath = origin.split(':', QString::SkipEmptyParts);
	origin = QString();
	foreach(QString path, originPath)
	{
		origin.prepend(path);
		origin.prepend('.');
	}
	origin = origin.mid(1);

	return(origin);
}

bool RdsDnsModel::save()
{
	RdsDaemonManager mgr;
	ReturnValue ret = mgr.init();

	if (ret.isError())
	{
		QMessageBox::critical(NULL, "Error", "Failed to get RdsDaemonManager service: " + ret.errString());
		return(false);
	}

	ret = mgr.reloadService("Dns");

	if (ret.isError())
	{
		QMessageBox::critical(NULL, "Error", "Failed to restart the DNS server: " + ret.errString());
		return(false);
	}

	return(true);
}

bool RdsDnsModel::validateRecord(const QStringList &value, int type)
{
	if (!value.count())
		return false;

	foreach(const QString& val, value)
	{
		if (!validateRecord(val, type))
			return false;
	}
	return true;
}

bool RdsDnsModel::validateRecord(const QString &value, int type)
{
	ReturnValue ret;
	QRegExp exp(".");
	QString error = "Must enter a value for the record";

	QString ip = "((25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\\.){3}(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)";
	QString hostname = "[a-zA-Z0-9-_.]+";

	// Use fallthrough case statements here!
	switch (type)
	{
			// IP only
		case RdsDnsZone::A:
			error = RdsDnsZone::typeToString((RdsDnsZone::RecordType)type) + " records must be an IP address";
			exp = QRegExp("^" + ip + "$");
			break;

			// IP or hostname
		case RdsDnsZone::SRV:
		case RdsDnsZone::NS:
			error = RdsDnsZone::typeToString((RdsDnsZone::RecordType)type) + " records must be an IP address or hostname";
			exp = QRegExp("^(" + ip + "|" + hostname + ")$");
			break;

			// Hostname only
		case RdsDnsZone::CNAME:
			error = RdsDnsZone::typeToString((RdsDnsZone::RecordType)type) + " records must be a hostname";
			exp = QRegExp("^" + hostname + "$");
			break;

			// Everything else?
		case RdsDnsZone::AAAA:
		case RdsDnsZone::AFSDB:
		case RdsDnsZone::CERT:
		case RdsDnsZone::DHCID:
		case RdsDnsZone::DLV:
		case RdsDnsZone::DNAME:
		case RdsDnsZone::DNSKEY:
		case RdsDnsZone::DS:
		case RdsDnsZone::HIP:
		case RdsDnsZone::IPSECKEY:
		case RdsDnsZone::KEY:
		case RdsDnsZone::LOC:
		case RdsDnsZone::MX:
		case RdsDnsZone::NAPTR:
		case RdsDnsZone::NSEC:
		case RdsDnsZone::NSEC3:
		case RdsDnsZone::NSEC3PARAM:
		case RdsDnsZone::PTR:
		case RdsDnsZone::RRSIG:
		case RdsDnsZone::SIG:
		case RdsDnsZone::SOA:
		case RdsDnsZone::SPF:
		case RdsDnsZone::SSHFP:
		case RdsDnsZone::TA:
		case RdsDnsZone::TXT:
		default:
			error = RdsDnsZone::typeToString((RdsDnsZone::RecordType)type) + " records must have a value";
			exp = QRegExp("^.+$");
			break;
	}

	if (!exp.exactMatch(value))
	{
		QMessageBox::critical(NULL, "Invalid Record", error);
		return false;
	}

	return true;
}
