/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU General Public License as published by   *
 *  the Free Software Foundation; either version 2 of the License, or (at  *
 *  your option) any later version.                                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  General Public License for more details.                               *
 *                                                                         *
 *  You should have received a copy of the GNU General Public License      *
 *  along with this program; if not, write to the                          *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/


#include "serverstatuswidget.h"
#include <RdsClient>
#include <QTimer>
#include <QDebug>
#include <QMessageBox>
#include <RdsDaemonManager>

ServerStatusWidget::ServerStatusWidget(QWidget* parent, Qt::WFlags fl)
		: QWidget(parent, fl), Ui::ServerStatusWidget()
{
	setupUi(this);
	QTimer *timer = new QTimer(this);
	QObject::connect(timer, SIGNAL(timeout()), this, SLOT(updateStatus()));
	timer->start(10000);

	updateStatus();
}

ServerStatusWidget::~ServerStatusWidget()
{
}

void ServerStatusWidget::updateStatus()
{
	ReturnValue ret = rdsClient()->status();
	if (ret.isError())
	{
		qWarning() << "Failed to get status information:" << ret;
		return;
	}

	QVariantMap status = ret.toMap();

	HostNameLabel->setText(status.value("hostname").toString());
	OsLabel->setText(status.value("os").toString());
	KernelLabel->setText(status.value("kernel").toString());
	DomainLabel->setText(status.value("realm").toString());

	TimeLabel->setText(status.value("datetime").toDateTime().time().toString());
	DateLabel->setText(status.value("datetime").toDateTime().date().toString());
	TimeZoneLabel->setText(status.value("timezone").toString());

	int uptime = status.value("uptime").toInt();
	int days = uptime / 86400;
	uptime = uptime % 86400;
	int hours = uptime / 3600;
	uptime = uptime % 3600;
	int minutes = uptime / 60;
	int seconds = uptime % 60;


	UpTimeLabel->setText(QString("%1 Days, %2 Hours, %3 Minutes, %4, Seconds").arg(days).arg(hours).arg(minutes).arg(seconds));

	InterfaceLabel->setText(status.value("interface").toString());
	MacLabel->setText(status.value("mac").toString());
	IpAddressLabel->setText(status.value("ip").toString());
	NetmaskLabel->setText(status.value("netmask").toString());
	GatewayLabel->setText(status.value("gateway").toString());
	QStringList dns = status.value("dns").toStringList();
	if (dns.size() >= 1) Dns1Label->setText(dns[0]);
	if (dns.size() >= 2) Dns2Label->setText(dns[1]);

	CpuTypeLabel->setText(status.value("cputype").toString().simplified());
	CpuCoresLabel->setText(status.value("cpucores").toString());
	CpuSpeedLabel->setText(status.value("cpuspeed").toString() + " Mhz");
	MemoryLabel->setText(status.value("mem").toString() + " MB");
	FreeMemoryLabel->setText(status.value("freemem").toString() + " MB");
	SwapLabel->setText(status.value("swap").toString() + " MB");
}

void ServerStatusWidget::on_ShutDownButton_clicked()
{
	if (QMessageBox::question(this, "Shut Down?", "Are you sure you want to shut down?", QMessageBox::Yes | QMessageBox::No, QMessageBox::No) == QMessageBox::Yes)
	{
		rdsClient()->shutdown();
	}
}

void ServerStatusWidget::on_RebootButton_clicked()
{
	if (QMessageBox::question(this, "Reboot?", "Are you sure you want to reboot?", QMessageBox::Yes | QMessageBox::No, QMessageBox::No) == QMessageBox::Yes)
	{
		rdsClient()->reboot();
	}
}

void ServerStatusWidget::on_RestartSambaButton_clicked()
{
	if (QMessageBox::question(this, "Restart Samba?", "Are you sure you want to restart Samba?", QMessageBox::Yes | QMessageBox::No, QMessageBox::No) == QMessageBox::Yes)
	{
		RdsDaemonManager mgr;
		ReturnValue ret = mgr.init();
		if (ret.isError())
		{
			QMessageBox::critical(this, "Error", "Failed to get daemon manager service: " + ret.errString());
			return;
		}
		
		ret = mgr.restartService("Samba");
		if (ret.isError())
		{
			QMessageBox::critical(this, "Error", "Failed to restart Samba: " + ret.errString());
			return;
		}
	}
}

void ServerStatusWidget::on_RestartDnsButton_clicked()
{
	if (QMessageBox::question(this, "Restart Dns?", "Are you sure you want to restart the DNS server?", QMessageBox::Yes | QMessageBox::No, QMessageBox::No) == QMessageBox::Yes)
	{
		RdsDaemonManager mgr;
		ReturnValue ret = mgr.init();
		if (ret.isError())
		{
			QMessageBox::critical(this, "Error", "Failed to get daemon manager service: " + ret.errString());
			return;
		}
		
		ret = mgr.restartService("Dns");
		if (ret.isError())
		{
			QMessageBox::critical(this, "Error", "Failed to restart the DNS server: " + ret.errString());
			return;
		}
	}
}

void ServerStatusWidget::on_RestartDhcpButton_clicked()
{
	if (QMessageBox::question(this, "Restart DHCP?", "Are you sure you want to restart the DHCP server?", QMessageBox::Yes | QMessageBox::No, QMessageBox::No) == QMessageBox::Yes)
	{
		RdsDaemonManager mgr;
		ReturnValue ret = mgr.init();
		if (ret.isError())
		{
			QMessageBox::critical(this, "Error", "Failed to get daemon manager service: " + ret.errString());
			return;
		}
		
		ret = mgr.restartService("Dhcp");
		if (ret.isError())
		{
			QMessageBox::critical(this, "Error", "Failed to restart the DHCP server: " + ret.errString());
			return;
		}
	}
}
