/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU General Public License as published by   *
 *  the Free Software Foundation; either version 2 of the License, or (at  *
 *  your option) any later version.                                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  General Public License for more details.                               *
 *                                                                         *
 *  You should have received a copy of the GNU General Public License      *
 *  along with this program; if not, write to the                          *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/


#include "usermanagerwidget.h"
#include <QMessageBox>
#include <RdsClient>
#include <QDebug>
#include <QInputDialog>
#include <RdsOrganizationalUnit>
#include <QMenu>
#include <QSettings>
#include <QShortcut>
#include <RdsUtils>
#include "usereditwidget.h"
#include "newuserdialog.h"
#include "oueditwidget.h"
#include "groupeditwidget.h"
#include "changepassworddialog.h"
#include "config.h"

QStringList __systemids;
QStringList __protectedids;

UserManagerWidget::UserManagerWidget(QWidget* parent, Qt::WFlags fl)
		: QWidget(parent, fl), Ui::UserManagerWidget()
{
	setupUi(this);
	pushButton_4->setIcon(QPixmap(findRdsIcon("icons/16x16/clear_left.png")));
	AddButton->setIcon(QPixmap(findRdsIcon("icons/32x32/add.png")));
	DeleteButton->setIcon(QPixmap(findRdsIcon("icons/32x32/remove.png")));
	ReloadButton->setIcon(QPixmap(findRdsIcon("icons/32x32/reload.png")));
	_edit = NULL;

	addSystemId(571);
	addSystemId("CA");
	addSystemId(572);
	addSystemId("DC");
	addSystemId("DD");
	addSystemId("EA");
	addSystemId(498);
	addSystemId("PA");
	addSystemId("LK");
	addSystemId("RS");
	addSystemId("RD");
	addSystemId("SA");
	addSystemId("RO");
	addProtectedId("DA");
	addProtectedId(513);
	addProtectedId("DG");
	addSystemIdByName("BuiltIn");
	addSystemIdByName("DnsUpdateProxy");
	addSystemIdByName("DnsAdmins");
	addSystemIdByName("dns-*");
	addSystemIdByName("rdsadmin*");
	addProtectedIdByName("Administrator");
	addProtectedIdByName("Guest");

	ReturnValue ret = rdsClient()->getService("UserGroupComputerManager");
	if (ret.isError())
	{
		QMessageBox::critical(this, "Failed to get user manager", ret.errString());
		return;
	}

	_manager = new RdsEntityManager(this);
	*_manager = ret;

	_model = new RdsUserGroupComputerModel(_manager, this);
	_sort = new RdsUserGroupComputerSortModel(_model);
	_sort->showUsers(true);
	_sort->showGroups(true);
	_sort->showComputers(false);
	_sort->setSourceModel(_model);
	UserView->setModel(_sort);
	UserView->sortByColumn(0, Qt::AscendingOrder);
	_selection = UserView->selectionModel();
	_model->setupSelection(_sort, UserView);
	UserView->header()->resizeSection(0, 200);

	QObject::connect(_model, SIGNAL(inputChanged(QString, QString)), this, SLOT(inputChanged(QString, QString)));
	QObject::connect(_model, SIGNAL(outputsChanged(QStringList)), this, SLOT(outputsChanged(QStringList)));
	QObject::connect(FilterEntry, SIGNAL(textChanged(QString)), _sort, SLOT(setFilterRegExp(QString)));

	UserEditWidget *w = new UserEditWidget(_manager, EditFrame);
	EditLayout->addWidget(w);
	_edit = w;
	w->setInput("");

	//Actions and Menu stuff
	UserView->setContextMenuPolicy(Qt::CustomContextMenu);
	QObject::connect(UserView, SIGNAL(customContextMenuRequested(const QPoint&)), this, SLOT(contextMenu(const QPoint&)));

	_newuseraction = new QAction("New User", this);
	QObject::connect(_newuseraction, SIGNAL(triggered()), this, SLOT(AddUser()));
	_newgroupaction = new QAction("New Group", this);
	QObject::connect(_newgroupaction, SIGNAL(triggered()), this, SLOT(AddGroup()));
	_newouaction = new QAction("New OU", this);
	QObject::connect(_newouaction, SIGNAL(triggered()), this, SLOT(AddOu()));
	_removeaction = new QAction("Delete", this);
	QObject::connect(_removeaction, SIGNAL(triggered()), this, SLOT(on_DeleteButton_clicked()));
	_setpasswordaction = new QAction("Change Password", this);
	QObject::connect(_setpasswordaction, SIGNAL(triggered()), this, SLOT(setPassword()));
	_showsystemaction = new QAction("Show System Objects", this);
	QObject::connect(_showsystemaction, SIGNAL(toggled(bool)), this, SLOT(showSystemObjects(bool)));
	_showsystemaction->setCheckable(true);
	_showusersaction = new QAction("Show Users", this);
	QObject::connect(_showusersaction, SIGNAL(toggled(bool)), this, SLOT(showUsers(bool)));
	_showusersaction->setCheckable(true);
	_showgroupsaction = new QAction("Show Groups", this);
	QObject::connect(_showgroupsaction, SIGNAL(toggled(bool)), this, SLOT(showGroups(bool)));
	_showgroupsaction->setCheckable(true);
	_cloneuseraction = new QAction("Clone User", this);
	QObject::connect(_cloneuseraction, SIGNAL(triggered(bool)), this, SLOT(cloneUser()));

	QSettings settings;
	if (settings.value("Users/ShowSystemsObjects").toBool())
	{
		_showsystemaction->setChecked(true);
		_sort->setHiddenEntities(QStringList());
	}
	else
	{
		_showsystemaction->setChecked(false);
		_sort->setHiddenEntities(__systemids);
	}

	if (settings.value("Users/HideUsers").toBool())
	{
		_showusersaction->setChecked(false);
		_sort->showUsers(false);
	}
	else
	{
		_showusersaction->setChecked(true);
		_sort->showUsers(true);
	}

	if (settings.value("Users/HideGroups").toBool())
	{
		_showgroupsaction->setChecked(false);
		_sort->showGroups(false);
	}
	else
	{
		_showgroupsaction->setChecked(true);
		_sort->showGroups(true);
	}

	QMenu *addmenu = new QMenu(this);
	addmenu->addAction(_newuseraction);
	addmenu->addAction(_newgroupaction);
	addmenu->addAction(_newouaction);
	AddButton->setMenu(addmenu);
	
	_removeaction->setShortcut(QKeySequence("Del"));
	new QShortcut(QKeySequence("Del"), this, SLOT(on_DeleteButton_clicked()));
}

UserManagerWidget::~UserManagerWidget()
{
}


void UserManagerWidget::inputChanged(QString id, QString type)
{
	_input = id;
	//qDebug() << "Input Changed:" << id << type;
	if (_edit != NULL)
	{
		if (_edit->unsavedChanges() && _edit->isEnabled())
		{
			if (QMessageBox::question(this, "Unsaved Changes", "The User you were editing has unsaved changes. Would you like to apply them?", QMessageBox::Apply | QMessageBox::Discard, QMessageBox::Discard) == QMessageBox::Apply)
			{
				_edit->apply();
			}
		}

		_edit->hide();
		_edit->deleteLater();
		_edit = NULL;
	}

	if (type == "user")
	{
		UserEditWidget *w = new UserEditWidget(_manager, EditFrame);
		EditLayout->addWidget(w);
		_edit = w;
		w->setInput(id);
	}
	else if (type == "group")
	{
		GroupEditWidget *w = new GroupEditWidget(_manager, EditFrame);
		EditLayout->addWidget(w);
		_edit = w;
		w->setInput(id);
	}
	else if (type == "ou")
	{
		OuEditWidget *w = new OuEditWidget(_manager, EditFrame);
		EditLayout->addWidget(w);
		_edit = w;
		w->setInput(id);
	}
	else
	{
		UserEditWidget *w = new UserEditWidget(_manager, EditFrame);
		EditLayout->addWidget(w);
		_edit = w;
		w->setInput("");
	}
}

void UserManagerWidget::outputsChanged(QStringList outputs)
{
	if (outputs.size() > 1)
		_setpasswordaction->setText("Change Passwords");
	else
		_setpasswordaction->setText("Change Password");
	//qDebug() << "Output Changed" << outputs;
	if (_edit != NULL) _edit->setOutputs(outputs);
}

void UserManagerWidget::AddUser()
{
	QString ou = "";
	RdsEntityModel::Cache *cache = _model->getCache(_input);
	if (cache != NULL)
	{
		if (cache->type() == "ou") ou = _input;
		else if ((cache->parent() != NULL) && (cache->parent()->type() == "ou")) ou = cache->parent()->id();
	}

	NewUserDialog dialog(ou, false, this);
	dialog.exec();
}

void UserManagerWidget::cloneUser()
{
	NewUserDialog dialog(_input, true,  this);
	dialog.exec();
}


void UserManagerWidget::AddGroup()
{
	QString ou = "";
	RdsEntityModel::Cache *cache = _model->getCache(_input);
	if (cache != NULL)
	{
		if (cache->type() == "ou") ou = _input;
		else if ((cache->parent() != NULL) && (cache->parent()->type() == "ou")) ou = cache->parent()->id();
	}

	bool ok = false;
	QString name = QInputDialog::getText(this, "Add Group", "Please enter the name for the new group.", QLineEdit::Normal, "", &ok);

	if (!ok) return;
	if (name == "")
	{
		QMessageBox::critical(this, "Error", "You must specify a name.");
		return;
	}

	if (!name.contains(QRegExp("^[0-9a-zA-Z][^+\\;,=]*$")))
	{
		QMessageBox::critical(NULL, "Error", "The name must not contain any of these characters (+ \\ ; , =).");
		return;
	}

	RdsOrganizationalUnit o(ou);
	ReturnValue ret = o.createGroup(name);

	if (ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to create group: " + ret.errString());
		return;
	}

	_manager->addEntity(ret.toString());
}

void UserManagerWidget::AddOu()
{
	QString ou = "";
	RdsEntityModel::Cache *cache = _model->getCache(_input);
	if (cache != NULL)
	{
		if (cache->type() == "ou") ou = _input;
		else if ((cache->parent() != NULL) && (cache->parent()->type() == "ou")) ou = cache->parent()->id();
	}

	bool ok = false;
	QString name = QInputDialog::getText(this, "Add OU", "Please enter the name for the new organizational unit.", QLineEdit::Normal, "", &ok);

	if (!ok) return;
	if (name == "")
	{
		QMessageBox::critical(this, "Error", "You must specify a name.");
		return;
	}

	if (!name.contains(QRegExp("^[0-9a-zA-Z][^+\\;,=]*$")))
	{
		QMessageBox::critical(NULL, "Error", "The name must not contain any of these characters (+ \\ ; , =).");
		return;
	}

	RdsOrganizationalUnit o(ou);
	ReturnValue ret = o.createOu(name);

	if (ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to create OU: " + ret.errString());
		return;
	}

	_manager->addEntity(ret.toString());
}

void UserManagerWidget::on_DeleteButton_clicked()
{
	QStringList outputs = _model->outputs();

	if (outputs.size() == 0)
	{
		QMessageBox::information(this, "Error", "You must select something to delete.");
		return;
	}
	else if (outputs.size() == 1)
	{
		RdsEntityModel::Cache *cache = _model->getCache(outputs[0]);
		if (cache == NULL) return;

		int result;

		if (cache->type() == "ou")
		{
			result = QMessageBox::question(this, "Delete OU", QString("Do you want to delete %1").arg(cache->name()),
			                               QMessageBox::Yes | QMessageBox::No, QMessageBox::No);
		}
		else if (cache->type() == "user")
		{
			result = QMessageBox::question(this, "Delete User", QString("Do you want to delete %1").arg(cache->name()),
			                               QMessageBox::Yes | QMessageBox::No, QMessageBox::No);
		}
		else if (cache->type() == "group")
		{
			result = QMessageBox::question(this, "Delete Group", QString("Do you want to delete %1").arg(cache->name()),
			                               QMessageBox::Yes | QMessageBox::No, QMessageBox::No);
		}
		else return;

		if (result != QMessageBox::Yes) return;
	}
	else
	{
		int result = QMessageBox::question(this, "Delete Items", QString("Do you want to delete %1 items?").arg(outputs.count()),
		                                   QMessageBox::Yes | QMessageBox::No, QMessageBox::No);

		if (result != QMessageBox::Yes)	return;
	}

	UserView->selectionModel()->clear();
	inputChanged("","");
	ReturnValue ret, err(true);

	//If we get here, we delete the outputs
	foreach(QString output, outputs)
	{
		RdsEntityModel::Cache *cache = _model->getCache(output);
		if (cache == NULL) continue;

		if ((__systemids.contains(output)) || (__protectedids.contains(output)))
		{
			err = ReturnValue(1, QString(cache->name() + " is a system object."));
			continue;
		}

		ret = rdsLdapSession()->remove(output, true);
		if (ret.isError())
			err = ret;
		else
			_manager->removeEntity(output);
	}

	if (err.isError())
	{
		QMessageBox::warning(this, "Warning", "Some items could not be deleted: " + err.errString());
	}
}

void UserManagerWidget::contextMenu(const QPoint &)
{
	QMenu menu;
	menu.addAction(_newuseraction);
	menu.addAction(_newgroupaction);
	menu.addAction(_newouaction);
	menu.addAction(_removeaction);
	menu.addSeparator();

	QString dn = _model->input();
	RdsEntityModel::Cache *cache = _model->getCache(dn);
	if (cache != NULL)
	{
		if (cache->type() == "user")
		{
			menu.addAction(_cloneuseraction);
			menu.addAction(_setpasswordaction);
		}
	}

	menu.addSeparator();
	menu.addAction(_showusersaction);
	menu.addAction(_showgroupsaction);
	menu.addAction(_showsystemaction);

	menu.exec(QCursor::pos());
}

void UserManagerWidget::setPassword()
{
	RdsEntityModel::Cache *cache = _model->getCache(_sort->mapToSource(_selection->currentIndex()));
	if (cache == NULL) return;

	//WHALE BONES
	ChangePasswordDialog dialog(_model->outputs(), this);
	dialog.exec();
}

void UserManagerWidget::showSystemObjects(bool show)
{
	QSettings settings;
	settings.setValue("Users/ShowSystemsObjects", show);
	if (show)
	{
		_sort->setHiddenEntities(QStringList());
	}
	else
	{
		_sort->setHiddenEntities(__systemids);
	}
}

void UserManagerWidget::showUsers(bool show)
{
	QSettings settings;
	settings.setValue("Users/HideUsers", !show);
	_sort->showUsers(show);
}

void UserManagerWidget::showGroups(bool show)
{
	QSettings settings;
	settings.setValue("Users/HideGroups", !show);
	_sort->showGroups(show);
}

void UserManagerWidget::addSystemId(RdsSid sid)
{
	ReturnValue ret = RdsUtils::getObjectBySid(sid);
	if (ret.isError())
	{
		qWarning() << "Cannot find object for SID:" << sid.toString();
		return;
	}

	__systemids << ret.toString();
}

void UserManagerWidget::addSystemId(int id)
{
	RdsSid sid = RdsSid::getDomainSid();
	sid.setRid(id);

	addSystemId(sid);
}

void UserManagerWidget::addSystemIdByName(QString name)
{
	ReturnValue ret = rdsLdapSession()->search(RdsUtils::baseDn(), QString("(cn=%1)").arg(name));
	if (ret.isError())
	{
		qWarning() << "Failed to find name:" << name << ret;
		exit(1);
	}

	LdapResults results = ret.value<LdapResults>();

	__systemids << results.keys();
}


void UserManagerWidget::addProtectedId(RdsSid sid)
{
	ReturnValue ret = RdsUtils::getObjectBySid(sid);
	if (ret.isError())
	{
		qWarning() << "Cannot find object for SID:" << sid.toString();
		return;
	}

	__protectedids << ret.toString();
}

void UserManagerWidget::addProtectedId(int id)
{
	RdsSid sid = RdsSid::getDomainSid();
	sid.setRid(id);

	addProtectedId(sid);
}

void UserManagerWidget::addProtectedIdByName(QString name)
{
	ReturnValue ret = rdsLdapSession()->search(RdsUtils::baseDn(), QString("(cn=%1)").arg(name));
	if (ret.isError())
	{
		qWarning() << "Failed to find name:" << name << ret;
		exit(1);
	}

	LdapResults results = ret.value<LdapResults>();

	__protectedids << results.keys();
}

void UserManagerWidget::on_ReloadButton_clicked()
{
	_model->reload();
}

