/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU General Public License as published by   *
 *  the Free Software Foundation; either version 2 of the License, or (at  *
 *  your option) any later version.                                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  General Public License for more details.                               *
 *                                                                         *
 *  You should have received a copy of the GNU General Public License      *
 *  along with this program; if not, write to the                          *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/
#include "rdscheckin.h"
#include <QDebug>
#include <QUrl>
#include <QNetworkReply>
#include <QMutexLocker>
#include <QFile>
#include <QTextStream>
#include <QUuid>
#include <RdsLdapSession>
#include <RdsUtils>
#include <RdsShareManager>
#include <RdsShare>
#include <RdsSettings>

#define INSTALL_ADDRESS "https://www.resara.com/licensing/api.php/install"
#define CHECKIN_ADDRESS "https://www.resara.com/licensing/api.php/checkin"
#define UPGRADE_ADDRESS "https://www.resara.com/licensing/api.php/upgrade"

RdsCheckin *RdsCheckin::_instance = NULL;


RdsCheckin::RdsCheckin(QObject *parent)
		: QObject(parent), _mutex(QMutex::Recursive)
{
	_manager = new QNetworkAccessManager(this);
	QObject::connect(_manager, SIGNAL(finished(QNetworkReply*)), this, SLOT(replyFinished(QNetworkReply*)));
}


RdsCheckin::~RdsCheckin()
{
}

void RdsCheckin::install()
{
	QMutexLocker locker(&_mutex);
	
	QUrl postdata;
	postdata.addQueryItem("type",getLicenseType());
	postdata.addQueryItem("uid",getUid());
	postdata.addQueryItem("version",getVersion());
	postdata.addQueryItem("arch",getArch());
	postdata.addQueryItem("cores",QString("%1").arg(getNumCores()));
	postdata.addQueryItem("ram",QString("%1").arg(getRam()));
	postdata.addQueryItem("distribution",getDistribution());
	postdata.addQueryItem("users",QString("%1").arg(getNumUsers()));
	postdata.addQueryItem("groups",QString("%1").arg(getNumGroups()));
	postdata.addQueryItem("computers",QString("%1").arg(getNumComputers()));
	postdata.addQueryItem("dcs",QString("%1").arg(getNumDcs()));
	postdata.addQueryItem("ous",QString("%1").arg(getNumOus()));
	postdata.addQueryItem("shares",QString("%1").arg(getNumShares()));
	postdata.addQueryItem("maps",QString("%1").arg(getNumMaps()));

	QNetworkReply* reply = _manager->post(QNetworkRequest(QUrl(INSTALL_ADDRESS)), postdata.encodedQuery());
	if (!reply)
	{
		qWarning() << "Failed to send the request to the server";
	}
	connect(reply, SIGNAL(sslErrors(const QList<QSslError> &)), reply, SLOT(ignoreSslErrors()));
}

void RdsCheckin::checkin()
{
	QMutexLocker locker(&_mutex);
	QUrl postdata;
	postdata.addQueryItem("type",getLicenseType());
	postdata.addQueryItem("uid",getUid());
	postdata.addQueryItem("version",getVersion());
	postdata.addQueryItem("arch",getArch());
	postdata.addQueryItem("cores",QString("%1").arg(getNumCores()));
	postdata.addQueryItem("ram",QString("%1").arg(getRam()));
	postdata.addQueryItem("distribution",getDistribution());
	postdata.addQueryItem("users",QString("%1").arg(getNumUsers()));
	postdata.addQueryItem("groups",QString("%1").arg(getNumGroups()));
	postdata.addQueryItem("computers",QString("%1").arg(getNumComputers()));
	postdata.addQueryItem("dcs",QString("%1").arg(getNumDcs()));
	postdata.addQueryItem("ous",QString("%1").arg(getNumOus()));
	postdata.addQueryItem("shares",QString("%1").arg(getNumShares()));
	postdata.addQueryItem("maps",QString("%1").arg(getNumMaps()));
	
	QNetworkReply* reply = _manager->post(QNetworkRequest(QUrl(CHECKIN_ADDRESS)), postdata.encodedQuery());
	if (!reply)
	{
		qWarning() << "Failed to send the request to the server";
	}
	connect(reply, SIGNAL(sslErrors(const QList<QSslError> &)), reply, SLOT(ignoreSslErrors()));
}

void RdsCheckin::checkUpgrade()
{
	QMutexLocker locker(&_mutex);
	QString uid = rdssettings()->value("uid").toString();
	if(uid == "")
	{
		QUrl postdata;
		postdata.addQueryItem("type",getLicenseType());
		postdata.addQueryItem("uid",getUid());
		postdata.addQueryItem("version",getVersion());
		postdata.addQueryItem("arch",getArch());
		postdata.addQueryItem("cores",QString("%1").arg(getNumCores()));
		postdata.addQueryItem("ram",QString("%1").arg(getRam()));
		postdata.addQueryItem("distribution",getDistribution());
		postdata.addQueryItem("users",QString("%1").arg(getNumUsers()));
		postdata.addQueryItem("groups",QString("%1").arg(getNumGroups()));
		postdata.addQueryItem("computers",QString("%1").arg(getNumComputers()));
		postdata.addQueryItem("dcs",QString("%1").arg(getNumDcs()));
		postdata.addQueryItem("ous",QString("%1").arg(getNumOus()));
		postdata.addQueryItem("shares",QString("%1").arg(getNumShares()));
		postdata.addQueryItem("maps",QString("%1").arg(getNumMaps()));
		
		QNetworkReply* reply = _manager->post(QNetworkRequest(QUrl(UPGRADE_ADDRESS)), postdata.encodedQuery());
		if (!reply)
		{
			qWarning() << "Failed to send the request to the server";
		}
		connect(reply, SIGNAL(sslErrors(const QList<QSslError> &)), reply, SLOT(ignoreSslErrors()));
	}
}


void RdsCheckin::replyFinished(QNetworkReply* reply)
{
	QMutexLocker locker(&_mutex);
	if (reply->error() != 0)
	{
		if (reply->error() == QNetworkReply::SslHandshakeFailedError)
		{
			qWarning() << "SSL handshake failed! Is your network blocking or intercepting HTTPS traffic?";
			emit(finished());
			return;
		}
		 qWarning() << "Failed checkin request:" << reply->readAll();
	}
	emit(finished());
}

RdsCheckin* RdsCheckin::instance()
{
	if(_instance == NULL) _instance = new RdsCheckin(NULL);
	return(_instance);
}

QString RdsCheckin::getLicenseType()
{
	return("community");
}

QString RdsCheckin::getUid()
{
	QString uid = rdssettings()->value("uid").toString();
	if(uid == "")
	{
		uid = QUuid::createUuid().toString();;
		rdssettings()->setValue("uid",uid);
	}
	
	return(uid);
}

QString RdsCheckin::getVersion()
{
	return(RDS_VERSION);
}

QString RdsCheckin::getArch()
{
	return(RdsUtils::runCommand("uname",QStringList() << "-m").toString().trimmed());
}

int RdsCheckin::getNumCores()
{
	int cores = 0;
	QFile file("/proc/cpuinfo");
	if(!file.open(QFile::ReadOnly))
	{
		qWarning() << "Failed to open /proc/cpuinfo";
		return(-1);
	}
	
	QTextStream stream(&file);
	QString line;
	
	while((line = stream.readLine()) != QString::Null())
	{
		if(line.toLower().startsWith("processor"))
		{
			cores++;
		}
	}
	
	return(cores);
}

int RdsCheckin::getRam()
{
	ReturnValue ret = RdsUtils::runCommand("free",QStringList() << "-m");
	if(ret.isError())
	{
		qWarning() << "Failed to get ram size:" << ret.errString();
		return(-1);
	}
	
	QString buf = ret.toString();
	QTextStream stream(&buf);
	
	stream.readLine(); //discar first line
	QStringList parts = stream.readLine().split(" ",QString::SkipEmptyParts);
	
	if(parts.size() < 2)
	{
		qWarning() << "Failed to get ram size:" << ret.errString();
		return(-1);
	}
	
	return(parts[1].toInt());
}

QString RdsCheckin::getDistribution()
{
	QFile file("/etc/lsb-release");
	if(!file.open(QFile::ReadOnly))
	{
		qWarning() << "Failed to open /etc/lsb-release";
		return("");
	}
	
	QTextStream stream(&file);
	QString line;
	
	while((line = stream.readLine()) != QString::Null())
	{
		if(line.toUpper().startsWith("DISTRIB_DESCRIPTION"))
		{
			int first = line.indexOf("\"") + 1;
			int last = line.lastIndexOf("\"");
			return(line.mid(first,last-first));
		}
	}
	
	return("");
}


int RdsCheckin::getNumUsers()
{
	ReturnValue ret = rdsLdapSession()->list(RdsUtils::baseDn(), "(&(objectClass=user)(!(objectClass=computer))(!(cn=rdsadmin*))(!(cn=dns*))(!(cn=krbtgt))(!(cn=guest))(!(cn=administrator)))");
	if(ret.isError())
	{
		qWarning() << "Failed to get number of users:" << ret.errString();
		return(-1);
	}
	else return(ret.toStringList().count());
}

int RdsCheckin::getNumGroups()
{
	ReturnValue ret = rdsLdapSession()->list(RdsUtils::baseDn(), "(&(objectClass=group)(!(isCriticalSystemObject=TRUE))(!(cn=DnsAdmins))(!(cn=DnsUpdateProxy)))");
	if(ret.isError())
	{
		qWarning() << "Failed to get number of groups:" << ret.errString();
		return(-1);
	}
	else return(ret.toStringList().count());
}

int RdsCheckin::getNumComputers()
{
	ReturnValue ret = rdsLdapSession()->list(RdsUtils::baseDn(), "(&(objectClass=computer)(!(primaryGroupId=516)))");
	if(ret.isError())
	{
		qWarning() << "Failed to get number of computers:" << ret.errString();
		return(-1);
	}
	else return(ret.toStringList().count());
}

int RdsCheckin::getNumDcs()
{
	ReturnValue ret = rdsLdapSession()->list(RdsUtils::baseDn(), "(&(objectClass=computer)(primaryGroupId=516))");
	if(ret.isError())
	{
		qWarning() << "Failed to get number of DCs:" << ret.errString();
		return(-1);
	}
	else return(ret.toStringList().count());
}

int RdsCheckin::getNumOus()
{
	ReturnValue ret = rdsLdapSession()->list(RdsUtils::baseDn(), "(&(objectClass=OrganizationalUnit)(!(OU=Domain Controllers)))");
	if(ret.isError())
	{
		qWarning() << "Failed to get number of OUs:" << ret.errString();
		return(-1);
	}
	else return(ret.toStringList().count());
}

int RdsCheckin::getNumShares()
{
	RdsShareManager mgr;
	ReturnValue ret = mgr.listShares();
	if(ret.isError())
	{
		qWarning() << "Failed to get number of shares:" << ret.errString();
		return(-1);
	}
	else return(ret.toStringList().count() - 2);
	
}

int RdsCheckin::getNumMaps()
{
	int maps = 0;
	RdsShareManager mgr;
	ReturnValue ret = mgr.listShares();
	if(ret.isError())
	{
		qWarning() << "Failed to get number of maps:" << ret.errString();
		return(-1);
	}
	
	foreach(QString sharename, ret.toStringList())
	{
		ret = mgr.share(sharename);
		if(ret.isError()) continue;
		
		RdsShare share = ret;
		if(share.mapString().toString().trimmed() != "") maps++;
	}
	
	return(maps);
}



