/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU Lesser General Public License as         *
 *  published by the Free Software Foundation; either version 2 of the     *
 *  License, or (at your option) any later version.                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  Lesser General Public License for more details.                        *
 *                                                                         *
 *  You should have received a copy of the GNU Lesser General Public       *
 *  License along with this program; if not, write to the                  *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/
#include "rdsldapobject.h"
#include "rdsldapobject_p.h"
#include <ReturnValue>
#include <QDebug>

RdsLdapObject::RdsLdapObject()
{
	QXT_INIT_PRIVATE(RdsLdapObject);
	qxt_d().dn = "";
	qxt_d().session = rdsLdapSession();
}

RdsLdapObject::RdsLdapObject(const QString &dn)
{
	QXT_INIT_PRIVATE(RdsLdapObject);
	qxt_d().dn = dn;
	qxt_d().session = rdsLdapSession();
}


RdsLdapObject::RdsLdapObject(const QString &dn, RdsLdapSession *session)
{
	qxt_d().dn = dn;
	qxt_d().session = session;
}

RdsLdapObject::RdsLdapObject(const RdsLdapObject &other)
{
	QXT_INIT_PRIVATE(RdsLdapObject);
	qxt_d().dn = other.qxt_d().dn;
	qxt_d().session = other.qxt_d().session;
}

RdsLdapObject::~RdsLdapObject()
{
}

RdsLdapObject& RdsLdapObject::operator=(const RdsLdapObject &other)
{
	qxt_d().dn = other.qxt_d().dn;
	qxt_d().session = other.qxt_d().session;
	return *this;
}

ReturnValue RdsLdapObject::search(const QString &filter, bool recursive) const
{
	return(qxt_d().session->search(qxt_d().dn,filter,recursive));
}

ReturnValue RdsLdapObject::read(const QStringList &attrs) const
{
	return(qxt_d().session->read(qxt_d().dn,attrs));
}

ReturnValue RdsLdapObject::readAttribute(const QString &attr) const
{
	ReturnValue ret = qxt_d().session->read(qxt_d().dn,QStringList() << attr);
	if(ret.isError()) return(ret);
	
	LdapResult result = ret.value<LdapResult>();
	
	if(!result.contains(attr.toLower()) || (result[attr.toLower()].count() == 0))
	{
		return(ReturnValue(1,attr + " attribute does not exist"));
	}
	
	return(QVariant::fromValue(result[attr.toLower()])); 
}

ReturnValue RdsLdapObject::list(const QString &filter, bool recursive) const
{
	return(qxt_d().session->list(qxt_d().dn,filter,recursive));
}

ReturnValue RdsLdapObject::add(const RdsLdapActions &actions)
{
	return(qxt_d().session->add(qxt_d().dn,actions));
}

ReturnValue RdsLdapObject::modify(const RdsLdapActions &actions)
{
	return(qxt_d().session->modify(qxt_d().dn,actions));
}

ReturnValue RdsLdapObject::remove(bool recursive)
{
	return(qxt_d().session->remove(qxt_d().dn, recursive));
}

QString RdsLdapObject::dn() const
{
	return(qxt_d().dn);
}

void RdsLdapObject::setDn(const QString &dn)
{
	qxt_d().dn = dn;
}

ReturnValue RdsLdapObject::rename(const QString &newname)
{
	ReturnValue ret = qxt_d().session->rename(dn(),newname);
	if(ret.isError()) return(ret);
	
	QString newdn = dn().replace(QRegExp("^[^,]*,"),newname + ",");
	setDn(newdn);
	return(newdn);
}

ReturnValue RdsLdapObject::move(const QString &newparent)
{
	ReturnValue ret = qxt_d().session->move(dn(),newparent);
	if(ret.isError()) return(ret);
	
	QString newdn = dn().replace(QRegExp(",.*$"),"," + newparent);
	setDn(newdn);
	return(newdn);
}

RdsLdapSession *RdsLdapObject::session() const
{
	return(qxt_d().session);
}

QDataStream& operator<<(QDataStream& d, const RdsLdapObject& object)
{
	d << object.dn();
	return(d);
}

QDataStream& operator>>(QDataStream& d, RdsLdapObject& object)
{
	QString dn; 
	d >> dn;
	object.setDn(dn);
	return(d);
}
