/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU Lesser General Public License as         *
 *  published by the Free Software Foundation; either version 2 of the     *
 *  License, or (at your option) any later version.                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  Lesser General Public License for more details.                        *
 *                                                                         *
 *  You should have received a copy of the GNU Lesser General Public       *
 *  License along with this program; if not, write to the                  *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/
#define LDAP_DEPRECATED 1 //ldap_simple_bind_s is depreciated!?
#include "rdsldapsession.h"
#include "rdsldapsession_p.h"
#include <ldap.h>
#include <ReturnValue>
#include <QDebug>
#include <QStringList>
#include <QRegExp>
#include <errno.h>
#include "lutil_ldap.h"
#include <signal.h>
#include "rdsldapactions_p.h"
#include "config.h"


QTRPC_REGISTER_METATYPE(LdapResults);
QTRPC_REGISTER_METATYPE(LdapResult);
QTRPC_REGISTER_METATYPE(LdapValues);

using namespace QtRpc;

QPointer<RdsLdapSession> RdsLdapSessionPrivate::globalsession = NULL;

RdsLdapSession::RdsLdapSession(QObject *parent)
		: ServiceProxy(parent)
{
	QXT_INIT_PRIVATE(RdsLdapSession);

	QString uri = LDAP_URI;
	int index = uri.indexOf("://");
	if (index != -1)
	{
		QString protocol = uri.left(index + 3);
		QString host = uri.mid(index + 3);
		host = host.replace("/", "%2F");

		uri = protocol + host;
	}

	//qDebug() << "Connecting to LDAP server:" << uri;

	qxt_d().uri = uri;
	qxt_d().authtype = AuthTypeNone;
	signal(SIGPIPE, SIG_IGN);
}

RdsLdapSession::RdsLdapSession(QString uri, QObject *parent)
		: ServiceProxy(parent)
{
	QXT_INIT_PRIVATE(RdsLdapSession);
	qxt_d().uri = uri;
	qxt_d().authtype = AuthTypeNone;

	//qDebug() << "Connecting to LDAP server:" << qxt_d().uri;

	signal(SIGPIPE, SIG_IGN);
}

RdsLdapSession::~RdsLdapSession()
{
}

QtRpc::ReturnValue RdsLdapSession::anonymousBind()
{
	LDAP *tmp;
	int ret = ldap_initialize(&tmp, qPrintable(qxt_d().uri));
	if (ret != LDAP_SUCCESS)
	{
		return(ReturnValue(ret, ldap_err2string(ret)));
	}


	qxt_d().ldap = QSharedPointer<LdapHandle>(new LdapHandle(tmp));

	int version = LDAP_VERSION3;
	ldap_set_option(qxt_d().ldap->ldap, LDAP_OPT_PROTOCOL_VERSION, &version);
	ldap_set_option(qxt_d().ldap->ldap, LDAP_OPT_REFERRALS, 0);

	struct berval *unused;
	struct berval creds;
	creds.bv_val = (char *)"";
	creds.bv_len = 0;
	ret = ldap_sasl_bind_s(qxt_d().ldap->ldap, "", NULL, &creds, NULL, NULL, &unused);
	if (ret != LDAP_SUCCESS)
	{
		return(ReturnValue(ret, ldap_err2string(ret)));
	}

	qxt_d().authtype = AuthTypeAnonymous;
	signal(SIGPIPE, SIG_IGN);

	return(true);
}

QtRpc::ReturnValue RdsLdapSession::simpleBind(const QString &dn, const QString &password)
{
	LDAP *tmp;
	int ret = ldap_initialize(&tmp, qPrintable(qxt_d().uri));
	if (ret != LDAP_SUCCESS)
	{
		return(ReturnValue(ret, ldap_err2string(ret)));
	}

	qxt_d().ldap = QSharedPointer<LdapHandle>(new LdapHandle(tmp));

	int version = LDAP_VERSION3;
	ldap_set_option(qxt_d().ldap->ldap, LDAP_OPT_PROTOCOL_VERSION, &version);
	ldap_set_option(qxt_d().ldap->ldap, LDAP_OPT_REFERRALS, 0);

	ret = ldap_simple_bind_s(qxt_d().ldap->ldap, qPrintable(dn), qPrintable(password));
	if (ret != LDAP_SUCCESS)
	{
		return(ReturnValue(ret, ldap_err2string(ret)));
	}

	qxt_d().authtype = AuthTypeSimpleBind;
	qxt_d().binddn = dn;
	qxt_d().password = password;
	signal(SIGPIPE, SIG_IGN);

	return(true);
}

ReturnValue RdsLdapSession::bind(const QString &user, const QString &password)
{
	LDAP *tmp;
	int ret = ldap_initialize(&tmp, qPrintable(qxt_d().uri));
	if (ret != LDAP_SUCCESS)
	{
		return(ReturnValue(ret, ldap_err2string(ret)));
	}

	qxt_d().ldap = QSharedPointer<LdapHandle>(new LdapHandle(tmp));

	int version = LDAP_VERSION3;
	ldap_set_option(qxt_d().ldap->ldap, LDAP_OPT_PROTOCOL_VERSION, &version);
	ldap_set_option(qxt_d().ldap->ldap, LDAP_OPT_REFERRALS, 0);

	void *defaults;
	unsigned sasl_flags = LDAP_SASL_QUIET;
	char *sasl_mech = NULL;
	char *sasl_realm = NULL;
	QByteArray us = user.toAscii();
	char *sasl_authc_id = us.data();
	char *sasl_authz_id = us.data();
	struct berval passwd;
	QByteArray pw = password.toAscii();
	passwd.bv_val = pw.data();
	passwd.bv_len = password.size();

	defaults = lutil_sasl_defaults(qxt_d().ldap->ldap,
	                               sasl_mech,
	                               sasl_realm,
	                               sasl_authc_id,
	                               passwd.bv_val,
	                               sasl_authz_id);

	ret = ldap_sasl_interactive_bind_s(qxt_d().ldap->ldap,
	                                   "",	/* character string, not really required */
	                                   sasl_mech,
	                                   NULL,	/* servercontrols */
	                                   NULL,	/* clientcontrols */
	                                   sasl_flags,
	                                   lutil_sasl_interact,
	                                   defaults);

	if (ret != LDAP_SUCCESS)
	{
		return(ReturnValue(ret, ldap_err2string(ret)));
	}

	qxt_d().authtype = AuthTypeUserPass;
	qxt_d().binddn = user;
	qxt_d().password = password;
	signal(SIGPIPE, SIG_IGN);

	return(true);
}

ReturnValue RdsLdapSession::kerberosBind()
{
	LDAP *tmp;
	int ret = ldap_initialize(&tmp, qPrintable(qxt_d().uri));
	if (ret != LDAP_SUCCESS)
	{
		return(ReturnValue(ret, QString("Failed to initialize: ") + ldap_err2string(ret)));
	}

	qxt_d().ldap = QSharedPointer<LdapHandle>(new LdapHandle(tmp));

	int version = LDAP_VERSION3;
	ldap_set_option(qxt_d().ldap->ldap, LDAP_OPT_PROTOCOL_VERSION, &version);
	ldap_set_option(qxt_d().ldap->ldap, LDAP_OPT_REFERRALS, 0);

	void *defaults;
	unsigned sasl_flags = LDAP_SASL_QUIET;
	char *sasl_mech = (char *)"GSSAPI";
	char *sasl_realm = NULL;
	char *sasl_authc_id = NULL;
	char *sasl_authz_id = NULL;

	defaults = lutil_sasl_defaults(qxt_d().ldap->ldap,
	                               sasl_mech,
	                               sasl_realm,
	                               sasl_authc_id,
	                               NULL,
	                               sasl_authz_id);

	ret = ldap_sasl_interactive_bind_s(qxt_d().ldap->ldap,
	                                   "",	/* character string, not really required */
	                                   sasl_mech,
	                                   NULL,	/* servercontrols */
	                                   NULL,	/* clientcontrols */
	                                   sasl_flags,
	                                   lutil_sasl_interact,
	                                   defaults);

	if (ret != LDAP_SUCCESS)
	{
		return(ReturnValue(ret, QString("Failed to bind: ") +  ldap_err2string(ret)));
	}

	int *handle = NULL;
	ldap_get_option(qxt_d().ldap->ldap, LDAP_OPT_SOCKBUF, &handle);

	qxt_d().authtype = AuthTypeKerberos;
	signal(SIGPIPE, SIG_IGN);

	return(true);
}
LDAP *RdsLdapSession::handle()
{
	if (qxt_d().ldap == NULL) return(NULL);
	return(qxt_d().ldap->ldap);
}

ReturnValue RdsLdapSession::search(const QString &basedn, const QString &filter) const
{
	return(search(basedn, filter, true));
}

ReturnValue RdsLdapSession::search(const QString &basedn, const QString &filter, bool recursive) const
{
	LdapResults entries;
	LDAPMessage *results = NULL;

	if (basedn == "") return(ReturnValue(1, "You must specify a base DN"));

	signal(SIGPIPE, SIG_IGN);

	int count = 0;
	int sleeptime = 10000;
	int result;
	do
	{
		result = ldap_search_ext_s(qxt_d().ldap->ldap, qPrintable(basedn), recursive ? LDAP_SCOPE_SUBTREE : LDAP_SCOPE_ONELEVEL,
		                           qPrintable(filter), NULL, 0, NULL, NULL, NULL, 0, &results);
		if (result != LDAP_SUCCESS)
		{
			if (results != NULL) ldap_msgfree(results);

			if ((result == LDAP_SERVER_DOWN) || (result == LDAP_LOCAL_ERROR))
			{
				ReturnValue ret = const_cast<RdsLdapSession*>(this)->reAuthenticate();
				if (ret.isError())
				{
					if (count == 10) return(ret);
					usleep(sleeptime);
					count++;
					sleeptime *= 2;
				}
			}
			else
			{
				return(ReturnValue(result, ldap_err2string(result)));
			}
		}
	}
	while ((result == LDAP_SERVER_DOWN) || (result == LDAP_LOCAL_ERROR));

	for (LDAPMessage *ent = ldap_first_entry(qxt_d().ldap->ldap, results); ent != NULL;ent = ldap_next_entry(qxt_d().ldap->ldap, ent))
	{
		LdapResult entry;

		BerElement *ber;
		for (char *attr = ldap_first_attribute(qxt_d().ldap->ldap, ent, &ber); attr != NULL; attr = ldap_next_attribute(qxt_d().ldap->ldap, ent, ber))
		{
			LdapValues values;
			berval **val = ldap_get_values_len(qxt_d().ldap->ldap, ent, attr);
			for (int i = 0; i < ldap_count_values_len(val); i++)
			{
				values << QByteArray(val[i]->bv_val, val[i]->bv_len);
			}
			entry[QString(attr).toLower()] = values;
			ldap_value_free_len(val);
			ldap_memfree(attr);
		}

		entries[ldap_get_dn(qxt_d().ldap->ldap,ent)] = entry;
	}
	ldap_msgfree(results);

	return(QVariant::fromValue(entries));
}

ReturnValue RdsLdapSession::read(const QString &basedn) const
{
	return(read(basedn, QStringList()));
}

QtRpc::ReturnValue RdsLdapSession::read(const QString &basedn, const QStringList &attrs) const
{
	LdapResult entry;
	LDAPMessage *results = NULL;

	signal(SIGPIPE, SIG_IGN);

	if (basedn == "") return(ReturnValue(1, "You must specify a base DN"));

	char **attributes = new char*[attrs.size() + 1];
	for (int i = 0; i < attrs.size(); i++)
	{
		attributes[i] = (char *)malloc((attrs[i].toAscii().size()) + 1);
		memcpy(attributes[i], attrs[i].toAscii().constData(), attrs[i].size() + 1);
	}
	attributes[attrs.size()] = NULL;

	int count = 0;
	int sleeptime = 10000;
	int result;
	do
	{
		result = ldap_search_ext_s(qxt_d().ldap->ldap, qPrintable(basedn), LDAP_SCOPE_BASE,
		                           "objectClass=*", attributes, 0, NULL, NULL, NULL, 0, &results);



		if (result != LDAP_SUCCESS)
		{
			if (results != NULL)
			{
				ldap_msgfree(results);
				results = NULL;
			}

			if ((result == LDAP_SERVER_DOWN) || (result == LDAP_LOCAL_ERROR))
			{
				ReturnValue ret = const_cast<RdsLdapSession*>(this)->reAuthenticate();
				if (ret.isError())
				{
					for (int i = 0; attributes[i] != NULL; i++)
					{
						free(attributes[i]);
						attributes[i] = NULL;
					}

					if (count == 10) return(ret);
					usleep(sleeptime);
					count++;
					sleeptime *= 2;
				}
			}
			else
			{
				for (int i = 0; attributes[i] != NULL; i++)
				{
					free(attributes[i]);
				}

				delete[] attributes;
				if (results != NULL)
				{
					ldap_msgfree(results);
					results = NULL;
				}
				return(ReturnValue(result, ldap_err2string(result)));
			}
		}
	}
	while ((result == LDAP_SERVER_DOWN) || (result == LDAP_LOCAL_ERROR));


	for (int i = 0; attributes[i] != NULL; i++)
	{
		free(attributes[i]);
		attributes[i] = NULL;
	}

	LDAPMessage *ent = ldap_first_entry(qxt_d().ldap->ldap, results);
	if (ent == NULL)
	{
		if (results != NULL)
		{
			ldap_msgfree(results);
			results = NULL;
		}
		delete[] attributes;
		return(ReturnValue(1, "Failed to read LDAP results"));
	}

	BerElement *ber = 0;
	for (char *attr = ldap_first_attribute(qxt_d().ldap->ldap, ent, &ber); attr != NULL; attr = ldap_next_attribute(qxt_d().ldap->ldap, ent, ber))
	{
		LdapValues values;
		berval **val = ldap_get_values_len(qxt_d().ldap->ldap, ent, attr);
		for (int i = 0; i < ldap_count_values_len(val); i++)
		{
			values << QByteArray(val[i]->bv_val, val[i]->bv_len);
		}
		entry[QString(attr).toLower()] = values;
		ldap_value_free_len(val);
		ldap_memfree(attr);
	}

	if (results != NULL)
	{
		ldap_msgfree(results);
		results = NULL;
	}

	delete[] attributes;
	return(QVariant::fromValue(entry));
}

ReturnValue RdsLdapSession::list(const QString &basedn) const
{
	return(list(basedn, "(objectClass=*)"));
}

ReturnValue RdsLdapSession::list(const QString &basedn, const QString &filter) const
{
	return(list(basedn, filter, true));
}

ReturnValue RdsLdapSession::list(const QString &basedn, const QString &filter, bool recursive) const
{
	QStringList entries;
	LDAPMessage *results = NULL;

	signal(SIGPIPE, SIG_IGN);

	if (basedn == "") return(ReturnValue(1, "You must specify a base DN"));

	const char *attrflags[] = {LDAP_NO_ATTRS, NULL};

	int count = 0;
	int sleeptime = 10000;
	int result;
	do
	{
		result = ldap_search_ext_s(qxt_d().ldap->ldap, qPrintable(basedn), recursive ? LDAP_SCOPE_SUBTREE : LDAP_SCOPE_ONELEVEL,
		                           qPrintable(filter), const_cast<char **>(attrflags), 0, NULL, NULL, NULL, 0, &results);
		if (result != LDAP_SUCCESS)
		{
			if (results != NULL) ldap_msgfree(results);

			if ((result == LDAP_SERVER_DOWN) || (result == LDAP_LOCAL_ERROR))
			{
				ReturnValue ret = const_cast<RdsLdapSession*>(this)->reAuthenticate();
				if (ret.isError())
				{
					if (count == 10) return(ret);
					usleep(sleeptime);
					count++;
					sleeptime *= 2;
				}
			}
			else
			{
				return(ReturnValue(result, ldap_err2string(result)));
			}
		}
	}
	while ((result == LDAP_SERVER_DOWN) || (result == LDAP_LOCAL_ERROR));

	for (LDAPMessage *ent = ldap_first_entry(qxt_d().ldap->ldap, results); ent != NULL;ent = ldap_next_entry(qxt_d().ldap->ldap, ent))
	{
		entries << ldap_get_dn(qxt_d().ldap->ldap, ent);
	}
	ldap_msgfree(results);

	return(QVariant::fromValue(entries));
}

RdsLdapSession *RdsLdapSession::globalSession()
{
	if (RdsLdapSessionPrivate::globalsession == NULL)
	{
		qFatal("The global LDAP object was used with ought being initialized!");
		return(NULL);
	}

	return(RdsLdapSessionPrivate::globalsession);
}

void RdsLdapSession::setGlobalSession(RdsLdapSession *session)
{
	RdsLdapSessionPrivate::globalsession = session;
}

bool RdsLdapSession::isGlobalSessionSet()
{
	return(RdsLdapSessionPrivate::globalsession != NULL);
}


ReturnValue RdsLdapSession::add(const QString &dn, const RdsLdapActions &actions)
{
	LDAPMod **mods = actions.getLdapActions();


	/*
	qDebug() << "Adding:" << actions.getActions().size();
	foreach(LdapAction action, actions.getActions())
	{
		qDebug() << "ACTION:" << action.type << action.name << action.values << action.data;
	}
	*/

	signal(SIGPIPE, SIG_IGN);

	if (dn == "") return(ReturnValue(1, "You must specify a DN"));

	int count = 0;
	int sleeptime = 10000;
	int result;
	do
	{
		result = ldap_add_ext_s(qxt_d().ldap->ldap, qPrintable(dn), mods, NULL, NULL);
		ldap_mods_free(mods, 0);
		if (result != LDAP_SUCCESS)
		{
			if ((result == LDAP_SERVER_DOWN) || (result == LDAP_LOCAL_ERROR))
			{
				ReturnValue ret = this->reAuthenticate();
				if (ret.isError())
				{
					if (count == 10) return(ret);
					usleep(sleeptime);
					count++;
					sleeptime *= 2;
				}
			}
			else
			{
				return(ReturnValue(result, ldap_err2string(result)));
			}
		}
	}
	while ((result == LDAP_SERVER_DOWN) || (result == LDAP_LOCAL_ERROR));

	return(true);
}

ReturnValue RdsLdapSession::modify(const QString &dn, const RdsLdapActions &actions)
{
	/*
	qDebug() << "Modifying:" << actions.getActions().size();
	foreach(LdapAction action, actions.getActions())
	{
		qDebug() << "ACTION:" << action.type << action.name << action.values << action.data;
	}
	*/


	LDAPMod **mods = actions.getLdapActions();

	signal(SIGPIPE, SIG_IGN);

	if (dn == "") return(ReturnValue(1, "You must specify a DN"));

	int count = 0;
	int sleeptime = 10000;
	int result;
	do
	{
		result = ldap_modify_ext_s(qxt_d().ldap->ldap, qPrintable(dn), mods, NULL, NULL);
		ldap_mods_free(mods, 0);
		if (result != LDAP_SUCCESS)
		{
			if ((result == LDAP_SERVER_DOWN) || (result == LDAP_LOCAL_ERROR))
			{
				ReturnValue ret = this->reAuthenticate();
				if (ret.isError())
				{
					if (count == 10) return(ret);
					usleep(sleeptime);
					count++;
					sleeptime *= 2;
				}
			}
			else
			{
				return(ReturnValue(result, ldap_err2string(result)));
			}
		}
	}
	while ((result == LDAP_SERVER_DOWN) || (result == LDAP_LOCAL_ERROR));

	return(true);
}

ReturnValue RdsLdapSession::remove(const QString &dn, bool recursive)
{
	signal(SIGPIPE, SIG_IGN);

	if (dn == "") return(ReturnValue(1, "You must specify a DN"));

	if (recursive)
	{
		ReturnValue ret = list(dn, "(objectClass=*)", true);
		if (ret.isError()) return(ret);
		QStringList children = ret.toStringList();

		foreach(QString child, children)
		{
			if (child.toLower() == dn.toLower()) continue;
			remove(child, recursive);
		}
	}

	int count = 0;
	int sleeptime = 10000;
	int result;
	do
	{
		result = ldap_delete_ext_s(qxt_d().ldap->ldap, qPrintable(dn), NULL, NULL);
		if (result != LDAP_SUCCESS)
		{
			if ((result == LDAP_SERVER_DOWN) || (result == LDAP_LOCAL_ERROR))
			{
				ReturnValue ret = this->reAuthenticate();
				if (ret.isError())
				{
					if (count == 10) return(ret);
					usleep(sleeptime);
					count++;
					sleeptime *= 2;
				}
			}
			else
			{
				return(ReturnValue(result, ldap_err2string(result)));
			}
		}
	}
	while ((result == LDAP_SERVER_DOWN) || (result == LDAP_LOCAL_ERROR));

	return(true);
}

ReturnValue RdsLdapSession::rename(const QString &dn, const QString &rdn)
{
	signal(SIGPIPE, SIG_IGN);

	if (dn == "") return(ReturnValue(1, "You must specify a DN"));
	if (rdn == "") return(ReturnValue(1, "You must specify a RDN"));

	int count = 0;
	int sleeptime = 10000;
	int result;
	do
	{
		result = ldap_rename_s(qxt_d().ldap->ldap, qPrintable(dn), qPrintable(rdn), NULL, true, NULL, NULL);
		if (result != LDAP_SUCCESS)
		{
			if ((result == LDAP_SERVER_DOWN) || (result == LDAP_LOCAL_ERROR))
			{
				ReturnValue ret = this->reAuthenticate();
				if (ret.isError())
				{
					if (count == 10) return(ret);
					usleep(sleeptime);
					count++;
					sleeptime *= 2;
				}
			}
			else
			{
				return(ReturnValue(result, ldap_err2string(result)));
			}
		}
	}
	while ((result == LDAP_SERVER_DOWN) || (result == LDAP_LOCAL_ERROR));

	return(true);
}

ReturnValue RdsLdapSession::move(const QString &dn, const QString &newparent)
{
	QString rdn = dn;
	rdn.replace(QRegExp(",.*$"), "");

	signal(SIGPIPE, SIG_IGN);

	if (dn == "") return(ReturnValue(1, "You must specify a DN"));
	if (newparent == "") return(ReturnValue(1, "You must specify a new parent"));

	//qDebug() << "Moving:" << dn << newparent << rdn;

	int count = 0;
	int sleeptime = 10000;
	int result;
	do
	{
		result = ldap_rename_s(qxt_d().ldap->ldap, qPrintable(dn), qPrintable(rdn), qPrintable(newparent), true, NULL, NULL);
		if (result != LDAP_SUCCESS)
		{
			if ((result == LDAP_SERVER_DOWN) || (result == LDAP_LOCAL_ERROR))
			{
				ReturnValue ret = this->reAuthenticate();
				if (ret.isError())
				{
					if (count == 10) return(ret);
					usleep(sleeptime);
					count++;
					sleeptime *= 2;
				}
			}
			else
			{
				return(ReturnValue(result, ldap_err2string(result)));
			}
		}
	}
	while ((result == LDAP_SERVER_DOWN) || (result == LDAP_LOCAL_ERROR));

	return(true);
}

ReturnValue RdsLdapSession::reAuthenticate()
{
	ReturnValue ret;
	for (int i = 0; i < 3; i++)
	{
		switch (qxt_d().authtype)
		{
			case AuthTypeNone:
				return(ReturnValue(1, "Not Authenticated"));
				break;
			case AuthTypeAnonymous:
				ret = anonymousBind();
				if (!ret.isError()) return(ret);
			case AuthTypeUserPass:
				ret = bind(qxt_d().binddn, qxt_d().password);
				if (!ret.isError()) return(ret);
				break;
			case AuthTypeKerberos:
				ret = kerberosBind();
				if (!ret.isError()) return(ret);
				break;
			case AuthTypeSimpleBind:
				ret = simpleBind(qxt_d().binddn, qxt_d().password);
				if (!ret.isError()) return(ret);
				break;
		}
	}

	return(ret);
}

RdsLdapSession &RdsLdapSession::operator =(const RdsLdapSession & object)
{
	qxt_d().ldap = object.qxt_d().ldap;
	qxt_d().uri = object.qxt_d().uri;
	qxt_d().authtype = object.qxt_d().authtype;
	qxt_d().binddn = object.qxt_d().binddn;
	qxt_d().password = object.qxt_d().password;
	return(*this);
}



