/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU General Public License as published by   *
 *  the Free Software Foundation; either version 2 of the License, or (at  *
 *  your option) any later version.                                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  General Public License for more details.                               *
 *                                                                         *
 *  You should have received a copy of the GNU General Public License      *
 *  along with this program; if not, write to the                          *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/
#include "rdsprovisionator.h"
#include <QDebug>
#include <RdsClient>

using namespace QtRpc;

RdsProvisionator::RdsProvisionator(QObject *parent)
		: ClientProxy(parent)
{
	_timer = new QTimer(this);
	QObject::connect(_timer, SIGNAL(timeout()), this, SLOT(timeout()));
	QObject::connect(this, SIGNAL(completed(ReturnValue)), this, SLOT(onCompleted(ReturnValue)));
	QObject::connect(this, SIGNAL(minorProgress(QString)), this, SLOT(onMinorProgress(QString)));
	QObject::connect(this, SIGNAL(majorProgress(QString)), this, SLOT(onMajorProgress(QString)));
}


RdsProvisionator::~RdsProvisionator()
{
}

void RdsProvisionator::onMinorProgress(const QString &msg)
{
	qDebug() << qPrintable("  * " + msg);
}

void RdsProvisionator::onMajorProgress(const QString &msg)
{
	qDebug() << qPrintable(" - " + msg);
}

void RdsProvisionator::onCompleted(const ReturnValue &ret)
{
	if (ret.isError())
	{
		qDebug() << "Failed to provision:" << ret.errNumber() << ret.errString();
		exit(2);
	}
	else
	{
		qDebug() << "Finished";
		exit(0);
	}
}

void RdsProvisionator::asyncConnect(QString user, QString pass, QString ip, QString oldip, int port, QString service, QVariantMap settings)
{
	_user = user;
	_pass = pass;
	_ip = ip;
	_oldip = oldip;
	_port = port;
	_service = service;
	_settings = settings;
	_reverted = false;
	
	_time.start();
	doConnect();
}

void RdsProvisionator::timeout()
{
	qDebug() << "  * Connection timed out.";
	doConnect();
}

void RdsProvisionator::connected(uint, ReturnValue ret)
{
	_timer->stop();

	if (ret.isError())
	{
		qDebug() << "  * Failed to connect:" << ret;
		doConnect();
	}
	else
	{
		qDebug() << "  * Connected to:" << _ip;
		if(_reverted)
		{
			qDebug();
			qCritical() << "Failed to connect to the server!";
			qCritical() << "The server has reverted to its old network settings.";
			exit(1);
		}

		ret = rdsClient()->provisionator();
		if (ret.isError())
		{
			qDebug() << "Failed to provision:" << ret.errNumber() << ret.errString();
			exit(1);
		}

		*this = ret;

		qDebug() << " * Provisioning";

		ret = provision(_settings);
		if (ret.isError())
		{
			qDebug() << "Failed to provision:" << ret.errNumber() << ret.errString();
			exit(1);
		}
	}
}

void RdsProvisionator::doConnect()
{
	if (_reverted)
	{
		if (_time.elapsed() >= 30000)
		{
			qDebug();
			qCritical() << "Failed to connect to the server!";
			qCritical() << "The server tried to revert to its old network settings, but has failed.";
			exit(1);
		}
	}
	else
	{
		if (_time.elapsed() >= 70000)
		{
			qDebug() << " - Failed to connect to the server within 60 secconds.";
			qDebug() << " - Trying the old IP address.";
			_ip = _oldip;
			_reverted = true;
			_time.restart();
		}
	}

	qDebug() << "  * Connecting To:" << _ip;
	rdsClient()->connect(this, SLOT(connected(uint, ReturnValue)),
	                     QString("tcp://%1:%2@%3:%4/%5").arg(_user).arg(_pass).arg(_ip).arg(_port).arg(_service));
	_timer->start(10000);
}
