/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU Lesser General Public License as         *
 *  published by the Free Software Foundation; either version 2 of the     *
 *  License, or (at your option) any later version.                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  Lesser General Public License for more details.                        *
 *                                                                         *
 *  You should have received a copy of the GNU Lesser General Public       *
 *  License along with this program; if not, write to the                  *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/
#include "config.h"
#include "rdsdnsmanager.h"
#include "rdsdnsmanager_p.h"
#include "rdsdnszone_p.h"
#include <QDebug>
#include <QFile>
#include <QFileInfo>
#include <QDir>
#include <QProcess>

#include "RdsEntity"
#include "RdsUtils"
#include "RdsDaemonManager"

#include <AuthToken>


#define RDS_LOCK QMutexLocker __locker(qxt_d().data->getMutex());
QString createOutput(const QVariantMap &data, const QVariantMap &comments = QVariantMap(), int indentation = 0);

QTRPC_SERVICEPROXY_PIMPL_IMPLEMENT(RdsDnsManager);
RDS_REGISTER_DAEMON(RdsDnsManager, "Dns");

RdsDnsManagerData& RdsDnsManagerPrivate::staticData()
{
	static RdsDnsManagerData staticData;
	return staticData;
}

QVariantMap insertData(QVariantMap map, QStringList breadCrumb, QVariant value)
{
	QString key;
	while (breadCrumb.last().isEmpty())
		breadCrumb.removeLast();
	while (key.isEmpty())
		key = breadCrumb.takeFirst();
	if (breadCrumb.count() == 0)
	{
		if (value.canConvert(QVariant::Map))
			map.insertMulti(key, value);
		else
			map.insert(key, value);
	}
	else
		map[key] = insertData(map.value(key).toMap(), breadCrumb, value);
	return map;
}

QVariantMap recursivelyUnite(const QVariantMap& map, const QVariantMap& other)
{
	QVariantMap output = map;
	for (QVariantMap::const_iterator i = other.begin(); i != other.end(); ++i)
	{
		if (i.value().canConvert(QVariant::Map) && output.value(i.key()).canConvert(QVariant::Map))
			output[i.key()] = recursivelyUnite(output.value(i.key()).toMap(), i.value().toMap());
		else
			output.insertMulti(i.key(), i.value());
	}
	return output;
}

RdsDnsManager::RdsDnsManager()
		: RdsEntityManager(NULL)
{
	RDS_LOCK;
	if (!qxt_d().data->parsed)
		reload();
}

RdsDnsManager::~RdsDnsManager()
{
}

ReturnValue RdsDnsManager::stopService()
{
	return RdsUtils::runCommand(RDS_INITD_BIND, QStringList() << "stop");
}

ReturnValue RdsDnsManager::startService()
{
	return RdsUtils::runCommand(RDS_INITD_BIND, QStringList() << "start");
}

ReturnValue RdsDnsManager::restartService()
{
	return RdsUtils::runCommand(RDS_INITD_BIND, QStringList() << "restart");
}

ReturnValue RdsDnsManager::reloadService()
{
	ReturnValue ret = RdsUtils::runCommand("rndc", QStringList() << "freeze");
        if(ret.isError()) return(ret);
        ret = RdsUtils::runCommand("rndc", QStringList() << "reload");
        if(ret.isError()) return(ret);
        ret = RdsUtils::runCommand("rndc", QStringList() << "thaw");
        return(ret);
}

ReturnValue RdsDnsManager::reloadConfig()
{
	return reload();
}

ReturnValue RdsDnsManager::auth(QtRpc::AuthToken token)
{
	createInternalObject();
	if (token.serverData().contains("authenticated") && (token.serverData().value("authenticated").toBool() == true))
		return(true);
	else
		return(ReturnValue(1, "Not Authenticated"));
}

ReturnValue RdsDnsManager::listEntities(const QString &dn, bool loadmore) const
{
	ReturnValue ret;
	if ((dn == "") || (dn == "root"))
	{
		RdsEntity entity;
		entity.setId("root");
		entity.setType("root");
		entity.setVisible(false);
		entity.setName("");
		entity.setParent("");

		ret = listEntities("manager", loadmore);
		if (!ret.isError()) entity.children() << ret.value<RdsEntity>();

		return ReturnValue::fromValue<RdsEntity>(entity);
	}
	else if (dn == "manager")
	{
		RdsEntity entity;
		entity.setId("manager");
		entity.setType("manager");
		entity.setVisible(true);
		entity.setName("DNS Settings");
		entity.setParent("root");

		ret = zones();
		if (!ret.isError())
		{
			foreach(QString zone, ret.toStringList())
			{
				ret = listEntities(zone, loadmore);
				if (!ret.isError()) entity.children() << ret.value<RdsEntity>();
			}
		}
		else
		{
			qCritical() << "Failed to list zones" << ret.errString();
		}
		return ReturnValue::fromValue<RdsEntity>(entity);
	}
	else if (dn.contains("#"))
	{
		// Record!!
		RdsEntity entity;
		QStringList path = dn.split(":");
		QString lastPathPart = path.at(path.count() - 1);
		path[path.count() - 1] = lastPathPart.left(lastPathPart.indexOf("#"));
		entity.setType("record");
		entity.setId(dn);
		entity.setName(lastPathPart.split("#").at(1));
		if (entity.name() == "IN")
		{
			entity.setName(path.first());
		}
		entity.setParent(path.join(":"));
		return ReturnValue::fromValue<RdsEntity>(entity);
	}
	else
	{
		RdsEntity entity;
		entity.setVisible(true);
		QString zoneName;
		QString origin;
		if (dn.contains(":"))
		{
			// origin
			QStringList path = dn.split(":");
			QStringList smallPath = path;
			smallPath.removeLast();
			entity.setType("origin");
			entity.setId(dn);
			entity.setName(path.last());
			entity.setParent(smallPath.join(":"));

			smallPath = path;
			smallPath.removeFirst();
			zoneName = path.first();
			foreach(QString dnPart, path)
			{
				origin = origin.prepend(dnPart + ".");
			}
			origin = origin.left(origin.count() - 1);
		}
		else
		{
			// zone
			entity.setId(dn);
			if (zoneType(dn).toString() == "master")
				entity.setType("zone");
			else
				entity.setType("slavezone");
			entity.setName(dn.left(dn.count() - 1));
			entity.setParent("manager");

			origin = dn;
			zoneName = dn;
		}
		if (!origin.isEmpty() && !zoneName.isEmpty() && (zoneType(zoneName).toString() == "master"))
		{
			ret = this->zone(zoneName);
			RdsDnsZone zone;
			if (!ret.isError())
			{
				zone = ret;
				if (origin != zoneName)
					ret = zone.listOrigins(origin.left(origin.count() - zoneName.count() - 1));
				else
					ret = zone.listOrigins();

				if (!ret.isError())
				{
					foreach(QString o, ret.toStringList())
					{
						ret = listEntities(dn + ":" + o, loadmore);
						if (!ret.isError()) entity.children() << ret.value<RdsEntity>();
					}
				}
				else
				{
					qCritical() << "Failed to listOrigins for" << zoneName << ret.errString();
				}
				ret = zone.listRecordsByOrigin(origin);
				if (!ret.isError())
				{
					RdsDnsRecordList list = ret.value<RdsDnsRecordList>();
					QStringList alreadyListed;
					foreach(RdsDnsRecord record, list)
					{
						QString listEnt = (dn + "#" + record.key() + "#" + QString::number(record.type()));
						if (alreadyListed.contains(listEnt))
							continue;
						alreadyListed << listEnt;
						ret = listEntities(listEnt, loadmore);
						if (!ret.isError()) entity.children() << ret.value<RdsEntity>();
					}
				}
				else
				{
					qCritical() << "Failed to listRecordsByOrigin for" << zoneName << ret.errString();
				}
			}
			else
			{
				qCritical() << "Failed to get the zone object for" << zoneName << ret.errString();
			}
		}
		return ReturnValue::fromValue<RdsEntity>(entity);
	}
}

ReturnValue RdsDnsManager::reload()
{
	RDS_LOCK;
	QFile file(QString(RDS_DNS_CONFIGDIR) + "named.conf");
	file.open(QFile::ReadOnly);

	qxt_d().data->values = QVariantMap();
	qxt_d().data->comments = QVariantMap();
	qxt_d().data->forwarders = QStringList();

	if (parse(file.readAll()))
	{
		qxt_d().data->parsed = true;
		return true;
	}
	return ReturnValue(ReturnValue::GenericError, "Failed to parse configuration file");
}

bool RdsDnsManager::parse(const QString &data)
{
	RDS_LOCK;
	qxt_d().data->valid = false;
	qxt_d().data->errorString = "Unknown error";
	bool endOfLine = false;
	bool commentBlock = false;
	quint32 wordNum = 1;
	quint32 lineNumber = 1;
	quint32 brackets = 0;
	QString word;
	QString line;
	QVariantMap values;
	QVariantMap commentMap;
	QStringList breadcrumb;
	QString value;
	QStringList comments;
	for (QString::const_iterator i = data.begin(); i != data.end(); ++i)
	{
		if (*i == '\n')
			lineNumber++;
		if (commentBlock)
		{
			if (*i == '*' && *(i + 1) == '/')
			{
				++i;
				commentBlock = false;
				if (comments.at(comments.count() - 1).startsWith("(RDS)"))
					comments.removeLast();
			}
			else if (*i == '\n')
			{
				if (comments.at(comments.count() - 1).startsWith("(RDS)"))
					comments.removeLast();
				comments << QString::Null();
			}
			else
				comments[comments.count() - 1] += *i;
			continue;
		}
		if (endOfLine)
		{
			if (*i == '\n')
			{
				if (comments.at(comments.count() - 1).startsWith("(RDS)"))
					comments.removeLast();
				endOfLine = false;
			}
			else
				comments[comments.count() - 1] += *i;
			continue;
		}
		if (*i == ';' || *i == ' ' || *i == '\t')
		{
			if (!word.isEmpty())
			{
				line += word + " ";
			}
			if (!word.isEmpty())
			{
				switch (wordNum)
				{
					case 1:
						breadcrumb << word;
						break;
					default:
					{
						value = (value + " " + word).trimmed();
						break;
					}
				}
				// parse the words
				word = QString::Null();
				wordNum++;
			}
			if (*i == ';' && !line.isEmpty())
			{
				if (breadcrumb.count() == 1 && breadcrumb.last() == "include")
				{
					QFile file(value.replace("\"", ""));
					if (file.open(QFile::ReadOnly))
					{
						QString str = file.readAll();
						if (!parse(str))
							return false;
					}
				}
				else
				{
					values = insertData(values, breadcrumb, value);
					if (comments.count() != 0)
						commentMap = insertData(commentMap, QStringList(breadcrumb) << value << "__COMMENT", comments);
				}
				breadcrumb.removeLast();
				line = QString::Null();
				value = QString::Null();
				comments = QStringList();
				wordNum = 1;
			}
			continue;
		}
		if (*i == '/' && *(i + 1) == '*')
		{
			++i;
			comments << QString::Null();
			commentBlock = true;
			continue;
		}
		if ((*i == '/' && *(i + 1) == '/') || (*i == '#'))
		{
			++i;
			comments << QString::Null();
			endOfLine = true;
			continue;
		}
		if (*i == '{')
		{
			brackets++;
			breadcrumb << value;
			if (comments.count() != 0)
				commentMap = insertData(commentMap, QStringList(breadcrumb) << "__COMMENT", comments);
			line = QString::Null();
			word = QString::Null();
			value = QString::Null();
			comments = QStringList();
			wordNum = 1;
			continue;
		}
		if (*i == '}')
		{
			brackets--;
			breadcrumb.removeLast();
			breadcrumb.removeLast();
			continue;
		}
		if (*i == ';' || *i == '\n')
		{
			continue;
		}
		if (*i == '\\')
		{
			i++;
		}
		word += *i;
	}
	if (values.contains("zone"))
	{
		foreach(QString zone, values["zone"].toMap().keys())
		{
			QString strippedZone = zone.split(" ", QString::SkipEmptyParts).first().replace("\"", "");
			if (strippedZone == "." ||
			        strippedZone == "localhost" ||
			        strippedZone == "127.in-addr.arpa" ||
			        strippedZone == "255.in-addr.arpa" ||
			        strippedZone == "0.in-addr.arpa")
			{
				continue;
			}
			RdsDnsManagerData::RdsZoneInfo info;
			info.zone = strippedZone;
			info.values = values["zone"].toMap()[zone].toMap();
			info.zoneKey = zone;
			for (QVariantMap::iterator i = info.values.begin(); i != info.values.end(); ++i)
			{
				if (i.value().canConvert(QVariant::String))
				{
					i.value() = i.value().toString().replace("\"", "");
				}
			}
			qxt_d().data->zones[strippedZone] = info;
		}
	}
	qxt_d().data->errorString = "No error";
	qxt_d().data->valid = true;
	qxt_d().data->values = recursivelyUnite(values, qxt_d().data->values);
	foreach(QString f, qxt_d().data->values.value("options").toMap().value("forwarders").toMap().keys())
	{
		if (!qxt_d().data->forwarders.contains(f))
			qxt_d().data->forwarders << f;
	}
	qxt_d().data->comments = recursivelyUnite(commentMap, qxt_d().data->values);
	return true;
}

QString createOutput(const QVariantMap &data, const QVariantMap &comments, int indentation)
{
	QString output;
	QVariantMap inserted;
	QVariantMap::const_iterator comment = comments.begin();
	for (QVariantMap::const_iterator i = data.begin(); i != data.end(); ++i)
	{
		if (inserted.values(i.key()).contains(i.value()))
			continue;
		inserted.insert(i.key(), i.value());
		comment = comments.constFind(i.key());
		QVariantMap newComments;
		if (comment != comments.end() && comment.value().canConvert(QVariant::Map))
			newComments = comment.value().toMap();

		if (i.value().canConvert(QVariant::Map))
		{
// 			if (i.value().toMap().count() > 0)
// 				qDebug() << "DNS evaluating:" << i.key() << i.value().toMap().begin().key();
			if ((i.value().toMap().size() == 1) && (i.value().toMap().begin().value().canConvert(QVariant::Map)))
			{
				comment = newComments.constFind(i.value().toMap().begin().key());
				if (comment != comments.end() && comment.value().canConvert(QVariant::Map))
				{
					newComments = comment.value().toMap();
					if (newComments.contains("__COMMENT"))
					{
						foreach(QString line, newComments.value("__COMMENT").toStringList())
						{
							output += QString(indentation, '\t') + "//" + line + "\n";
						}
					}
				}
				else
				{
					newComments = QVariantMap();
				}
				output += QString(indentation, '\t') + i.key() + ' ' + i.value().toMap().begin().key() + " {\n";
				output += createOutput(i.value().toMap().begin().value().toMap(), newComments, indentation + 1);
				output += QString(indentation, '\t') + "};\n";
			}
			else
			{
				if (newComments.contains("__COMMENT"))
				{
					foreach(QString line, newComments.value("__COMMENT").toStringList())
					{
						output += QString(indentation, '\t') + "//" + line + "\n";
					}
				}
				output += QString(indentation, '\t') + i.key() + " {\n";
				output += createOutput(i.value().toMap(), newComments, indentation + 1);
				output += QString(indentation, '\t') + "};\n";
			}
		}
		else
		{
			if (newComments.contains(i.value().toString()))
				newComments = newComments.value(i.value().toString()).toMap();
			else
				newComments = QVariantMap();
			if (newComments.contains("__COMMENT"))
			{
				foreach(QString line, newComments.value("__COMMENT").toStringList())
				{
					output += QString(indentation, '\t') + "//" + line + "\n";
				}
			}
			output += QString(indentation, '\t') + i.key();
			if (!i.value().toString().isEmpty())
			{
				output += ' ' + i.value().toString();
			}
			output += ";\n";
		}
	}
	return output;
}

ReturnValue RdsDnsManager::namedConf() const
{
	RDS_LOCK;
	QString output = QString() +
	                 "// This configuration file was generated by RDS. We do not recommend\n"
	                 "// you edit it manually. Instead, please use the GUI configuration tools.\n"
	                 "// This is the primary configuration file for the BIND DNS server named.\n"
	                 "//\n"
	                 "// Please read /usr/share/doc/bind9/README.Debian.gz for information on the\n"
	                 "// structure of BIND configuration files in Debian, *BEFORE* you customize\n"
	                 "// this configuration file.\n"
	                 "//\n"
	                 "// If you are just adding zones, please do that in " + RDS_DNS_CONFIGDIR + "named.conf.local\n"
	                 "\n"
	                 "include \"" + RDS_DNS_CONFIGDIR + "named.conf.options\";\n"
	                 "\n"
	                 "// prime the server with knowledge of the root servers\n"
	                 "zone \".\" {\n"
	                 "	type hint;\n"
	                 "	file \"" + RDS_DNS_CONFIGDIR + "db.root\";\n"
	                 "};\n"
	                 "\n"
	                 "// be authoritative for the localhost forward and reverse zones, and for\n"
	                 "// broadcast zones as per RFC 1912\n"
	                 "\n"
	                 "zone \"localhost\" {\n"
	                 "	type master;\n"
	                 "	file \"" + RDS_DNS_CONFIGDIR + "db.local\";\n"
	                 "};\n"
	                 "\n"
	                 "zone \"127.in-addr.arpa\" {\n"
	                 "	type master;\n"
	                 "	file \"" + RDS_DNS_CONFIGDIR + "db.127\";\n"
	                 "};\n"
	                 "\n"
	                 "zone \"0.in-addr.arpa\" {\n"
	                 "	type master;\n"
	                 "	file \"" + RDS_DNS_CONFIGDIR + "db.0\";\n"
	                 "};\n"
	                 "\n"
	                 "zone \"255.in-addr.arpa\" {\n"
	                 "	type master;\n"
	                 "	file \"" + RDS_DNS_CONFIGDIR + "db.255\";\n"
	                 "};\n"
	                 "\n"
	                 "include \"" + RDS_DNS_CONFIGDIR + "named.conf.local\";\n";
	return output;
}

ReturnValue RdsDnsManager::namedConfLocal() const
{
	RDS_LOCK;
	QString output =
	    "//(RDS) This configuration file was generated by RDS. We do not recommend\n"
	    "//(RDS) you edit it manually. Instead, please use the GUI configuration tools.\n\n";

	QVariantMap data;
	for (QVariantMap::const_iterator i = qxt_d().data->values.value("zone").toMap().constBegin(); i != qxt_d().data->values.value("zone").toMap().constEnd(); ++i)
	{
		QString strippedZone = i.key().split(" ", QString::SkipEmptyParts).first().replace("\"", "");
		if (strippedZone == "." ||
		        strippedZone == "localhost" ||
		        strippedZone == "127.in-addr.arpa" ||
		        strippedZone == "255.in-addr.arpa" ||
		        strippedZone == "0.in-addr.arpa")
		{
			continue;
		}

		QVariantMap tmpData, tmpData2;
		tmpData = i.value().toMap();
		tmpData.insert("check-names", "ignore");
		tmpData2.insert('"' + strippedZone + '"', tmpData);
		data.insertMulti("zone", tmpData2);
	}
	output += createOutput(data, qxt_d().data->comments);
	return output;
}

ReturnValue RdsDnsManager::namedConfOptions() const
{
	RDS_LOCK;
	QString output =
	    "//(RDS) This configuration file was generated by RDS. We do not recommend\n"
	    "//(RDS) you edit it manually. Instead, please use the GUI configuration tools.\n\n";

	QVariantMap data;
	if (qxt_d().data->values.contains("options"))
	{
		data = qxt_d().data->values.value("options").toMap();
		data.remove("forwarders");
		qxt_d().data->values["options"] = data;
	}
	if (qxt_d().data->forwarders.count() > 0)
	{
		data = QVariantMap();
		foreach(QString f, qxt_d().data->forwarders)
		{
			data[f];
		}
		QVariantMap tmpdata = qxt_d().data->values.value("options").toMap();
		tmpdata["forwarders"] = data;
		qxt_d().data->values["options"] = tmpdata;
	}
	data = qxt_d().data->values;
	data.remove("zone");
	output += "\n" + createOutput(data, qxt_d().data->comments);

	return output;
}

ReturnValue RdsDnsManager::save() const
{
	//qDebug() << "Saving DNS";
	RDS_LOCK;
	QString conf;
	QString local;
	QString options;
	ReturnValue ret;

	ret = const_cast<RdsDnsManager *>(this)->prepareToSave();
	if (ret.isError())
		return ret;

	ret = namedConf();
	if (ret.isError())
		return ret;
	conf = ret.toString();

	ret = namedConfLocal();
	if (ret.isError())
		return ret;
	local = ret.toString();

	ret = namedConfOptions();
	if (ret.isError())
		return ret;
	options = ret.toString();

	if (QFile::exists(RDS_DNS_CONFIGDIR))
	{
		QFileInfo info(RDS_DNS_CONFIGDIR);
		if (!info.isDir())
		{
			if (!QFile::remove(RDS_DNS_CONFIGDIR))
				return ReturnValue(1, QString("Bind config directory exists as a file and is not removable. Please remove ") + RDS_DNS_CONFIGDIR + " and try again");
			if (!QDir().mkdir(RDS_DNS_CONFIGDIR))
				return ReturnValue(1, QString("Failed to create the bind config directory. Check the permissions of ") +  RDS_DNS_CONFIGDIR + " and try again");
		}
	}
	else
	{
		if (!QDir().mkdir(RDS_DNS_CONFIGDIR))
			return ReturnValue(1, QString("Failed to create the bind config directory. Check the permissions of ") + RDS_DNS_CONFIGDIR + " and try again");
	}
	QProcess::execute("rm", QStringList() <<  "-r" << RDS_DNS_CONFIGDIR_BACKUP);
	if (QProcess::execute("cp", QStringList() << "-a" << RDS_DNS_CONFIGDIR << RDS_DNS_CONFIGDIR_BACKUP) != 0)
	{
		return ReturnValue(1, "Failed to backup the DNS files");
	}

	QFile file(QString(RDS_DNS_CONFIGDIR) + "named.conf");
	if (!file.open(QFile::Truncate | QFile::WriteOnly))
	{
		QProcess::execute("rm", QStringList() << "-r" << RDS_DNS_CONFIGDIR);
		QDir().rename(RDS_DNS_CONFIGDIR_BACKUP, RDS_DNS_CONFIGDIR);
		return ReturnValue(1, QString("Failed to open ") + RDS_DNS_CONFIGDIR +  "named.conf for writing");
	}

	QFile fileLocal(QString(RDS_DNS_CONFIGDIR) +  "named.conf.local");
	if (!fileLocal.open(QFile::Truncate | QFile::WriteOnly))
	{
		QProcess::execute("rm", QStringList() << "-r" << RDS_DNS_CONFIGDIR);
		QDir().rename(RDS_DNS_CONFIGDIR_BACKUP, RDS_DNS_CONFIGDIR);
		return ReturnValue(1, QString("Failed to open ") + RDS_DNS_CONFIGDIR + "named.conf.local for writing");
	}

	QFile fileOptions(QString(RDS_DNS_CONFIGDIR) + "named.conf.options");
	if (!fileOptions.open(QFile::Truncate | QFile::WriteOnly))
	{
		QProcess::execute("rm", QStringList() << "-r" << RDS_DNS_CONFIGDIR);
		QDir().rename(RDS_DNS_CONFIGDIR_BACKUP, RDS_DNS_CONFIGDIR);
		return ReturnValue(1, QString("Failed to open ") + RDS_DNS_CONFIGDIR + "named.conf.options for writing");
	}

	file.write(conf.toLocal8Bit());
	fileLocal.write(local.toLocal8Bit());
	fileOptions.write(options.toLocal8Bit());
	QProcess proc;
	proc.setProcessChannelMode(QProcess::ForwardedChannels);
	proc.start("named-checkconf", QStringList() << QString(RDS_DNS_CONFIGDIR) + "named.conf");
	proc.waitForFinished();
	if (proc.exitCode() != 0)
	{
// 		system("rm -r " RDS_DNS_CONFIGDIR);
		QProcess::execute("mv", QStringList() << RDS_DNS_CONFIGDIR << " /tmp/badbind/");
		QDir().rename(RDS_DNS_CONFIGDIR_BACKUP, RDS_DNS_CONFIGDIR);
		return ReturnValue(ReturnValue::GenericError, "The generated bind config files appears to be invalid. Check that there are no abnormalities in the data");
	}
	QProcess::execute("rm", QStringList() << "-r" << RDS_DNS_CONFIGDIR_BACKUP);
	return true;
}

ReturnValue RdsDnsManager::zones() const
{
	RDS_LOCK;
	return QStringList(qxt_d().data->zones.keys());
}

ReturnValue RdsDnsManager::addZone(const QString &z)
{
	RDS_LOCK;
	QString zone = (z.endsWith(".") ? z : z + '.');

	if (qxt_d().data->zones.contains(zone))
	{
		return ReturnValue(1, "A zone by that name already exists");
	}
	QVariantMap zoneData;
	RdsDnsZone * newzone = new RdsDnsZone();
	newzone->qxt_d().data->valid = true;
	newzone->qxt_d().data->domainName = zone;
	newzone->setDomainName(zone);
	if (!QFile::exists(RDS_DNS_ZONEDIR + zone + "db"))
	{
		newzone->setPrimaryNameServer(zone);
		newzone->setAdminNameServer("admin." + zone);
		newzone->setSoaValues(QStringList() << "1" << "28800" << "3600" << "604800" << "38400");
// 		type
	}
	else
	{
		QFile file(RDS_DNS_ZONEDIR + zone + "db");
		if (!file.open(QFile::ReadOnly))
		{
			delete newzone;
			return ReturnValue(1, "Failed to open zone configuration file, " + file.fileName());
		}
		else
		{
			newzone->parse(file.readAll());
		}
	}
	qxt_d().data->zones[zone].values["type"] = "master";
	qxt_d().data->zones[zone].values["file"] = RDS_DNS_ZONEDIR + zone + "db";
	qxt_d().data->zones[zone].zone = zone;
	qxt_d().data->zones[zone].zoneKey = "\"" + zone + "\"";
	qxt_d().data->values = insertData(qxt_d().data->values, QStringList() << "zone" << "\"" + zone + "\"" << "type", "master");
	qxt_d().data->values = insertData(qxt_d().data->values, QStringList() << "zone" << "\"" + zone + "\"" << "file", RDS_DNS_ZONEDIR + zone + "db");
	return newzone;
}

ReturnValue RdsDnsManager::removeZone(const QString &zone)
{
	RDS_LOCK;
	if (!qxt_d().data->zones.contains(zone))
		return ReturnValue(ReturnValue::GenericWarning, "A zone by that name cannot be found");
	if (QFile::exists(qxt_d().data->zones.value(zone).values.value("file").toString()))
		QFile::remove(qxt_d().data->zones.value(zone).values.value("file").toString());

	qxt_d().data->zones.remove(zone);
// 	qDebug() << "I just removed" << zone << qxt_d().data->zones.keys();
	RdsDnsZonePrivate::dataMutex().lock();

	QVariantMap data = qxt_d().data->values.value("zone").toMap();
	for (QVariantMap::iterator i = data.begin(); i != data.end(); ++i)
	{
		if (i.key().startsWith("\"" + zone + "\"") ||
		        i.key() == zone ||
		        i.key().startsWith(zone + " "))
		{
			i = data.erase(i);
			i--;
		}
	}
	if (data.count() > 0)
		qxt_d().data->values["zone"] = data;
	else
		qxt_d().data->values.remove("zone");
	if (RdsDnsZonePrivate::dataHash().contains(zone))
	{
		RdsDnsZonePrivate::dataHash().value(zone)->removed();
	}
	RdsDnsZonePrivate::dataMutex().unlock();
	return true;
}

ReturnValue RdsDnsManager::zone(const QString &z) const
{
	RDS_LOCK;
	QString zone = (z.endsWith(".") ? z : z + '.');
	if (!qxt_d().data->zones.contains(zone))
	{
		return ReturnValue(ReturnValue::GenericError, zone + " must be added before being opened.");
	}

	if (qxt_d().data->zones[zone].values["type"] != "master")
	{
		return ReturnValue(ReturnValue::GenericError, zone + " is a slave zone.");
	}

	QFile file(qxt_d().data->zones.value(zone).values.value("file").toString());
	if (file.fileName().isEmpty())
		return ReturnValue(ReturnValue::GenericError, "Failed to figure out where the zone file is.");

	if (!file.open(QFile::ReadOnly))
		return ReturnValue(ReturnValue::GenericError, "Failed to open zone configuration file:" + qxt_d().data->zones.value(zone).values.value("file").toString());

	RdsDnsZone* zoneObject = new RdsDnsZone();
	ReturnValue ret = zoneObject->parse(file.readAll());
	if (ret.isError())
	{
		delete zoneObject;
		return ret;
	}
	return zoneObject;
}

ReturnValue RdsDnsManager::forwarders() const
{
	RDS_LOCK;
	return qxt_d().data->forwarders;
}

ReturnValue RdsDnsManager::setForwarders(const QStringList& f)
{
	RDS_LOCK;
	qxt_d().data->forwarders = f;
	return true;
}

ReturnValue RdsDnsManager::listValues() const
{
	RDS_LOCK;
	return QStringList(qxt_d().data->values.keys());
}

ReturnValue RdsDnsManager::containsValue(const QString& key) const
{
	RDS_LOCK;
	return qxt_d().data->values.contains(key);
}

ReturnValue RdsDnsManager::value(const QString& key) const
{
	RDS_LOCK;
	return qxt_d().data->values.value(key);
}

ReturnValue RdsDnsManager::setValue(const QString& key, const QString& value)
{
	RDS_LOCK;
	qxt_d().data->values.insert(key, value);
	return true;
}

RdsDnsManager& RdsDnsManager::operator=(const RdsDnsManager & other)
{
	Q_UNUSED(other);
	return *this;
}

ReturnValue RdsDnsManager::prepareToSave()
{
	qxt_d().data->values = insertData(qxt_d().data->values, QStringList() << "options" << "tkey-domain",
	                                  "\"" + RdsUtils::realm().toUpper() + "\"");
	qxt_d().data->values = insertData(qxt_d().data->values, QStringList() << "options" << "tkey-gssapi-credential",
	                                  "\"DNS/" + RdsUtils::realm().toLower() + "\"");

	if (QFile::exists(RDS_SAMBA_PRIVATE + "named.conf.update"))
	{
		system(qPrintable(QString("chown bind.bind %1named.conf.update").arg(RDS_SAMBA_PRIVATE)));
		for (QHash<QString, RdsDnsManagerData::RdsZoneInfo>::const_iterator i = qxt_d().data->zones.constBegin(); i != qxt_d().data->zones.constEnd(); ++i)
		{
			if (i.value().values["type"] != "master")
				continue;
			qxt_d().data->values = insertData(qxt_d().data->values, QStringList() << "zone" << i.value().zoneKey << "include",
			                                  QString("\"" + RDS_SAMBA_PRIVATE + "named.conf.update\"").replace("//", "/"));
		}
	}

	QFile file("/etc/default/bind9");
	if (!file.open(QFile::ReadOnly))
	{
		qWarning() << "Failed to open /etc/default/bind9 for reading!";
		return(false); //non fatal error
	}

	QStringList lines;
	bool foundstart = false;

	while (!file.atEnd())
	{
		QString line = file.readLine().replace("\n", "");

		if (line == "#RDS START") foundstart = true;
		if (line == "#RDS END")
		{
			foundstart = false;
			continue;
		}
		if (foundstart) continue;

		lines << line;
	}

	file.close();

	if (!file.open(QFile::WriteOnly | QFile::Truncate))
	{
		qWarning() << "Failed to open /etc/default/bind9 for writing!";
		return(false); //non fatal error
	}

	foreach(QString line, lines)
	{
		file.write(line.toAscii());
		file.write("\n");
	}

	file.write(QString("#RDS START\n"
	                   "KEYTAB_FILE='%1dns.keytab'\n"
	                   "KRB5_KTNAME='%2dns.keytab'\n"
	                   "export KEYTAB_FILE\n"
	                   "export KRB5_KTNAME\n"
	                   "#RDS END\n").arg(RDS_SAMBA_PRIVATE).arg(RDS_SAMBA_PRIVATE).toAscii());

	file.close();

	system(qPrintable(QString("chown bind.bind %1dns.keytab").arg(RDS_SAMBA_PRIVATE)));

	return(true);
}

ReturnValue RdsDnsManager::zoneType(const QString &z) const
{
	RDS_LOCK;
	QString zone = (z.endsWith(".") ? z : z + '.');
	if (!qxt_d().data->zones.contains(zone))
	{
		return ReturnValue(ReturnValue::GenericError, zone + " must be added before being opened.");
	}

	return(qxt_d().data->zones[zone].values["type"]);
}

