/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU Lesser General Public License as         *
 *  published by the Free Software Foundation; either version 2 of the     *
 *  License, or (at your option) any later version.                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  Lesser General Public License for more details.                        *
 *                                                                         *
 *  You should have received a copy of the GNU Lesser General Public       *
 *  License along with this program; if not, write to the                  *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/
#include "rdsmountpoint.h"
#include "rdsmountpoint_p.h"

#include <QFile>
#include <QTemporaryFile>
#include <QProcess>
#include <QFileInfo>
#include <QDebug>
#include <RdsVolume>
#include <RdsUtils>

using namespace QtRpc;

QTRPC_SERVICEPROXY_PIMPL_IMPLEMENT(RdsMountPoint);

RdsMountPoint::RdsMountPoint()
		: ServiceProxy(NULL)
{
	QXT_INIT_PRIVATE(RdsMountPoint);
}

RdsMountPoint::RdsMountPoint(const char *mountPoint)
{
	QXT_INIT_PRIVATE(RdsMountPoint);
	qxt_d().mountPoint = RdsUtils::normalizePath(mountPoint);
}

RdsMountPoint::RdsMountPoint(const QString &mountPoint)
		: ServiceProxy(NULL)
{
	QXT_INIT_PRIVATE(RdsMountPoint);
	qxt_d().mountPoint = RdsUtils::normalizePath(mountPoint);
}

RdsMountPoint::RdsMountPoint(const RdsMountPoint &other)
		: ServiceProxy(NULL)
{
	QXT_INIT_PRIVATE(RdsMountPoint);
	qxt_d().mountPoint = RdsUtils::normalizePath(other.qxt_d().mountPoint);
}

RdsMountPoint::~RdsMountPoint()
{
}

ReturnValue RdsMountPoint::volume()
{
	QTemporaryFile file;
	file.open();
	QString tmpfile = file.fileName();
	file.close();
	QProcess::execute("cp", QStringList() << "/proc/mounts" << tmpfile);
	if (!file.open())
	{
		return ReturnValue(1, "Failed to open /proc/mounts!");
	}
	QString line;
	while (!file.atEnd())
	{
		line = file.readLine();
		QString mountPoint = line;
		mountPoint = mountPoint.mid(mountPoint.indexOf(" ") + 1);
		mountPoint = mountPoint.left(mountPoint.indexOf(" "));
		if (mountPoint == qxt_d().mountPoint)
			break;
		if (QFile::exists(mountPoint))
		{
			// Check if it's a symlink
			QFileInfo info(mountPoint);
			mountPoint = info.canonicalFilePath();
			if (mountPoint == qxt_d().mountPoint)
				break;
		}
		line = QString();
	}
	if (line.isNull())
		return ReturnValue(1, "Device is not mounted");
	QString volume = line;
	volume = volume.left(volume.indexOf(" "));
	return(new RdsVolume(volume));
}

ReturnValue RdsMountPoint::path() const
{
	return qxt_d().mountPoint;
}

ReturnValue RdsMountPoint::size() const
{
	if (!QFile::exists(qxt_d().mountPoint))
		return ReturnValue("The mount point no longer exists.");
	struct statfs buff;
	if (statfs(qPrintable(qxt_d().mountPoint), &buff) != 0)
		return ReturnValue(1, strerror(errno));

	quint64 value = 0xC0000000;
	value = value << 32;
	if ((static_cast<quint64>(buff.f_blocks)) & value)
		return ReturnValue(1, "Size is too large to be displayed in a 64 bit integer");
	return (static_cast<quint64>(buff.f_blocks) << 2);
}

ReturnValue RdsMountPoint::freeSpace() const
{
	if (!QFile::exists(qxt_d().mountPoint))
		return ReturnValue("The mount point no longer exists.");
	struct statfs buff;
	if (statfs(qPrintable(qxt_d().mountPoint), &buff) != 0)
		return ReturnValue(1, strerror(errno));

	quint64 value = 0xC0000000;
	value = value << 32;
	if ((static_cast<quint64>(buff.f_blocks)) & value)
		return ReturnValue(1, "Size is too large to be displayed in a 64 bit integer");
	return (static_cast<qlonglong>(buff.f_bfree) << 2);
}

ReturnValue RdsMountPoint::supportsNtAcls() const
{
	if (!QFile::exists(qxt_d().mountPoint))
		return ReturnValue("The mount point no longer exists.");
	if (listxattr(qPrintable(qxt_d().mountPoint), 0, 0) == -1)
	{
		if (errno == ENOTSUP)
			return false;
	}
	return true;
}

QString RdsMountPoint::mountPointOfPath(const QString &str)
{
	QTemporaryFile file;
	QString line;
	QString mountPoint;
	file.open();
	QString tmpfile = file.fileName();
	file.close();
	QProcess::execute("cp", QStringList() << "/proc/mounts" << tmpfile);
	if (!file.open())
	{
		return RdsUtils::normalizePath(str);
	}
	while (!file.atEnd())
	{
		line = file.readLine();
		line = line.mid(line.indexOf(" ") + 1);
		line = line.left(line.indexOf(" "));
		line = RdsMountPointPrivate::decodeMountPath(line);
		if (line.count() > mountPoint.count() && str.startsWith(line))
			mountPoint = line;
	}
	
	if(mountPoint != "" && mountPoint != "/") return RdsUtils::normalizePath(mountPoint);
	
	QFile fsfile("/etc/fstab");
	if (!fsfile.open(QFile::ReadOnly))
	{
		return str;
	}
	
	while (!fsfile.atEnd())
	{
		line = fsfile.readLine();
		line = line.simplified();
		if(line.startsWith("#")) continue;
		
		line = line.mid(line.indexOf(" ") + 1);
		line = line.left(line.indexOf(" "));
		line = RdsMountPointPrivate::decodeMountPath(line);
		if (line.count() > mountPoint.count() && str.startsWith(line))
			mountPoint = line;
	}
	
	if(mountPoint == "") mountPoint = "/";
	return RdsUtils::normalizePath(mountPoint);
}

QString RdsMountPointPrivate::encodeMountPath(QString path)
{
	path = path.replace("\\","\\134");
	path = path.replace(" ","\\040");
	path = path.replace("\t","\\011");
	
	return(path);
}

QString RdsMountPointPrivate::decodeMountPath(QString path)
{
	QByteArray ascii = path.toAscii();
	const char *data = ascii.constData();
	QString result;
	
	
	for(int i=0; i < path.size(); i++)
	{
		if(data[i] == '\\')
		{
			//If there isn't enough string left for an escape sequence, just return what we've got
			if((i+4) >= path.size()) return(result);
			
			QString tmp;
			tmp += data[i+1];
			tmp += data[i+2];
			tmp += data[i+3];
			
			bool ok=false;
			int escape = tmp.toInt(&ok, 8);
			
			if(ok)
			{
				result += (char)escape;
			}
			
			i += 3; //skip over the remaining escape sequence
		}
		else
		{
			result += data[i];
		}
	}
	
	return(result);
}

RdsMountPoint &RdsMountPoint::operator=(const RdsMountPoint & object)
{
	qxt_d().mountPoint = object.qxt_d().mountPoint;
	return(*this);
}

ReturnValue RdsMountPoint::isMounted()
{
	QTemporaryFile file;
	QString line;
	file.open();
	QString tmpfile = file.fileName();
	file.close();
	QProcess::execute("cp", QStringList() << "/proc/mounts" << tmpfile);
	if (!file.open())
	{
		return false;
	}
	while (!file.atEnd())
	{
		line = file.readLine();
		line = line.mid(line.indexOf(" ") + 1);
		line = line.left(line.indexOf(" "));
		line = RdsMountPointPrivate::decodeMountPath(line);
		if (line == qxt_d().mountPoint)
			return true;
	}
	
	return false;
}

ReturnValue RdsMountPoint::mount()
{
	QProcess p;
	p.setProcessChannelMode(QProcess::MergedChannels);
	p.start("mount", QStringList() << qxt_d().mountPoint + "/");
	if (!p.waitForFinished(25000))
	{
		qWarning() << p.readAll();
		return(ReturnValue(1, "Mount timed out"));
	}

	if (p.exitCode() != 0)
	{
		return(ReturnValue(1, p.readAll()));
	}

	return(true);
}

ReturnValue RdsMountPoint::unMount()
{
	QProcess p;
	p.setProcessChannelMode(QProcess::MergedChannels);
	p.start("umount", QStringList() << qxt_d().mountPoint + "/");
	if (!p.waitForFinished(25000))
	{
		qWarning() << p.readAll();
		return(ReturnValue(1, "Unmount timed out"));
	}

	if (p.exitCode() != 0)
	{
		return(ReturnValue(1, p.readAll()));
	}

	return(true);
}
