/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU Lesser General Public License as         *
 *  published by the Free Software Foundation; either version 2 of the     *
 *  License, or (at your option) any later version.                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  Lesser General Public License for more details.                        *
 *                                                                         *
 *  You should have received a copy of the GNU Lesser General Public       *
 *  License along with this program; if not, write to the                  *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/
 
#include "config.h"
#include "rdsnetworksettings.h"
#include "rdsnetworksettings_p.h"
#include <QProcess>
#include <QFile>
#include <QTextStream>
#include <QDebug>
#include <QTimer>
#include <QMetaObject>
#include <RdsUtils>

using namespace QtRpc;

QTRPC_SERVICEPROXY_PIMPL_IMPLEMENT(RdsNetworkSettings);

RdsNetworkSettings::RdsNetworkSettings(QObject *parent)
		: ServiceProxy(parent)
{
	QXT_INIT_PRIVATE(RdsNetworkSettings);
}


RdsNetworkSettings::RdsNetworkSettings(const RdsNetworkSettings &other)
		: ServiceProxy(NULL)
{
	Q_UNUSED(other);
}

RdsNetworkSettings::~RdsNetworkSettings()
{
}

RdsNetworkSettings &RdsNetworkSettings::operator=(const RdsNetworkSettings & other)
{
	Q_UNUSED(other);
	return(*this);
}

ReturnValue RdsNetworkSettings::auth(QtRpc::AuthToken token)
{
	if (token.serverData().contains("authenticated") && (token.serverData().value("authenticated").toBool() == true))
		return(true);
	else
		return(ReturnValue(1, "Not Authenticated"));
}

ReturnValue RdsNetworkSettings::loadNetworkConfig()
{
	QFile file(RDS_NETWORK_INTERFACES);
	if (!file.open(QFile::ReadOnly))
	{
		return(ReturnValue(1, QString("Failed to open ") + RDS_NETWORK_INTERFACES + " for reading"));
	}

	qxt_d().interfaces.clear();

	QTextStream stream(&file);
	QString line;
	QString comment;
	QString interface;
	QStringList autointerfaces;
	QMap<QString, QString> options;
	QString type;

	while ((line = stream.readLine()) != QString::Null())
	{
		QStringList parts = line.split(" ", QString::SkipEmptyParts);

		if (parts.size() == 0) continue;

		if (line.startsWith("auto"))
		{
			if (parts.size() >= 2) autointerfaces << parts[1];
		}
	}

	stream.seek(0);

	while ((line = stream.readLine()) != QString::Null())
	{
		if (line.startsWith("#"))
		{
			comment = line;
			continue;
		}
		int index = line.indexOf("#");
		if (index >= 0) line = line.mid(0, index - 1);

		QStringList parts = line.split(" ", QString::SkipEmptyParts);

		if (parts.size() == 0) continue;


		if (line.startsWith("auto"))
		{
			//do nothing here
		}
		else if (line.startsWith("iface"))
		{
			if (interface != "")
			{
				setInterface(interface, type, autointerfaces.contains(interface), options, "");
				options.clear();
			}
			if (parts.size() < 4) continue;
			interface = parts[1];
			type = parts[3];
			comment = "";
		}
		else
		{
			QString name = parts[0];
			parts.removeAt(0);
			options[name] = parts.join(" ");
		}
	}

	setInterface(interface, type, autointerfaces.contains(interface), options, comment); //add the last interface

	file.close();

	return(true);
}

ReturnValue RdsNetworkSettings::saveNetworkConfig()
{
	QFile file(RDS_NETWORK_INTERFACES);
	if (!file.open(QFile::WriteOnly | QFile::Truncate))
	{
		return(ReturnValue(1, QString("Failed to open ") + RDS_NETWORK_INTERFACES + " for writing"));
	}

	QTextStream stream(&file);

	stream << "# This file describes the network interfaces available on your system" << endl;
	stream << "# and how to activate them. For more information, see interfaces(5)." << endl;
	stream << endl;

	foreach(Interface iface, qxt_d().interfaces)
	{
		if (iface.comment() != "") stream << iface.comment() << endl;
		if (iface.autoUp()) stream << "auto " << iface.name() << endl;
		stream << "iface " << iface.name() << " inet " << iface.type() << endl;
		foreach(QString option, iface.options().keys())
		{
			stream << option << " " << iface.options()[option] << endl;
		}
		stream << endl;
	}

	file.close();

	return(true);
}

ReturnValue RdsNetworkSettings::setInterface(const QString &name, const QString &type, bool autoup, const QMap<QString, QString> &options, const QString &comment)
{
	Interface iface;
	iface.setName(name);
	iface.setType(type);
	iface.setAutoUp(autoup);
	iface.options() = options;
	iface.setComment(comment);

	qxt_d().interfaces[name] = iface;
	return(true);
}

ReturnValue RdsNetworkSettings::removeInterface(const QString &name)
{
	if (!qxt_d().interfaces.keys() .contains(name)) return(false);
	qxt_d().interfaces.remove(name);
	return(true);
}

ReturnValue RdsNetworkSettings::listInterfaces() const
{
	return(QStringList(qxt_d().interfaces.keys()));
}

QList<RdsNetworkSettings::Interface>RdsNetworkSettings::getInterfaces()
{
	return(qxt_d().interfaces.values());
}

ReturnValue RdsNetworkSettings::ifUp(const QString &interface)
{
	QProcess p;
	p.start("ifup", QStringList() << "--force" << interface);
	if (!p.waitForFinished(30000)) return(ReturnValue(1, "ifup timed out " + interface));
	if (p.exitCode() == 0) return(true);
	else return(ReturnValue(1, "ifup failed " + interface));
}

ReturnValue RdsNetworkSettings::ifDown(const QString &interface)
{
	QProcess p;
	p.start("ifdown", QStringList() << "--force" << interface);
	if (!p.waitForFinished(30000)) return(ReturnValue(1, "ifdown timed out " + interface));
	if (p.exitCode() == 0) return(true);
	else return(ReturnValue(1, "ifdown failed " + interface));
}

ReturnValue RdsNetworkSettings::allIfUp()
{
	ReturnValue ret = listAllInterfaces();
	if (ret.isError()) return(true);
	foreach(QString iface, ret.toString())
	{
		if (iface.startsWith("eth")) ifUp(iface);
	}
	return(true);
}

ReturnValue RdsNetworkSettings::allIfDown()
{
	ReturnValue ret = listAllInterfaces();
	if (ret.isError()) return(true);
	foreach(QString iface, ret.toString())
	{
		if (iface.startsWith("eth")) ifDown(iface);
	}
	return(true);
}

RdsNetworkSettings::Interface RdsNetworkSettings::getInterface(QString interface)
{
	Interface iface;
	if (qxt_d().interfaces.keys().contains(interface))
		return(qxt_d().interfaces[interface]);
	else
		return(iface);
}

ReturnValue RdsNetworkSettings::getInterfaceLinkState(const QString &interface) const
{
	QProcess proc;
	proc.start("ethtool", QStringList() << interface);
	if (!proc.waitForFinished()) return(ReturnValue(1, "ethtool timed out " + interface));
	if (proc.exitStatus() != 0) return(ReturnValue(1, "ethtool failed " + interface));

	QTextStream stream(&proc);

	QString line;
	while ((line = stream.readLine()) != QString::Null())
	{
		if (line.contains("Link detected:"))
		{
			QStringList parts = line.split(" ", QString::SkipEmptyParts);
			if (parts.count() >= 3)
			{
				if (parts[2].toLower() == "yes") return(true);
				else return(false);
			}
		}
	}

	return(ReturnValue(1, "No link status information " + interface));
}

ReturnValue RdsNetworkSettings::getInterfaceLinkSpeed(const QString &interface) const
{
	QProcess proc;
	proc.start("ethtool", QStringList() << interface);
	if (!proc.waitForFinished()) return(ReturnValue(1, "ethtool timed out " + interface));
	if (proc.exitStatus() != 0) return(ReturnValue(1, "ethtool failed " + interface));

	QTextStream stream(&proc);

	QString line;
	while ((line = stream.readLine()) != QString::Null())
	{
		if (line.contains("Speed:"))
		{
			QStringList parts = line.split(" ", QString::SkipEmptyParts);
			if (parts.count() >= 2)
			{
				return(parts[1].replace("Mb/s", "").toInt());
			}
		}
	}

	return(ReturnValue(1, "No link speed information " + interface));
}

ReturnValue RdsNetworkSettings::getInterfaceIp(QString interface) const
{
	QProcess proc;
	proc.start("ifconfig", QStringList() << interface);
	if (!proc.waitForFinished()) return(ReturnValue(1, "ifconfig timed out " + interface));
	if (proc.exitStatus() != 0) return(ReturnValue(1, "ifconfig failed " + interface));

	QTextStream stream(&proc);

	QString line;
	while ((line = stream.readLine()) != QString::Null())
	{
		if (line.contains("inet addr:"))
		{
			QStringList parts = line.split(" ", QString::SkipEmptyParts);
			if (parts.count() >= 2)
				return(parts[1].replace("addr:", ""));
		}
	}

	return(ReturnValue(1, "invalid interface " + interface));
}

ReturnValue RdsNetworkSettings::getInterfaceNetmask(QString interface) const
{
	QProcess proc;
	proc.start("ifconfig", QStringList() << interface);
	if (!proc.waitForFinished()) return(ReturnValue(1, "ifconfig timed out " + interface));
	if (proc.exitStatus() != 0) return(ReturnValue(1, "ifconfig failed " + interface));

	QTextStream stream(&proc);

	QString line;
	while ((line = stream.readLine()) != QString::Null())
	{
		if (line.contains("inet addr:"))
		{
			QStringList parts = line.split(" ", QString::SkipEmptyParts);
			if (parts.count() >= 4)
				return(parts[3].replace("Mask:", ""));
		}
	}

	return(ReturnValue(1, "invalid interface " + interface));
}

ReturnValue RdsNetworkSettings::getInterfaceMac(QString interface) const
{
	QProcess proc;
	proc.start("ifconfig", QStringList() << interface);
	if (!proc.waitForFinished()) return(ReturnValue(1, "ifconfig timed out " + interface));
	if (proc.exitStatus() != 0) return(ReturnValue(1, "ifconfig failed " + interface));

	QTextStream stream(&proc);

	QString line;
	while ((line = stream.readLine()) != QString::Null())
	{
		if (line.contains("Link encap:Ethernet"))
		{
			QStringList parts = line.split(" ", QString::SkipEmptyParts);
			if (parts.count() >= 5)
				return(parts[4]);
		}
	}

	return(ReturnValue(1, "invalid interface " + interface));
}

ReturnValue RdsNetworkSettings::listAllInterfaces() const
{
	QStringList interfaces;
	QProcess proc;
	proc.start("ifconfig", QStringList() << "-a");
	if (!proc.waitForFinished()) return(ReturnValue(1, "ifconfig timed out"));
	if (proc.exitStatus() != 0) return(ReturnValue(1, "ifconfig failed"));

	QTextStream stream(&proc);

	QString line;
	while ((line = stream.readLine()) != QString::Null())
	{
		if (line.contains("Link encap:Ethernet"))
		{
			QStringList parts = line.split(" ");
			interfaces << parts[0];
		}
	}

	return(interfaces);
}

ReturnValue RdsNetworkSettings::getDefaultInterface() const
{
	QProcess proc;
	proc.start("route", QStringList() << "-n");
	if (!proc.waitForFinished()) return(ReturnValue(1, "route timed out"));
	if (proc.exitStatus() != 0) return(ReturnValue(1, "route failed"));

	QTextStream stream(&proc);

	QString line;
	while ((line = stream.readLine()) != QString::Null())
	{
		QStringList parts = line.split(QRegExp("[\\s]+"), QString::SkipEmptyParts);
		if(parts.size() != 8) continue;
		if((parts[3] == "UG") && (parts[2] == "0.0.0.0")) return(parts[7]);
	}

	return(ReturnValue(1,"Failed to get default interface"));
}

ReturnValue RdsNetworkSettings::getDefaultGateway() const
{
	QProcess proc;
	proc.start("route", QStringList() << "-n");
	if (!proc.waitForFinished()) return(ReturnValue(1, "route timed out"));
	if (proc.exitStatus() != 0) return(ReturnValue(1, "route failed"));

	QTextStream stream(&proc);

	QString line;
	while ((line = stream.readLine()) != QString::Null())
	{
		QStringList parts = line.split(QRegExp("[\\s]+"), QString::SkipEmptyParts);
		if(parts.size() != 8) continue;
		if((parts[3] == "UG") && (parts[2] == "0.0.0.0")) return(parts[1]);
	}

	return(ReturnValue(1,"Failed to get default gateway"));
}

ReturnValue RdsNetworkSettings::getDnsServers() const
{
	QStringList servers;
	QFile file("/etc/resolv.conf");
	if(!file.open(QFile::ReadOnly))
		return(ReturnValue(1,"Failed to open /etc/resolv.conf for reading"));
	
	QTextStream stream(&file);

	QString line;
	while ((line = stream.readLine()) != QString::Null())
	{
		QStringList parts = line.split(QRegExp("[\\s]+"), QString::SkipEmptyParts);
		if(parts.size() != 2) continue;
		
		if((parts[0] == "nameserver") && (parts[1] != "127.0.0.1"))
			servers << parts[1];
	}

	return(servers);
}

ReturnValue RdsNetworkSettings::getDomain() const
{
	QFile file("/etc/resolv.conf");
	if(!file.open(QFile::ReadOnly))
		return(ReturnValue(1,"Failed to open /etc/resolv.conf for reading"));
	
	QTextStream stream(&file);

	QString line;
	while ((line = stream.readLine()) != QString::Null())
	{
		QStringList parts = line.split(QRegExp("[\\s]+"), QString::SkipEmptyParts);
		if(parts.size() != 2) continue;
		
		if((parts[0] == "domain") || (parts[0] == "search"))
			return(parts[1]);
	}

	return(ReturnValue(1,"Failed to find domain name"));
}

ReturnValue RdsNetworkSettings::killDhcpClient()
{
	RdsUtils::runCommand("killall", QStringList() << "dhclient");
	RdsUtils::runCommand("killall", QStringList() << "dhclient3");
	return(true); //if they weren't killed, its not an error
}

RdsNetworkSettings::Interface::Interface()
{

}

RdsNetworkSettings::Interface::Interface(const Interface &other)
{
	qxt_d().name = other.qxt_d().name;
	qxt_d().type = other.qxt_d().type;
	qxt_d().autoup = other.qxt_d().autoup;
	qxt_d().options = other.qxt_d().options;
	qxt_d().comment = other.qxt_d().comment;
}

RdsNetworkSettings::Interface::~Interface()
{

}

RdsNetworkSettings::Interface &RdsNetworkSettings::Interface::operator=(const Interface & other)
{
	qxt_d().name = other.qxt_d().name;
	qxt_d().type = other.qxt_d().type;
	qxt_d().autoup = other.qxt_d().autoup;
	qxt_d().options = other.qxt_d().options;
	qxt_d().comment = other.qxt_d().comment;
	return(*this);
}

QString RdsNetworkSettings::Interface::name() const
{
	return(qxt_d().name);
}

void RdsNetworkSettings::Interface::setName(const QString &name)
{
	qxt_d().name = name;
}

QString RdsNetworkSettings::Interface::type() const
{
	return(qxt_d().type);
}

void RdsNetworkSettings::Interface::setType(const QString &type)
{
	qxt_d().type = type;
}

bool RdsNetworkSettings::Interface::autoUp() const
{
	return(qxt_d().autoup);
}

void RdsNetworkSettings::Interface::setAutoUp(bool up)
{
	qxt_d().autoup = up;
}

QMap<QString, QString> &RdsNetworkSettings::Interface::options()
{
	return(qxt_d().options);
}

const QMap<QString, QString> &RdsNetworkSettings::Interface::options() const
{
	return(qxt_d().options);
}

const QMap<QString, QString> &RdsNetworkSettings::Interface::optionsConst() const
{
	return(qxt_d().options);
}

QString RdsNetworkSettings::Interface::comment()
{
	return(qxt_d().comment);
}

void RdsNetworkSettings::Interface::setComment(const QString &comment)
{
	qxt_d().comment = comment;
}

QDataStream& operator<<(QDataStream& d, const RdsNetworkSettings::Interface& iface)
{
	d << iface.qxt_d().name;
	d << iface.qxt_d().type;
	d << iface.qxt_d().autoup;
	d << iface.qxt_d().options;
	d << iface.qxt_d().comment;
	return(d);
}

QDataStream& operator>>(QDataStream& d, RdsNetworkSettings::Interface& iface)
{
	d >> iface.qxt_d().name;
	d >> iface.qxt_d().type;
	d >> iface.qxt_d().autoup;
	d >> iface.qxt_d().options;
	d >> iface.qxt_d().comment;
	return(d);
}
