/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU Lesser General Public License as         *
 *  published by the Free Software Foundation; either version 2 of the     *
 *  License, or (at your option) any later version.                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  Lesser General Public License for more details.                        *
 *                                                                         *
 *  You should have received a copy of the GNU Lesser General Public       *
 *  License along with this program; if not, write to the                  *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/
#include "config.h"
#include "rdssharemanager.h"
#include "rdssharemanager_p.h"
#include <QFile>
#include <QString>
#include <QStringList>
#include <RdsShare>
#include <RdsEntity>
#include <QDir>
#include <RdsUtils>
#include <QDebug>
#include <RdsLdapSession>
#include <RdsAdObject>
#include <QFileInfo>
#include <QSettings>
#include <QTextCodec>
#include <QMutexLocker>
#include <RdsVolume>
#include <RdsDaemonManager>
#include <RdsNtAcl>

QMutex RdsShareManagerPrivate::volmutex;
RdsVolumeManager *RdsShareManagerPrivate::volmgr = NULL;
QHash<QString, QString> RdsShareManagerPrivate::volumepaths;

QTRPC_SERVICEPROXY_PIMPL_IMPLEMENT(RdsShareManager);

RdsShareManager::RdsShareManager()
		: RdsEntityManager(NULL)
{
	QXT_INIT_PRIVATE(RdsShareManager);
	RdsShareManagerPrivate::connectVolumeEvents(this);
}

RdsShareManager::RdsShareManager(const QString &fileName)
		: RdsEntityManager(NULL)
{
	QXT_INIT_PRIVATE(RdsShareManager);
	qxt_d().parser.setFileName(fileName);
	RdsShareManagerPrivate::connectVolumeEvents(this);
}

RdsShareManager::RdsShareManager(const RdsShareManager &other)
		: RdsEntityManager(NULL)
{
	QXT_INIT_PRIVATE(RdsShareManager);
	qxt_d().parser = other.qxt_d().parser;
	RdsShareManagerPrivate::connectVolumeEvents(this);
}

RdsShareManager::~RdsShareManager()
{
}

RdsShareManager& RdsShareManager::operator=(const RdsShareManager & other)
{
	qxt_d().parser = other.qxt_d().parser;
	return *this;
}

ReturnValue RdsShareManager::fileName() const
{
	return qxt_d().parser.fileName();
}

ReturnValue RdsShareManager::setFileName(const QString &fileName)
{
	return(qxt_d().parser.setFileName(fileName));
}

ReturnValue RdsShareManager::listShares() const
{
	ReturnValue ret = qxt_d().parser.listSections();
	if (ret.isError()) return(ret);
	QStringList list = ret.toStringList();
	list.removeAll("globals");
	list.removeAll("global");
	list.removeAll("printers");
	list.removeAll("print$");
	return list;
}

ReturnValue RdsShareManager::removeShare(const QString &share)
{
	ReturnValue ret = qxt_d().parser.removeSection(share);
	return(ret);
}

ReturnValue RdsShareManager::removeShare(const RdsShare &share)
{
	ReturnValue ret = share.name();
	if (ret.isError()) return(ret);

	return(removeShare(ret.toString()));
}

ReturnValue RdsShareManager::addShare(const QString &share, const QString &path)
{
	if (!QFile::exists(path))
		return ReturnValue(1, "The path, " + path + ", does not exist");

	if(listShares().toStringList().contains(share))
		return ReturnValue(1, QString("The share %1 already exists.").arg(share));
	
	ReturnValue ret = qxt_d().parser.enableSection(share);
	if (ret.isError()) return(ret);
	ret = qxt_d().parser.setValue(share, "path", path);
	if (ret.isError()) return(ret);
	return true;
}

ReturnValue RdsShareManager::share(const QString &str) const
{
	ReturnValue ret = listShares();
	if (ret.isError()) return(ret);
	if (!ret.toStringList().contains(str)) return(ReturnValue(1, "Share does not exist"));

	return(new RdsShare(str));
}

ReturnValue RdsShareManager::auth(QtRpc::AuthToken token)
{
	createInternalObject();
	if (token.serverData().contains("authenticated") && (token.serverData().value("authenticated").toBool() == true))
		return(true);
	else
		return(ReturnValue(1, "Not Authenticated"));
}

ReturnValue RdsShareManager::listEntities(const QString &id, bool loadmore) const
{
	if ((id == "") || (id == "root"))
	{
		RdsEntity entity;
		entity.setId("root");
		entity.setType("root");
		entity.setVisible(false);
		entity.setName("");
		entity.setParent("");

		ReturnValue ret = listShares();
		if (!ret.isError())
		{

			foreach(QString share, ret.toStringList())
			{
				ret = listEntities("@SHARE/" + share + "/", loadmore);
				if (ret.isError()) continue;
				entity.children() << ret.value<RdsEntity>();
			}
		}

		return(QVariant::fromValue<RdsEntity>(entity));
	}
	else if (id.startsWith("@SHARE/") && (id.count('/') == 2) && id.endsWith("/"))
	{
		ReturnValue ret = RdsUtils::getShareName(id);
		if (ret.isError()) return(ret);
		QString sharename = ret.toString();

		ret = share(sharename);
		if (ret.isError()) return(ret);
		RdsShare share;
		share = ret;

		ret = share.path();
		if (ret.isError()) return(ret);
		QString path = ret.toString();

		RdsEntity entity;
		entity.setId(id);
		entity.setType("share");
		entity.setVisible(true);
		entity.setName(sharename);
		entity.setParent("root");
		entity.setGroup(true);
		QString description = share.description().toString();
		if (description != "")
		{
			entity.metadata()["description"] = description;
		}
		else
		{
			QString mapstring = share.mapString().toString();
			mapstring.replace("()", "");
			mapstring.replace("(", ",");
			mapstring.replace(")", "");
			mapstring.replace("$", "");
			QStringList parts = mapstring.split(",", QString::SkipEmptyParts);
			if (parts.size() > 0)
			{
				QString drive = parts[0].trimmed();
				if (parts.size() > 1) drive = drive + " for ";
				//qDebug() << "Parts:" << parts;
				parts.removeAt(0);
				QStringList tmp;
				foreach(QString t, parts)
				{
					tmp << t.trimmed();
				}
				mapstring = drive + tmp.join(", ");
				entity.metadata()["description"] = mapstring;
			}
		}

		if (loadmore)
		{
			QDir dir(path);

			foreach(QString child, dir.entryList(QStringList(), QDir::QDir::AllDirs))
			{
				if ((child == "..") || (child == ".")) continue;

				QString childid = QString("@SHARE/%1/%2").arg(sharename).arg(child);


				ret = listEntities(childid, false);
				if (ret.isError()) continue;
				entity.children() << ret.value<RdsEntity>();
			}
		}
		else
		{
			entity.setHasMore(true);
		}

		return(QVariant::fromValue<RdsEntity>(entity));
	}
	else if (id.startsWith("@SHARE/"))
	{
		ReturnValue ret = RdsUtils::getShareName(id);
		if (ret.isError()) return(ret);
		QString sharename = ret.toString();

		ret = share(sharename);
		if (ret.isError()) return(ret);

		RdsShare share;
		share = ret;

		ret = share.path();
		if (ret.isError()) return(ret);
		QString sharepath = RdsUtils::normalizePath(ret.toString());

		QString path = RdsUtils::normalizePath(sharepath + "/" + RdsUtils::getRelativePath(id).toString());

		QDir dir(path);
		QDir parent = dir;
		if (!parent.cdUp()) return(ReturnValue(1, "Cannot find parent"));

		QString parentid;
		if (parent.path() == sharepath)
		{
			parentid = "@SHARE/" + sharename + "/";
		}
		else
		{
			parentid = RdsUtils::normalizePath(id);
			int index = parentid.lastIndexOf("/");
			parentid = parentid.left(index);
		}


		RdsEntity entity;
		entity.setId(id);
		entity.setType("folder");
		entity.setVisible(true);
		entity.setName(dir.dirName());
		entity.setParent(parentid);
		entity.setGroup(true);


		if (loadmore)
		{
			foreach(QString child, dir.entryList(QStringList(), QDir::QDir::AllDirs))
			{
				if ((child == "..") || (child == ".")) continue;

				QString tmp = RdsUtils::normalizePath(id);
				QString childid = QString("%1/%2").arg(tmp).arg(child);

				ret = listEntities(childid, false);
				if (ret.isError()) continue;
				entity.children() << ret.value<RdsEntity>();
			}
		}
		else
		{
			entity.setHasMore(true);
		}

		return(QVariant::fromValue<RdsEntity>(entity));
	}
	else
	{
		return(ReturnValue(1, "Malformed Path: " + id));
	}
}

ReturnValue RdsShareManager::save()
{
	RdsShareManagerPrivate::writeLogonScript(this);
	return(qxt_d().parser.save());
}

void RdsShareManagerPrivate::writeLogonScript(RdsShareManager *mgr)
{
	char hostname[512];
	gethostname(hostname, 512);

	QString path = QString(RDS_SAMBA_SYSVOL) + RdsUtils::realm().toLower() + "/scripts/";

	QFileInfo info(path);
	if (!info.isDir())
	{
		RdsUtils::createDirectory(path, true);
	}

	QFile includefile(RDS_DATA_PATH + "/drivemap.vbs");
	if (!includefile.open(QFile::ReadOnly))
	{
		qWarning() << "Failed to open include file for reading!" << RDS_DATA_PATH + "/drivemap.vbs";
		return;
	}

	QFile file(path + "drivemap-" + hostname + ".vbs");
	if (!file.open(QFile::WriteOnly))
	{
		qWarning() << "Failed to open file for writing:" << path + "/Logon/drivemap-" + hostname +".vbs";
		return;
	}

	file.write("'This file is auto-generated all edits will be lost\r\n\r\n");
	file.write(includefile.readAll());
	file.write("\r\n\r\n");

	foreach(QString sharename, mgr->listShares().toStringList())
	{
		ReturnValue ret = mgr->share(sharename);
		if (ret.isError()) continue;

		RdsShare share = ret;
		if (share.mapString().toString() == "") continue;

		file.write(getMapCommand(share.mapString().toString(), sharename).toAscii());
	}

	file.close();
	includefile.close();

	includefile.setFileName(RDS_DATA_PATH + "/launchapp.vbs");
	if (!includefile.open(QFile::ReadOnly))
	{
		qWarning() << "Failed to open launcher file for reading!" << RDS_DATA_PATH + "/launchapp.vbs";
		return;
	}

	file.setFileName(path + "launchapp-" + hostname + ".vbs");
	if (!file.open(QFile::WriteOnly))
	{
		qWarning() << "Failed to open launchapp file for writing:" << path + "/Logon/drivemap-" + hostname + ".bat";
		return;
	}

	QString tmp = includefile.readAll();
	includefile.close();

	tmp.replace("%%SCRIPT_PATH%%", "\\\\" + QString(hostname) + "\\sysvol\\" +
	            RdsUtils::realm().toLower() + "\\Scripts\\");
	tmp.replace("%%HOSTNAME%%", hostname);

	file.write(tmp.toAscii());
	file.close();

	file.setFileName(path + "launchapp.bat");
	if (!file.open(QFile::WriteOnly))
	{
		qWarning() << "Failed to open launchapp.bat file for writing:" << path + "/Logon/launchapp.bat";
		return;
	}

	QTextStream stream(&file);
	stream << QString("FOR %%F IN (") + "\\\\" + QString(hostname) + "\\sysvol\\" +
	RdsUtils::realm().toLower() + "\\Scripts\\" + "launchapp*.vbs) DO start /MIN %%F\r\n";
	file.close();

	//Fix permissions on the files we just created
	RdsNtAcl::updateInheritedPermissions(path, true);

	QFile::remove(path + "launchapp.vbs");
	QFile::remove(path + "drivemap.vbs");
	
	//updateIniFile(path + "scripts.ini");
}

QStringList RdsShareManagerPrivate::parseList(QString str)
{
	QStringList list;
	bool escape = false;
	QString tmp;

	for (int i = 0; i < str.size(); i++)
	{
		if (escape)
		{
			escape = false;
			tmp += str[i];
		}
		else
		{
			if (str[i] == '\\')
			{
				escape = true;
			}
			else if (str[i] == ',')
			{
				if (tmp.trimmed() == "") continue;
				list << tmp.trimmed();
				tmp = "";
			}
			else
			{
				tmp += str[i];
			}
		}
	}

	if (tmp.trimmed() != "")
	{
		list << tmp.trimmed();
		tmp = "";
	}

	return(list);
}

QString RdsShareManagerPrivate::getMapCommand(QString mapstring, QString sharename)
{
	char hostname[512];
	gethostname(hostname, 512);

	if (mapstring == "") return("");
	else if (mapstring.endsWith(":"))
	{
		return("mapDrive \"" + mapstring + "\", \"\\\\" + hostname + "\\" + sharename + "\"\r\n");
	}
	else
	{


		QString local = mapstring.trimmed();
		QRegExp regex("^\\s*([a-zA-Z*]*:)\\s*\\((.*)\\)\\s*\\((.*)\\)\\s*\\((.*)\\)\\s*$");
		regex.exactMatch(local);

		if (regex.captureCount() != 4)
		{
			qWarning() << "Malformed map string!" << regex.captureCount() << regex.capturedTexts() ;
			return("");
		}

		QString drivestr;
		QStringList users, groups, computers;

		drivestr = regex.cap(1);
		users = parseList(regex.cap(2));
		groups = parseList(regex.cap(3));
		computers = parseList(regex.cap(4));

		QString text;

		foreach(QString tmp, users)
		{
			text += QString("mapIfUser \"%1\", \"%2\", \"\\\\%3\\%4\"\r\n")
			        .arg(tmp).arg(drivestr).arg(hostname).arg(sharename);
		}

		foreach(QString tmp, groups)
		{
			text += QString("mapIfMember \"%1\", \"%2\", \"\\\\%3\\%4\"\r\n")
			        .arg(tmp).arg(drivestr).arg(hostname).arg(sharename);
		}

		foreach(QString tmp, computers)
		{
			if(tmp.endsWith("$"))
				tmp = tmp.left(tmp.size() -1);
			text += QString("mapIfComputer \"%1\", \"%2\", \"\\\\%3\\%4\"\r\n")
			        .arg(tmp).arg(drivestr).arg(hostname).arg(sharename);
		}

		return(text);
	}
}

void RdsShareManagerPrivate::updateIniFile(QString path)
{
	QHash<QString, QList<QPair<QString, QString> > > ini;
	QString section;
	QStringList cmds;
	QStringList args;
	QList<QPair<QString, QString> > others;

	QFile file(path);
	QTextStream stream(&file);
	if (!file.open(QFile::ReadOnly))
	{
		qWarning() << "Failed to open INI file for reading:" << path;
	}
	else
	{
		QString line;

		while ((line = stream.readLine()) != QString::Null())
		{
			if (line == "") continue;

			line = line.trimmed();
			if (line.startsWith("["))
			{
				section = line;
				section = section.replace("[", "");
				section = section.replace("]", "");
			}
			else
			{
				int index = line.indexOf("=");
				if (index == -1) continue;
				ini[section] << QPair<QString, QString>(line.left(index), line.mid(index + 1));
			}
		}

		file.close();
	}

	if (ini.contains("Logon"))
	{
		QList<QPair<QString, QString> > list = ini["Logon"];
		QPair<QString, QString> pair;
		foreach(pair, list)
		{
			if (pair.first.endsWith("CmdLine"))
			{
				cmds << pair.second;
			}
			else if (pair.first.endsWith("Parameters"))
			{
				args << pair.second;
			}
			else
			{
				others << pair;
			}
		}
	}

	if (!cmds.contains("launchapp.vbs"))
	{
		cmds << "launchapp.vbs";
		args << "";
	}

	QList<QPair<QString, QString> > list;
	for (int i = 0; i < cmds.size(); i++)
	{
		QPair<QString, QString> pair;
		pair.first = QString("%1CmdLine").arg(i);
		pair.second = cmds[i];
		list << pair;
		pair.first = QString("%1Parameters").arg(i);
		if (i < args.size())
		{
			pair.second = args[i];
		}
		else
		{
			pair.second = "";
		}
		list << pair;
	}

	list << others;
	ini["Logon"] = list;

	if (!file.open(QFile::WriteOnly | QFile::Truncate))
	{
		qWarning() << "Failed to open INI file for writing:" << path;
	}

	stream.setCodec("UTF-16");
	stream.setGenerateByteOrderMark(true);
	foreach(QString key, ini.keys())
	{
		stream << "\r\n[" + key + "]\r\n";
		QPair<QString, QString> pair;
		foreach(pair, ini[key])
		{
			stream << pair.first + "=" + pair.second + "\r\n";
		}
	}

	file.close();
}

void RdsShareManagerPrivate::connectVolumeEvents(RdsShareManager *mgr)
{
	QMutexLocker lock(&volmutex);
	if (volmgr == NULL)
	{
		updateVolumePaths();
		volmgr = new RdsVolumeManager();
		volmgr->init();
		QObject::connect(mgr, SIGNAL(destroyed()), volmgr, SLOT(deleteLater()), Qt::QueuedConnection);
		QObject::connect(volmgr, SIGNAL(entityAdded(QString)), mgr, SLOT(volumeAdded(QString)), Qt::QueuedConnection);
		QObject::connect(volmgr, SIGNAL(entityRemoved(QString)), mgr, SLOT(volumeRemoved(QString)), Qt::QueuedConnection);
		QObject::connect(volmgr, SIGNAL(entityUpdated(QString)), mgr, SLOT(volumeUpdated(QString)), Qt::QueuedConnection);
	}
}

void RdsShareManager::volumeAdded(QString id)
{
	Q_UNUSED(id);
	QMutexLocker lock(&RdsShareManagerPrivate::volmutex);
	RdsShareManagerPrivate::updateVolumePaths();
}

void RdsShareManager::volumeRemoved(QString id)
{
	Q_UNUSED(id);
	QMutexLocker lock(&RdsShareManagerPrivate::volmutex);
	RdsShareManagerPrivate::updateVolumePaths();
}

void RdsShareManager::volumeUpdated(QString id)
{
	if (!RdsShareManagerPrivate::volumepaths.contains(id))
	{
		//qDebug() << "Ignoring update of volume:" << id;
		return;
	}

	QString oldpath = RdsShareManagerPrivate::volumepaths[id];

	ReturnValue ret = RdsShareManagerPrivate::volmgr->volume(id);
	if (ret.isError())
	{
		qWarning() << "Failed to get volume:" << ret << id;
		return;
	}

	RdsVolume volume;
	volume = ret;

	ret = volume.path();
	if (ret.isError())
	{
		qWarning() << "Failed to get volume mount point:" << ret << id;
		return;
	}

	QString path = ret.toString();
	if (path != oldpath)
	{
		//A volume was renamed
		RdsShareManagerPrivate::volumepaths[id] = path;

		bool shouldsave = false;

		foreach(QString shareid, listShares().toStringList())
		{
			ret = share(shareid);
			if (ret.isError())
			{
				qWarning() << "Failed to get share:" << ret << shareid;
				return;
			}

			RdsShare share;
			share = ret;

			ret = share.path();
			if (ret.isError())
			{
				qWarning() << "Failed to get share path:" << ret << shareid;
				return;
			}

			QString sharepath = ret.toString();
			if (sharepath.startsWith(oldpath))
			{
				sharepath = path + sharepath.mid(oldpath.size());

				//qDebug() << "Updating Share:" << shareid << path << oldpath << sharepath;

				ret = share.setPath(sharepath);
				if (ret.isError())
				{
					qWarning() << "Failed to set share path:" << shareid << ret;
					continue;
				}
				updateEntity("@SHARE/" + shareid + "/");
				shouldsave = true;
			}
		}

		if (shouldsave)
		{
			ret = save();
			if (ret.isError())
			{
				qWarning() << "Failed to save shares:" << ret;
				return;
			}

			RdsDaemonManager dmgr;

			ret = dmgr.reloadService("Samba");
			if (ret.isError())
			{
				qWarning() << "Failed to save smb.conf:" << ret;
				return;
			}
		}
	}
}

void RdsShareManagerPrivate::updateVolumePaths()
{
	volumepaths.clear();

	foreach(QString id, volmgr->list().toStringList())
	{
		ReturnValue ret = volmgr->volume(id);
		if (ret.isError())
		{
			qWarning() << "Failed to get volume:" << ret << id;
			continue;
		}

		RdsVolume volume;
		volume = ret;

		ret = volume.path();
		if (ret.isError())
		{
			qWarning() << "Failed to get volume mount point:" << ret << id;
			continue;
		}

		volumepaths[id] = ret.toString();
	}
}
