/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU Lesser General Public License as         *
 *  published by the Free Software Foundation; either version 2 of the     *
 *  License, or (at your option) any later version.                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  Lesser General Public License for more details.                        *
 *                                                                         *
 *  You should have received a copy of the GNU Lesser General Public       *
 *  License along with this program; if not, write to the                  *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/
#ifndef RDSCOMPUTER_H
#define RDSCOMPUTER_H

#include <RdsAdObject>
#include <QxtPimpl>
#include <rdsglobal.h>

class RdsComputerPrivate;

/**
	@author Brendan Powers <brendan@resara.com>
*/
class RDS_SHARED_EXPORT RdsComputer : public RdsAdObject
{
	QXT_DECLARE_PRIVATE(RdsComputer);
public:
	RdsComputer();
	RdsComputer(const RdsComputer &computer);
	RdsComputer(const QString &dn);
	~RdsComputer();

	/**
	 *         Gets the computer's primary group
	 * @return QString with the primary group name on success, error on failure
	 */
	ReturnValue primaryGroup() const;
	/**
	 *        Sets the computers primary group
	 * @param dn the dn of the primary group
	 * @return true on success, error on failure
	 */
	ReturnValue setPrimaryGroup(const QString &dn);
	/**
	 *         Gets the common name
	 * @return QString containing the common name
	 */
	ReturnValue cn() const;
	/**
	 *        sets the common name
	 * @param cn the common name
	 * @return QString containing the new dn on success, error on failure
	 */
	ReturnValue setCn(const QString &cn);
	/**
	 *        Gets a list of groups the computer is a member of
	 * @return QStringList containing the primary group followed by any other groups it is a member of on success, error on failure
	 */
	ReturnValue groups() const;
	/**
	 *        Adds this computer to a group
	 * @param groupdn the dn of the group to join
	 * @return true on success, error on failure.
	 */
	ReturnValue joinGroup(const QString &groupdn);
	/**
	 *        Removes this computer from a group
	 * @param groupdn the dn of the group to leave
	 * @return true on success, error on failure
	 */
	ReturnValue leaveGroup(const QString &groupdn);

	/**
	 *        Gets the description from the Ldap Object's "description" attribute
	 * @return QString containing the description on success, error on failure.
	 */
	ReturnValue description() const;
	/**
	 *        Sets the Description for the Ldap Object's "description" attribute.
	 * @param desc the description
	 * @return true on success, error on failure
	 */
	ReturnValue setDescription(const QString &desc);
	/**
	 *        Gets the notes stored in the "info" attribute of the RdsComputer's ldap object.
	 * @return QString containing the notes on success, error on failure.
	 */
	ReturnValue notes() const;
	/**
	 *        Sets the "info" attribute of the RdsComuter's ldap object. Passing an empty QString will remove the attribute.
	 * @param notes the notes to store in the ldap object
	 * @return true on success, error on failure
	 */
	ReturnValue setNotes(const QString &notes);
	/**
	 *        Gets the value stored in the "location" attribute of the RdsComputer's ldap object
	 * @return QString containing the location on success, error on failure.
	 */
	ReturnValue location() const;
	/**
	 *        Sets the "location" attribute of the RdsComputer's ldap object. Passing an empty QString will remove the attribute.
	 * @param location the location of the computer
	 * @return true on success, error on failure.
	 */
	ReturnValue setLocation(const QString &location);
	/**
	 *        Gets the Host Name for the RdsComputer
	 * @return QString containing the Host Name on success, error on failure.
	 */
	ReturnValue hostName() const;
	/**
	 *        Sets the Host Name for the RdsComputer
	 * @param hostname the Host Name
	 * @return true on success, error on failure.
	 */
	ReturnValue setHostName(const QString &hostname);
	/**
	 *        Gets the RdsComputer's "sAMAccountName" attribute
	 * @return QString containing the name on success, error on failure
	 */
	ReturnValue computerName() const;
	/**
	 *        Sets the RdsComputer's "sAMAccountName" attribute
	 * @param computer the computer name to set
	 * @return true on success, error on failure
	 */
	ReturnValue setComputerName(const QString &computer);
	/**
	 *        Gets the Operating System of the computer
	 * @return QString containing the OS name on success, error on failure.
	 */
	ReturnValue operatingSystem() const;
	/**
	 *        Sets the name of the Operating System on the computer
	 * @param os the name of the OS
	 * @return true on success, error on failure
	 */
	ReturnValue setOperatingSystem(const QString &os);
	/**
	 *        Gets the Version of the OS on the computer
	 * @return QString containing the OS version on success, error on failure
	 */
	ReturnValue operatingSystemVersion() const;
	/**
	 *        Sets the OS version for the computer
	 * @param version 
	 * @return true on success, error on failure
	 */
	ReturnValue setOperatingSystemVersion(const QString &version);
	/**
	 *        Gets the service pack number of the computer
	 * @return QString containing the service pack number
	 */
	ReturnValue servicePack() const;
	/**
	 *        Sets the service pack number of the computer
	 * @param servicepack 
	 * @return true on success, error on failure
	 */
	ReturnValue setServicePack(const QString &servicepack);
};

RDS_SHARED_EXPORT QDataStream& operator<<(QDataStream& d, const RdsComputer& object);
RDS_SHARED_EXPORT QDataStream& operator>>(QDataStream& d, RdsComputer& object);
Q_DECLARE_METATYPE(RdsComputer);

#endif
