/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU Lesser General Public License as         *
 *  published by the Free Software Foundation; either version 2 of the     *
 *  License, or (at your option) any later version.                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  Lesser General Public License for more details.                        *
 *                                                                         *
 *  You should have received a copy of the GNU Lesser General Public       *
 *  License along with this program; if not, write to the                  *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/
#include "rdsntacl.h"
#include "rdsntacl_p.h"
#include <QDebug>
#include <QProcess>
#include <QFile>
#include <ReturnValue>
#include <RdsSid>
#include <QDir>
#include <QFileInfo>
#include "rdsglobal.h"

#ifdef Q_OS_LINUX
#include <errno.h>
#include <sys/types.h>
#include <attr/xattr.h>
#endif

QTRPC_REGISTER_METATYPE(RdsNtAcl);

RdsNtAcl::RdsNtAcl()
		: RdsNtSecurityDescriptor()
{
	qxt_d().data = new RdsNtAclData();
}

RdsNtAcl::RdsNtAcl(const char *str)
		: RdsNtSecurityDescriptor(str)
{
	qxt_d().data = new RdsNtAclData();
}

RdsNtAcl::RdsNtAcl(const QString &str)
		: RdsNtSecurityDescriptor(str)
{
	qxt_d().data = new RdsNtAclData();
}

RdsNtAcl::RdsNtAcl(const QByteArray &ar)
		: RdsNtSecurityDescriptor(ar)
{
	qxt_d().data = new RdsNtAclData();
}

RdsNtAcl::RdsNtAcl(const RdsNtAcl &other)
		: RdsNtSecurityDescriptor(other)
{
	qxt_d().data = other.qxt_d().data;
}

RdsNtAcl::~RdsNtAcl()
{
}

QByteArray RdsNtAcl::toBinary() const
{
	QByteArray ar;

	quint64 fileHeader = 0x00020000;
	fileHeader = fileHeader << 32;
	fileHeader |= 0x00010001;
	ar.append(QByteArray::fromRawData((char*)&fileHeader, sizeof(fileHeader)));
	ar.append(RdsNtSecurityDescriptor::toBinary(8));
	return ar;
}

QtRpc::ReturnValue RdsNtAcl::save(const QString& path, bool forceinherit) const
{
#ifdef Q_OS_LINUX
	if (!QFile::exists(path))
		return ReturnValue(1, "No such file or directory");
	QByteArray binary = toBinary();
	int ret = setxattr(qPrintable(path), "security.NTACL", binary.constData(), binary.count(), 0);
	if (ret < 0)
		return ReturnValue(errno, strerror(errno));

	propagateInheritedPermissions(path, forceinherit);
	return true;
#else
	return ReturnValue(1, "You cannot set NTACLs in Windows");
#endif
}

QString RdsNtAcl::getNtAclString(const QString &path)
{
	if (!QFile::exists(path))
		return QString();
	return RdsNtAcl::fromFile(path).toString();
}

ReturnValue RdsNtAcl::fromFile(const QString &path)
{
#ifdef Q_OS_LINUX
	if (!QFile::exists(path))
		return ReturnValue(2, "file does not exist: " + path);

	QVector<char> vect(0);
	int ret = 0;
	do
	{
		vect.resize(vect.size() + 255);
		ret = getxattr(qPrintable(path), "security.NTACL", vect.data(), vect.size());
	}
	while (ret == -1 && errno == ERANGE);

	if (ret < 0)
	{
		return(ReturnValue(errno, strerror(errno)));
	}

	return QVariant::fromValue<RdsNtAcl>(RdsNtAcl(QByteArray::fromRawData(vect.data(), ret)));
#else
	return ReturnValue(1, "You cannot get NTACLs in Windows");
#endif
}

ReturnValue RdsNtAcl::propagateInheritedPermissions(const QString &path, bool forceinherit)
{
#ifdef Q_OS_LINUX
	QFileInfo info(path);
	if (info.isFile())
	{
		//qDebug() << path << "is a file";
		return(true);
	}

	//qDebug() << "updating permissions for" << path << "do to inheritence";

	ReturnValue ret = RdsNtAcl::fromFile(path);
	if (ret.isError()) return(ret);
	RdsNtAcl parentacl = ret.value<RdsNtAcl>();

	QDir dir(path);
	QStringList files = dir.entryList(QStringList());

	foreach(QString file, files)
	{
		if ((file == ".") || (file == "..")) continue;

		ret = RdsNtAcl::fromFile(path + "/" + file);
		if (ret.isError() && ret.errNumber() != ENOATTR)
		{
			qWarning() << "Failed to update ACLs for " + path + "/" + file + ":" << ret;
			continue;
		}
		RdsNtAcl acl;
		if (!ret.isError())
		{
			//qDebug() << "Reading Permissions:" << path + "/" + file;
			acl = ret.value<RdsNtAcl>();

			if (!forceinherit && acl.daclFlags().testFlag(RdsNtAcl::Protected))
				continue;

			//Remove existing inherited entries, or all entries if we are forcing inheritence
			foreach(RdsAce ace, acl.dacl())
			{
				if (((ace.flags() & RdsAce::Inherited) == RdsAce::Inherited) || forceinherit)
				{
					acl.dacl().removeAll(ace);
					//qDebug() << "Removing ACE... " << acl.dacl().size() << "ACE's remaining.";
				}
			}
		}

		info = QFileInfo(path + "/" + file);
		if (info.isFile())
		{
			//Add ACEs from the parent that have the FileInherit flag
			foreach(RdsAce ace, parentacl.dacl())
			{
				if ((ace.flags() & RdsAce::FileInherit) == RdsAce::FileInherit)
				{
					ace.setFlags(ace.flags() | RdsAce::Inherited);
					acl.dacl() << ace;
				}
			}
		}
		else
		{
			//Add ACEs from the parent that have the FolderInherit flag
			foreach(RdsAce ace, parentacl.dacl())
			{
				if ((ace.flags() & RdsAce::FolderInherit) == RdsAce::FolderInherit)
				{
					ace.setFlags(ace.flags() | RdsAce::Inherited);
					acl.dacl() << ace;
				}
			}
		}

		if (forceinherit)
		{
			acl.setDaclFlags(acl.daclFlags() & !Protected);
		}

		//Write out permissions
		QByteArray binary = acl.toBinary();
		int ret = setxattr(qPrintable(path + "/" + file), "security.NTACL", binary.constData(), binary.count(), 0);
		if (ret < 0)
		{
			qWarning() << "Failed to save ACLs for " + path + "/" + file + ":" << ReturnValue(errno, strerror(errno));
			continue;
		}

		if (info.isDir())
			propagateInheritedPermissions(path + "/" + file);
	}

	return(true);
#else
	return ReturnValue(1, "You cannot propagate permissions on Windows");
#endif
}

ReturnValue RdsNtAcl::updateInheritedPermissions(const QString &path, bool forceinherit)
{
#ifdef Q_OS_LINUX
	QDir dir(path), parent(path);
	bool isfile = QFileInfo(path).isFile();

	//If we can't get the parent dir, then there's nothing to do
	if (!parent.cdUp())
		return(2, "Failed to find a parent");

	ReturnValue ret;
	ret = fromFile(parent.path());
	if (ret.isError())
		return ret;

	RdsNtAcl perm;
	RdsNtAcl parentperm = ret.value<RdsNtAcl>();

	ret = fromFile(path);
	if (ret.isError()) //if the path has no ACL set, copy the parent ACL but remove permissions
	{
		perm = parentperm;
		perm.dacl().clear();
	}
	else
	{
		perm = ret.value<RdsNtAcl>();
	}

	//Remove existing inherited entries, or all entries if we are forcing inheritence
	foreach(RdsAce ace, perm.dacl())
	{
		if (((ace.flags() & RdsAce::Inherited) == RdsAce::Inherited) || forceinherit)
		{
			perm.dacl().removeAll(ace);
		}
	}

	if (isfile)
	{
		//Add ACEs from the parent that have the FileInherit flag
		foreach(RdsAce ace, parentperm.dacl())
		{
			if ((ace.flags() & RdsAce::FileInherit) == RdsAce::FileInherit)
			{
				ace.setFlags(ace.flags() | RdsAce::Inherited);
				perm.dacl() << ace;
			}
		}
	}
	else
	{
		//Add ACEs from the parent that have the FolderInherit flag
		foreach(RdsAce ace, parentperm.dacl())
		{
			if ((ace.flags() & RdsAce::FolderInherit) == RdsAce::FolderInherit)
			{
				ace.setFlags(ace.flags() | RdsAce::Inherited);
				perm.dacl() << ace;
			}
		}
	}

	if (forceinherit)
	{
		perm.setDaclFlags(perm.daclFlags() & !Protected);
	}

	//Write out permissions
	QByteArray binary = perm.toBinary();
	int result = setxattr(qPrintable(path), "security.NTACL", binary.constData(), binary.count(), 0);
	if (result < 0)
	{
		qWarning() << "Failed to save ACLs for " + path + ":" << ReturnValue(errno, strerror(errno));
		return(ReturnValue(errno, strerror(errno)));
	}

	if (!isfile)
		propagateInheritedPermissions(path);

	return(true);
#else
	return ReturnValue(1, "You cannot update permissions on Windows");
#endif
}

ReturnValue RdsNtAcl::getInheritedPermissions(const QString &path)
{
	QDir dir(path);
	QList<RdsNtAcl> permissions;
	ReturnValue ret;
	//bool isfile = QFileInfo(path).isFile();

	while (dir.cdUp())
	{
		ret = fromFile(dir.path());
		if (ret.isError()) continue;
		RdsNtAcl acl = ret.value<RdsNtAcl>();
		//qDebug() << "Loading:" << dir.path() << acl.daclFlags() <<  acl.dacl().size() << acl.sacl().size();
		foreach(RdsAce ace, acl.dacl())
		{
			//qDebug() << "ACE:" << ace;

			QString flags;
			if ((ace.flags() & RdsAce::ObjectInherit) != 0) flags += " ObjectInherit";
			if ((ace.flags() & RdsAce::ContainerInherit) != 0) flags += " ContainerInherit";
			if ((ace.flags() & RdsAce::NoPropogate) != 0) flags += " NoPropogate";
			if ((ace.flags() & RdsAce::InheritOnly) != 0) flags += " InheritOnly";
			if ((ace.flags() & RdsAce::Inherited) != 0) flags += " Inherited";
			if ((ace.flags() & RdsAce::Success) != 0) flags += " Success";
			if ((ace.flags() & RdsAce::Failure) != 0) flags += " Failure";

			//qDebug() << "Flags:" << ace.flags() << flags << ace.sid();
		}
		permissions.prepend(acl);
		if ((acl.daclFlags() & AutoInherited) == 0) break; //if this acl doesn't inherit permissions from its parent break
	}

	return(true);
}

RdsNtAcl RdsNtAcl::operator=(const RdsNtAcl & other)
{
	qxt_d().data = other.qxt_d().data;
	RdsNtSecurityDescriptor::operator=(other);
	return *this;
}

QDebug operator<<(QDebug dbg, const RdsNtAcl& acl)
{
	dbg.nospace() << "Owner: " << acl.owner() << "\nGroup: " << acl.group();
	if (acl.dacl().count() > 0)
	{
		dbg << "\nDACL Permissions:\n{";
		foreach(RdsAce ace, acl.dacl())
		{
			dbg <<
			"\n	{" <<
			"\n		" << ace.toString() <<
			"\n	}";
		}
		dbg << "\n}";
	}
	if (acl.sacl().count() > 0)
	{
		dbg << "\nSACL Permissions:\n{";
		foreach(RdsAce ace, acl.sacl())
		{
			dbg <<
			"\n	{" <<
			"\n		" << ace.toString() <<
			"\n	}";
		}
		dbg << "\n}";
	}
	return dbg.space();
}

QDataStream& operator<<(QDataStream& d, const RdsNtAcl& acl)
{
	return operator<<(d, static_cast<const RdsNtSecurityDescriptor&>(acl));
}

QDataStream& operator>>(QDataStream& d, RdsNtAcl& acl)
{
	return operator>>(d, static_cast<RdsNtSecurityDescriptor&>(acl));
}


/*
RdsNtAcl::Inheritence RdsNtAclPrivate::stringToInheritence(const QString &str)
{
	RdsNtAcl::Inheritence i = RdsNtAcl::ThisFolder | RdsNtAcl::Inherited;
	int index = 0;
	while ((index + 1) < str.count())
	{
		QStringRef ref(&str, index, 2);
		if (ref == "OI")
			i |= RdsNtAcl::Files;
		if (ref == "CI")
			i |= RdsNtAcl::Folders;
		if (ref == "NP")
			i &= ~ RdsNtAcl::Inherited;
		if (ref == "IO")
			i &= ~ RdsNtAcl::ThisFolder;
		index += 2;
	}
	return i;
}

QString RdsNtAclPrivate::inheritenceToString(const RdsNtAcl::Inheritence &inheritence)
{
	QString str;
	if (inheritence&RdsNtAcl::Files)
		str += "OI";
	if (inheritence&RdsNtAcl::Folders)
		str += "CI";
	if (!(inheritence&RdsNtAcl::Inherited))
		str += "NP";
	if (!(inheritence&RdsNtAcl::ThisFolder))
		str += "IO";
	return str;
}

RdsNtAcl::PermissionFlags RdsNtAclPrivate::stringToPermissions(const QString &str)
{
	RdsNtAcl::PermissionFlags p;
	int index = 0;
	while ((index + 1) < str.count())
	{
		QStringRef ref(&str, index, 2);
		if (ref == "RP")
			p |= RdsNtAcl::WriteExtendedAttributes;
		if (ref == "WP")
			p |= RdsNtAcl::TraverseFolder;
		if (ref == "CR")
			p |= RdsNtAcl::WriteAttributes;
		if (ref == "CC")
			p |= RdsNtAcl::ListFolder;
		if (ref == "DC")
			p |= RdsNtAcl::CreateFiles;
		if (ref == "LC")
			p |= RdsNtAcl::CreateFolders;
		if (ref == "LO")
			p |= RdsNtAcl::ReadAttributes;
		if (ref == "RC")
			p |= RdsNtAcl::ReadPermissions;
		if (ref == "WO")
			p |= RdsNtAcl::TakeOwnership;
		if (ref == "WD")
			p |= RdsNtAcl::WritePermissions;
		if (ref == "SD")
			p |= RdsNtAcl::Delete;
		if (ref == "DT")
			p |= RdsNtAcl::DeleteFiles;
		if (ref == "SW")
			p |= RdsNtAcl::ReadExtendedAttributes;
		index += 2;
	}
	return p;
}

QString RdsNtAclPrivate::permissionsToString(const RdsNtAcl::PermissionFlags &permissions)
{
// 	RPWPCRCCDCLCLORCWOWDSDDTSW
	QString str;
	if (permissions & RdsNtAcl::WriteExtendedAttributes)
		str += "RP";
	if (permissions & RdsNtAcl::TraverseFolder)
		str += "WP";
	if (permissions & RdsNtAcl::WriteAttributes)
		str += "CR";
	if (permissions & RdsNtAcl::ListFolder)
		str += "CC";
	if (permissions & RdsNtAcl::CreateFiles)
		str += "DC";
	if (permissions & RdsNtAcl::CreateFolders)
		str += "LC";
	if (permissions & RdsNtAcl::ReadAttributes)
		str += "LO";
	if (permissions & RdsNtAcl::ReadPermissions)
		str += "RC";
	if (permissions & RdsNtAcl::TakeOwnership)
		str += "WO";
	if (permissions & RdsNtAcl::WritePermissions)
		str += "WD";
	if (permissions & RdsNtAcl::Delete)
		str += "SD";
	if (permissions & RdsNtAcl::DeleteFiles)
		str += "DT";
	if (permissions & RdsNtAcl::ReadExtendedAttributes)
		str += "SW";
	return str;
}

quint32 RdsNtAclPrivate::permissionsToBitmask(const RdsNtAcl::PermissionFlags &p)
{
	quint32 b = 0;
	if (p & RdsNtAcl::TraverseFolder)
		b |= 0x00000020;
	if (p & RdsNtAcl::ListFolder)
		b |= 0x00000001;
	if (p & RdsNtAcl::ReadAttributes)
		b |= 0x00000080;
	if (p & RdsNtAcl::ReadExtendedAttributes)
		b |= 0x00000008;
	if (p & RdsNtAcl::CreateFiles)
		b |= 0x00000002;
	if (p & RdsNtAcl::CreateFolders)
		b |= 0x00000004;
	if (p & RdsNtAcl::WriteAttributes)
		b |= 0x00000100;
	if (p & RdsNtAcl::WriteExtendedAttributes)
		b |= 0x00000010;
	if (p & RdsNtAcl::DeleteFiles)
		b |= 0x00000040;
	if (p & RdsNtAcl::Delete)
		b |= 0x00010000;
	if (p & RdsNtAcl::ReadPermissions)
		b |= 0x00020000;
	if (p & RdsNtAcl::WritePermissions)
		b |= 0x00040000;
	if (p & RdsNtAcl::TakeOwnership)
		b |= 0x00080000;
	return b;
}

RdsNtAcl::PermissionFlags RdsNtAclPrivate::bitmaskToPermissions(quint32 b)
{
	RdsNtAcl::PermissionFlags p = 0;
	if (b & 0x00000020)
		p |= RdsNtAcl::TraverseFolder;
	if (b & 0x00000001)
		p |= RdsNtAcl::ListFolder;
	if (b & 0x00000080)
		p |= RdsNtAcl::ReadAttributes;
	if (b & 0x00000008)
		p |= RdsNtAcl::ReadExtendedAttributes;
	if (b & 0x00000002)
		p |= RdsNtAcl::CreateFiles;
	if (b & 0x00000004)
		p |= RdsNtAcl::CreateFolders;
	if (b & 0x00000100)
		p |= RdsNtAcl::WriteAttributes;
	if (b & 0x00000010)
		p |= RdsNtAcl::WriteExtendedAttributes;
	if (b & 0x00000040)
		p |= RdsNtAcl::DeleteFiles;
	if (b & 0x00010000)
		p |= RdsNtAcl::Delete;
	if (b & 0x00020000)
		p |= RdsNtAcl::ReadPermissions;
	if (b & 0x00040000)
		p |= RdsNtAcl::WritePermissions;
	if (b & 0x00080000)
		p |= RdsNtAcl::TakeOwnership;
	return p;
}
*/

QList<RdsAce> RdsNtAcl::sortDacl(QList<RdsAce> dacl) const
{
	QList<RdsAce> iallow, ideny, deny, allow, aces;

	foreach(RdsAce ace, dacl)
	{
		if (ace.type() == RdsAce::Allow)
		{
			if ((ace.flags() & RdsAce::Inherited) != 0)
			{
				iallow << ace;
			}
			else
			{
				allow << ace;
			}
		}
		else
		{
			if ((ace.flags() & RdsAce::Inherited) != 0)
			{
				ideny << ace;
			}
			else
			{
				deny << ace;
			}
		}
	}

	aces.append(deny);
	aces.append(allow);
	aces.append(ideny);
	aces.append(iallow);

	return(aces);
}

