/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU Lesser General Public License as         *
 *  published by the Free Software Foundation; either version 2 of the     *
 *  License, or (at your option) any later version.                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  Lesser General Public License for more details.                        *
 *                                                                         *
 *  You should have received a copy of the GNU Lesser General Public       *
 *  License along with this program; if not, write to the                  *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/
// #define NT_SEC_DEBUG
#include "rdsglobal.h"

#include "rdsntsecuritydescriptor.h"
#include "rdsntsecuritydescriptor_p.h"
#include <QString>
#include <QDebug>
#include <QtEndian>

RdsNtSecurityDescriptor::RdsNtSecurityDescriptor()
{
	qxt_d().data = new RdsNtSecurityDescriptorData();
}

RdsNtSecurityDescriptor::RdsNtSecurityDescriptor(const char *securityString)
{
	qxt_d().data = new RdsNtSecurityDescriptorData();
	qxt_d().parseString(securityString);
}

RdsNtSecurityDescriptor::RdsNtSecurityDescriptor(const QString &securityString)
{
	qxt_d().data = new RdsNtSecurityDescriptorData();
	qxt_d().parseString(securityString);
}

RdsNtSecurityDescriptor::RdsNtSecurityDescriptor(const QByteArray &securityString)
{
	qxt_d().data = new RdsNtSecurityDescriptorData();
	ReturnValue ret = qxt_d().parseByteArray(securityString);
	if (ret.isError())
		qWarning() << ret;
}

RdsNtSecurityDescriptor::RdsNtSecurityDescriptor(const RdsNtSecurityDescriptor &other)
{
	qxt_d().data = other.qxt_d().data;
}

RdsNtSecurityDescriptor::~RdsNtSecurityDescriptor()
{
}

RdsNtSecurityDescriptor& RdsNtSecurityDescriptor::operator=(const RdsNtSecurityDescriptor & other)
{
	qxt_d().data = other.qxt_d().data;
	return *this;
}

RdsNtSecurityDescriptor& RdsNtSecurityDescriptor::operator=(const QString & str)
{
	qxt_d().parseString(str);
	return *this;
}

RdsNtSecurityDescriptor& RdsNtSecurityDescriptor::operator=(const QByteArray & str)
{
	qxt_d().parseByteArray(str);
	return *this;
}

RdsNtSecurityDescriptor& RdsNtSecurityDescriptor::operator=(const char * str)
{
	qxt_d().parseString(str);
	return *this;
}

bool RdsNtSecurityDescriptor::operator==(const RdsNtSecurityDescriptor &other) const
{
	VERIFY_VALUES(owner);
	VERIFY_VALUES(group);
	VERIFY_VALUES(daclFlags);
	VERIFY_VALUES(saclFlags);
	VERIFY_VALUES(dacl);
	VERIFY_VALUES(sacl);
	return true;
}

bool RdsNtSecurityDescriptor::operator!=(const RdsNtSecurityDescriptor &other) const
{
	return !operator==(other);
}

bool RdsNtSecurityDescriptor::operator==(const QString &str) const
{
	return operator==(RdsNtSecurityDescriptor(str));
}

bool RdsNtSecurityDescriptor::operator!=(const QString &str) const
{
	return !operator==(str);
}

bool RdsNtSecurityDescriptor::operator==(const QByteArray &str) const
{
	return operator==(RdsNtSecurityDescriptor(str));
}

bool RdsNtSecurityDescriptor::operator!=(const QByteArray &str) const
{
	return !operator==(str);
}

bool RdsNtSecurityDescriptor::operator==(const char *str) const
{
	return operator==(RdsNtSecurityDescriptor(str));
}

bool RdsNtSecurityDescriptor::operator!=(const char *str) const
{
	return !operator==(str);
}

RdsSid RdsNtSecurityDescriptor::owner() const
{
	return qxt_d().data->owner;
}

void RdsNtSecurityDescriptor::setOwner(const RdsSid &owner)
{
	qxt_d().data->owner = owner;
}

RdsSid RdsNtSecurityDescriptor::group() const
{
	return qxt_d().data->group;
}

void RdsNtSecurityDescriptor::setGroup(const RdsSid &group)
{
	qxt_d().data->group = group;
}

RdsNtSecurityDescriptor::AclFlags RdsNtSecurityDescriptor::daclFlags() const
{
	return qxt_d().data->daclFlags;
}

void RdsNtSecurityDescriptor::setDaclFlags(RdsNtSecurityDescriptor::AclFlags daclFlags)
{
	qxt_d().data->daclFlags = daclFlags;
}

RdsNtSecurityDescriptor::AclFlags RdsNtSecurityDescriptor::saclFlags() const
{
	return qxt_d().data->saclFlags;
}

void RdsNtSecurityDescriptor::setSaclFlags(RdsNtSecurityDescriptor::AclFlags saclFlags)
{
	qxt_d().data->saclFlags = saclFlags;
}

const QList<RdsAce> &RdsNtSecurityDescriptor::dacl() const
{
	return qxt_d().data->dacl;
}

const QList<RdsAce> &RdsNtSecurityDescriptor::daclConst() const
{
	return qxt_d().data->dacl;
}

QList<RdsAce> &RdsNtSecurityDescriptor::dacl()
{
	return qxt_d().data->dacl;
}

void RdsNtSecurityDescriptor::setDacl(const QList<RdsAce> &dacl)
{
	qxt_d().data->dacl = dacl;
}

const QList<RdsAce> &RdsNtSecurityDescriptor::sacl() const
{
	return qxt_d().data->sacl;
}

const QList<RdsAce> &RdsNtSecurityDescriptor::saclConst() const
{
	return qxt_d().data->sacl;
}

QList<RdsAce> &RdsNtSecurityDescriptor::sacl()
{
	return qxt_d().data->sacl;
}

void RdsNtSecurityDescriptor::setSacl(const QList<RdsAce> &sacl)
{
	qxt_d().data->sacl = sacl;
}

QString RdsNtSecurityDescriptor::toString() const
{
	QString ret;
	ret += "O:";
	ret += qxt_d().data->owner.toShortString();
	ret += "G:";
	ret += qxt_d().data->group.toShortString();
	ret += "D:";
	ret += qxt_d().aclFlagsToString(qxt_d().data->daclFlags);
	foreach(RdsAce ace, qxt_d().data->dacl)
	{
		ret += ace.toString();
	}
	ret += "S:";
	ret += qxt_d().aclFlagsToString(qxt_d().data->saclFlags);
	foreach(RdsAce ace, qxt_d().data->sacl)
	{
		ret += ace.toString();
	}
	return ret;
}


QByteArray RdsNtSecurityDescriptor::toBinary(qint32 offset) const
{
	// Format:
	// uint16 revision 0-2
	// uint16 type 2-4
	// uint32/64 owner sid pointer 4-8
	// uint32/64 group sid pointer 8-12
	// uint32/64 sacl pointer 12-16
	// uint32/64 dacl pointer 16-20
	// *owner = RdsSid
	// *group = RdsSid
	// *sacl = ACL
	// *dacl = ACL
	// ACL Format
	// uint16/32 acl revision number
	// uint16 size
	// uint32 number of aces
	// number-of-aces number of RdsAce values

	QByteArray ar;

	QList<RdsAce> dacl = sortDacl(qxt_d().data->dacl);
	QList<RdsAce> sacl = sortSacl(qxt_d().data->sacl);

	quint16 type = qxt_d().data->flags;
	if (qxt_d().data->dacl.size() > 0) type |= DaclPresent;
	if (qxt_d().data->sacl.size() > 0) type |= SaclPresent;
	type |= SelfRelative;
	type |= qxt_d().data->daclFlags << 8;
	type |= qxt_d().data->saclFlags << 9;
	ar.append(QByteArray::fromHex("0100"));
	ar.append(QByteArray::fromRawData((char*)&type, sizeof(type)));
	ar.append(QByteArray::fromHex("00000000")); // owner
	ar.append(QByteArray::fromHex("00000000")); // group
	ar.append(QByteArray::fromHex("00000000")); // sacl
	ar.append(QByteArray::fromHex("00000000")); // dacl
	if (!qxt_d().data->owner.isNull())
	{
		quint32 count = ar.count() + offset;
		ar.replace(4, 4, QByteArray::fromRawData((char*)&count, sizeof(count)));
		ar.append(qxt_d().data->owner.toBinary());
	}
	if (!qxt_d().data->group.isNull())
	{
		quint32 count = ar.count() + offset;
		ar.replace(8, 4, QByteArray::fromRawData((char*)&count, sizeof(count)));
		ar.append(qxt_d().data->group.toBinary());
	}
	if (qxt_d().data->sacl.count() != 0)
	{
		quint32 count = ar.count() + offset;
		ar.replace(12, 4, QByteArray::fromRawData((char*)&count, sizeof(count)));
		quint32 start = ar.count();
		ar.append(QByteArray::fromHex("0400"));
		quint32 sizePos = ar.count();
		ar.append(QByteArray::fromHex("0000"));
		count = qxt_d().data->sacl.count();
		ar.append(QByteArray::fromRawData((char*)&count, sizeof(count)));
		for (int i = 0; i < sacl.count(); ++i)
		{
			ar.append(sacl.at(i).toBinary());
		}
		count = ar.count() - start;
		ar.replace(sizePos, 2, QByteArray::fromRawData((char*)&count, 2));
	}
	if (qxt_d().data->dacl.count() != 0)
	{
		quint32 count = ar.count() + offset;
		ar.replace(16, 4, QByteArray::fromRawData((char*)&count, sizeof(count)));
		quint32 start = ar.count();
		ar.append(QByteArray::fromHex("0400"));
		quint32 sizePos = ar.count();
		ar.append(QByteArray::fromHex("0000"));
		count = qxt_d().data->dacl.count();
		ar.append(QByteArray::fromRawData((char*)&count, sizeof(count)));
		for (int i = 0; i < dacl.count(); ++i)
		{
			ar.append(dacl.at(i).toBinary());
		}
		count = ar.count() - start;
		ar.replace(sizePos, 2, QByteArray::fromRawData((char*)&count, 2));
	}

	return ar;
}

ReturnValue RdsNtSecurityDescriptorPrivate::parseString(const QString &str)
{
	/*
	String format:

	"O:"
	SID Owner
	"G:"
	SID Group
	"D:"
	dacl_flags
	Array of RdsAces (dacl)
	"S:"
	sacl_flags
	Array of RdsAces (sacl)
	*/
	int index = 0;
	index = (str.indexOf(':', index) + 1); // start of element
	data->owner = str.mid(index, str.indexOf(':', index) - index - 1); // Owner SID

	index = (str.indexOf(':', index) + 1); // start of element
	data->group = str.mid(index, str.indexOf(':', index) - index - 1); // Group SID

	index = (str.indexOf(':', index) + 1); // start of element
	QString value = str.mid(index, str.indexOf('(', index) - index); // dacl_flags
	data->daclFlags = stringToAclFlags(value);

	//dacls
	data->dacl.clear();

	while ((str.indexOf(':', index) + 1) > (str.indexOf('(', index) + 1))
	{
		index = (str.indexOf('(', index)); // start of ace, including '('
		QString acl = str.mid(index, str.indexOf(')', index) - index + 1); // dacl_flags

		data->dacl.append(RdsAce(acl));

		index = (str.indexOf(')', index) + 1); // end of element
	}

	index = (str.indexOf(':', index) + 1); // start of element
	value = str.mid(index, str.indexOf('(', index) - index); // sacl_flags
	data->saclFlags = stringToAclFlags(value);

	//sacls
	data->sacl.clear();

	while (str.indexOf('(', index) != -1)
	{
		index = (str.indexOf('(', index)); // start of ace, including '('
		QString acl = str.mid(index, str.indexOf(')', index) - index + 1); // dacl_flags
		data->sacl.append(RdsAce(acl));

		index = (str.indexOf(')', index) + 1); // end of element
	}

	return true;
}

ReturnValue RdsNtSecurityDescriptorPrivate::parseByteArray(const QByteArray &data)
{
	// Format:
	// uint16 revision
	// uint16 type
	// uint32/64 owner sid pointer
	// uint32/64 group sid pointer
	// uint32/64 sacl pointer
	// uint32/64 dacl pointer
	// *owner = RdsSid
	// *group = RdsSid
	// *sacl = ACL
	// *dacl = ACL
	// ACL Format
	// uint16/32 acl revision number
	// uint16 size
	// uint32 number of aces
	// number-of-aces number of RdsAce values
	/* security_descriptor->type bits
	typedef [public,bitmap16bit] bitmap {
		SEC_DESC_OWNER_DEFAULTED        = 0x0001,
		SEC_DESC_GROUP_DEFAULTED        = 0x0002,
		SEC_DESC_DACL_PRESENT           = 0x0004,
		SEC_DESC_DACL_DEFAULTED         = 0x0008,
		SEC_DESC_SACL_PRESENT           = 0x0010,
		SEC_DESC_SACL_DEFAULTED         = 0x0020,
		SEC_DESC_DACL_TRUSTED           = 0x0040,
		SEC_DESC_SERVER_SECURITY        = 0x0080,
		SEC_DESC_DACL_AUTO_INHERIT_REQ  = 0x0100,
		SEC_DESC_SACL_AUTO_INHERIT_REQ  = 0x0200,
		SEC_DESC_DACL_AUTO_INHERITED    = 0x0400,
		SEC_DESC_SACL_AUTO_INHERITED    = 0x0800,
		SEC_DESC_DACL_PROTECTED         = 0x1000,
		SEC_DESC_SACL_PROTECTED         = 0x2000,
		SEC_DESC_RM_CONTROL_VALID       = 0x4000,
		SEC_DESC_SELF_RELATIVE          = 0x8000
	} security_descriptor_type;
	*/
	quint8 discard;

	this->data->sacl.clear();
	this->data->dacl.clear();
	const char* dataPtr = data.constData();
	const char* offset = dataPtr;

	quint64 fileHeader = 0x00020000;
	fileHeader = fileHeader << 32;
	fileHeader |= 0x00010001;
	if (*((const quint64*)data.constData()) == fileHeader)
		offset += 8;

	quint16 revision;
	READ_INT_VALUE(revision, offset);
	quint16 type;
	READ_INT_VALUE(type, offset);
	qint32 owner;
	READ_INT_VALUE(owner, offset);
	qint32 group;
	READ_INT_VALUE(group, offset);
	quint32 sacl;
	READ_INT_VALUE(sacl, offset);
	quint32 dacl;
	READ_INT_VALUE(dacl, offset);

	if (owner > data.count())
	{
		qCritical() << "Invalid nTSecurityDescriptor, owner pointer is too large:" << owner << data.count();
		return ReturnValue(1, QString("Invalid nTSecurityDescriptor, owner pointer is too large: %1").arg(owner));
	}
	if (group > data.count())
	{
		qCritical() << "Invalid nTSecurityDescriptor, group pointer is too large:" << group << data.count();
		return ReturnValue(1, QString("Invalid nTSecurityDescriptor, group pointer is too large: %1").arg(group));
	}
	this->data->flags = static_cast<RdsNtSecurityDescriptor::Flags>(type & 0xc0ff);
	if (owner)
		this->data->owner = RdsSid(QByteArray::fromRawData((data.constData() + owner), (data.count() - owner)));
	else
		this->data->owner = RdsSid();
	if (group)
		this->data->group = RdsSid(QByteArray::fromRawData((data.constData() + group), (data.count() - group)));
	else
		this->data->owner = RdsSid();

	// the flags they're anded with are all the possible values they could be, then we bitshift them down so that they're 1, 4, and 8 instead of the much larger versions they are by default
	// above you can see what the values are, and if you do out the math it works out right...
	// this checks the bitshift count, just to prove that it works correctly really quickly....
// 	qDebug() << "Sacl" << QString::number((0x2a00 >> 9), 16);
// 	qDebug() << "Dacl" << QString::number((0x1500 >> 8), 16);
	// if you or the flags together you'll see that they equal 0x2a00 and 0x1500
	// QED, BITCH!
	this->data->saclFlags = static_cast<RdsNtSecurityDescriptor::AclFlags>((type & 0x2a00) >> 9);
	this->data->daclFlags = static_cast<RdsNtSecurityDescriptor::AclFlags>((type & 0x1500) >> 8);

	// SACL
	if (sacl)
	{
		offset = dataPtr;
		offset += sacl;
		quint16 aclRevision;
		READ_INT_VALUE(aclRevision, offset);
		quint16 size;
		READ_INT_VALUE(size, offset);
		quint32 count;
		READ_INT_VALUE(count, offset);

		for (uint i = 0; i < count; ++i)
		{
			READ_INT_VALUE(discard, offset); //type
			READ_INT_VALUE(discard, offset); //flags
			quint16 size;
			READ_INT_VALUE(size, offset);
			offset -= 4;
			if ((offset - dataPtr) + size > data.count())
				return ReturnValue(1, "Parse error is ACL array: Size is larger than available data");
			this->data->sacl.append(RdsAce(QByteArray::fromRawData(offset, size)));
			offset += size;
			if (offset - dataPtr > data.count())
				return ReturnValue(1, "Parse error is ACL array: Size is larger than available data");
		}
	}

	// DACL
	if (dacl)
	{
		offset = dataPtr;
		offset += dacl;
		quint16 aclRevision;
		READ_INT_VALUE(aclRevision, offset);
		quint16 size;
		READ_INT_VALUE(size, offset);
		quint32 count;
		READ_INT_VALUE(count, offset);

		for (uint i = 0; i < count; ++i)
		{
			READ_INT_VALUE(discard, offset); //type
			READ_INT_VALUE(discard, offset); //flags
			quint16 size;
			READ_INT_VALUE(size, offset);
			offset -= 4;
			if ((offset - dataPtr) + size > data.count())
				return ReturnValue(1, "Parse error is ACL array: Size is larger than available data");
			this->data->dacl.append(RdsAce(QByteArray::fromRawData(offset, size)));
			offset += size;
			if (offset - dataPtr > data.count())
				return ReturnValue(1, "Parse error is ACL array: Size is larger than available data");
		}
	}

	return true;
}

/*
 * RdsAce
 */

RdsAce::RdsAce()
{
	QXT_INIT_PRIVATE(RdsAce);
	qxt_d().data = new RdsAceData();
}

RdsAce::RdsAce(const char *ace)
{
	QXT_INIT_PRIVATE(RdsAce);
	qxt_d().data = new RdsAceData();
	qxt_d().parseString(ace);
}

RdsAce::RdsAce(const QString &ace)
{
	QXT_INIT_PRIVATE(RdsAce);
	qxt_d().data = new RdsAceData();
	qxt_d().parseString(ace);
}

RdsAce::RdsAce(const QByteArray &ace)
{
	QXT_INIT_PRIVATE(RdsAce);
	qxt_d().data = new RdsAceData();
	ReturnValue ret = qxt_d().parseByteArray(ace);
	if (ret.isError())
		qWarning() << ret;
}

RdsAce::RdsAce(const RdsAce &other)
{
	QXT_INIT_PRIVATE(RdsAce);
	qxt_d().data = other.qxt_d().data;
}

RdsAce::~RdsAce()
{
}

RdsAce& RdsAce::operator=(const RdsAce & other)
{
	qxt_d().data = other.qxt_d().data;
	return *this;
}

RdsAce& RdsAce::operator=(const QString & str)
{
	qxt_d().parseString(str);
	return *this;
}

RdsAce& RdsAce::operator=(const QByteArray & str)
{
	qxt_d().parseByteArray(str);
	return *this;
}

RdsAce& RdsAce::operator=(const char * str)
{
	qxt_d().parseString(str);
	return *this;
}

bool RdsAce::operator==(const RdsAce &other) const
{
	VERIFY_VALUES(type);
	VERIFY_VALUES(flags);
	VERIFY_VALUES(access);
	VERIFY_VALUES(object);
	VERIFY_VALUES(inheritedObject);
	VERIFY_VALUES(sid);
	return true;
}

bool RdsAce::operator!=(const RdsAce &other) const
{
	return !operator==(other);
}

bool RdsAce::operator==(const QString &str) const
{
	return operator==(RdsAce(str));
}

bool RdsAce::operator!=(const QString &str) const
{
	return !operator==(str);
}

bool RdsAce::operator==(const QByteArray &str) const
{
	return operator==(RdsAce(str));
}

bool RdsAce::operator!=(const QByteArray &str) const
{
	return !operator==(str);
}

bool RdsAce::operator==(const char *str) const
{
	return operator==(RdsAce(str));
}

bool RdsAce::operator!=(const char *str) const
{
	return !operator==(str);
}

RdsAce::Type RdsAce::type() const
{
	return qxt_d().data->type;
}

void RdsAce::setType(Type type)
{
	qxt_d().data->type = type;
}

RdsAce::Flags RdsAce::flags() const
{
	return qxt_d().data->flags;
}

void RdsAce::setFlags(Flags flags)
{
	qxt_d().data->flags = flags;
}

RdsAce::AccessFlags RdsAce::access() const
{
	return static_cast<RdsAce::AccessFlags>(qxt_d().data->access);
}

void RdsAce::setAccess(AccessFlags access)
{
	qxt_d().data->access = (access &
	                        (All |
	                         Read |
	                         Write |
	                         Execute |
	                         StandardAll |
	                         ReadControl |
	                         Delete |
	                         WriteDac |
	                         WriteOwner |
	                         ReadProperty |
	                         WriteProperty |
	                         CreateChild |
	                         DeleteChild |
	                         ListChildren |
	                         SelfWrite |
	                         ListObject |
	                         DeleteTree |
	                         ControlAccess |
	                         FileAll |
	                         ReadFile |
	                         WriteFile |
	                         ExecuteFile |
	                         KeyAll |
	                         ReadKey |
	                         WriteKey |
	                         ExecuteKey |
	                         MLNoWrite |
	                         MLNoRead |
	                         MLNoExecute));
}

RdsGuid RdsAce::object() const
{
	return qxt_d().data->object;
}

void RdsAce::setObject(const RdsGuid &object)
{
	qxt_d().data->object = object;
}

RdsGuid RdsAce::inheritedObject() const
{
	return qxt_d().data->inheritedObject;
}

void RdsAce::setInheritedObject(const RdsGuid &inheritedObject)
{
	qxt_d().data->inheritedObject = inheritedObject;
}

RdsSid RdsAce::sid() const
{
	return qxt_d().data->sid;
}

void RdsAce::setSid(const RdsSid &sid)
{
	qxt_d().data->sid = sid;
}

QString RdsAce::toString() const
{
	return
	    '(' +
	    RdsAcePrivate::typeToString(qxt_d().data->type) +
	    ';' +
	    RdsAcePrivate::flagsToString(qxt_d().data->flags) +
	    ';' +
	    RdsAcePrivate::accessToString(static_cast<RdsAce::AccessFlags>(qxt_d().data->access)) +
	    ';' +
	    qxt_d().data->object.toString() +
	    ';' +
	    qxt_d().data->inheritedObject.toString() +
	    ';' +
	    qxt_d().data->sid.toShortString() +
	    ')';
}

QByteArray RdsAce::toBinary() const
{
// 	Format:
// 	uint8 Type
// 	uint8 Flags
// 	uint16 Size
// 	uint32 Access Mask
// 	if (Type > 4) read AceObject
// 	ndr_pull_dom_sid(ndr, NDR_SCALARS, &r->trustee)
	//
// 	AceObject Format:
// 		uint32 objectFlags
// 		if(objectFlags & SEC_ACE_OBJECT_TYPE_PRESENT) ndr_pull_GUID(ndr, NDR_SCALARS, objectFlags)
// 		if(objectFlags & SEC_ACE_INHERITED_OBJECT_TYPE_PRESENT) ndr_pull_GUID(ndr, NDR_SCALARS, objectInherit);
	//
// 	GUID Format: (128 bit total)
//		uint32 time_low
// 		uint16 time_mid
// 		uint16 time_hi_and_version
// 		uint8[2] clock_seq
// 		uint8[6] node
	quint32 access = qxt_d().data->access;
	access |= 0x00100000;

	QByteArray ar;
	ar.append(QByteArray::fromRawData((char*)&qxt_d().data->type, sizeof(quint8)));
	ar.append(QByteArray::fromRawData((char*)&qxt_d().data->flags, sizeof(quint8)));
	ar.append(QByteArray::fromHex("0000")); // size
	ar.append(QByteArray::fromRawData((char*)&access, sizeof(quint32)));
	if (qxt_d().data->type > 4)
	{
		quint32 objectFlags = 0;
		QByteArray tmpar;

		if (!qxt_d().data->object.isNull())
		{
			tmpar.append(qxt_d().data->object.toBinary());
			objectFlags |= 0x1;
		}

		if (!qxt_d().data->inheritedObject.isNull())
		{
			tmpar.append(qxt_d().data->inheritedObject.toBinary());
			objectFlags |= 0x2;
		}

		ar.append(QByteArray::fromRawData((char*)&objectFlags, sizeof(quint32)));
		ar.append(tmpar);

	}
	ar.append(qxt_d().data->sid.toBinary());
	quint16 count = ar.count();
	ar.replace(2, 2, QByteArray::fromRawData((char*)&count, sizeof(count)));
	return ar;
}

ReturnValue RdsAcePrivate::parseString(const QString &str)
{
	/*
	String format:
	'('
	Type string
	';'
	string bitmask for inheritence
	';'
	either int or string bitmask for permissions
	';'
	object_guid
	';'
	inherit_object_guid
	';'
	SID user this applies to
	')'
	*/
	int index = 0;
	index = (str.indexOf('(', index) + 1); // start of element
	QString value = str.mid(index, str.indexOf(';', index) - index); // 1-2 character type definition
	data->type = stringToType(value); // convert the fetched string to a Type

	index = (str.indexOf(';', index) + 1); // next element
	value = str.mid(index, str.indexOf(';', index) - index); // bitmask of inheritence related values.
	data->flags = stringToFlags(value); // convert the fetched string to a Flags

	index = (str.indexOf(';', index) + 1); // next element
	value = str.mid(index, str.indexOf(';', index) - index); // bitmask of access values.
	data->access = stringToAccess(value); // convert the fetched string to a AccessFlags

	index = (str.indexOf(';', index) + 1); // next element
	data->object = str.mid(index, str.indexOf(';', index) - index); // get the object guid

	index = (str.indexOf(';', index) + 1); // next element
	data->inheritedObject = str.mid(index, str.indexOf(';', index) - index); // get the inherited object guid

	index = (str.indexOf(';', index) + 1); // next element
	data->sid = str.mid(index, str.indexOf(')', index) - index); // this one ends in a ) instead of a ;...

	return true;
}

ReturnValue RdsAcePrivate::parseByteArray(const QByteArray &data)
{
// 	Format:
// 	uint8 Type
// 	uint8 Flags
// 	uint16 Size
// 	uint32 Access Mask
// 	if (Type > 4) read AceObject
// 	ndr_pull_dom_sid(ndr, NDR_SCALARS, &r->trustee)
//
// 	AceObject Format:
// 		uint32 objectFlags
// 		if(objectFlags & SEC_ACE_OBJECT_TYPE_PRESENT) ndr_pull_GUID(ndr, NDR_SCALARS, objectFlags)
// 		if(objectFlags & SEC_ACE_INHERITED_OBJECT_TYPE_PRESENT) ndr_pull_GUID(ndr, NDR_SCALARS, objectInherit);
//
// 	GUID Format: (128 bit total)
//		uint32 time_low
// 		uint16 time_mid
// 		uint16 time_hi_and_version
// 		uint8[2] clock_seq
// 		uint8[6] node

	const char* dataPtr = data.constData();
	const char* offset = dataPtr;

	quint8 type;
	READ_INT_VALUE(type, offset);
	quint8 flags;
	READ_INT_VALUE(flags, offset);
	quint16 size;
	READ_INT_VALUE(size, offset);
	quint32 access;
	READ_INT_VALUE(access, offset);
	if (type > 4)
	{
		quint32 objectFlags;
		READ_INT_VALUE(objectFlags, offset);
		if (objectFlags & 0x1)
		{
			this->data->object = RdsGuid(QByteArray::fromRawData(offset, data.count() - (offset - dataPtr)));
			quint64 discard;
			READ_INT_VALUE(discard, offset);
			READ_INT_VALUE(discard, offset);
		}
		if (objectFlags & 0x2)
		{
			this->data->inheritedObject = RdsGuid(QByteArray::fromRawData(offset, data.count() - (offset - dataPtr)));
			quint64 discard;
			READ_INT_VALUE(discard, offset);
			READ_INT_VALUE(discard, offset);
		}
	}

	this->data->type = (RdsAce::Type)type;
	this->data->flags = (RdsAce::Flags)flags;
	if (this->data->flags == 0)
		this->data->flags = RdsAce::NoPropogate;
	this->data->access = access;
	this->data->sid = RdsSid(QByteArray::fromRawData(offset, data.count() - (offset - dataPtr)));
	return true;
}

/*
 * Converters
 */

QString RdsNtSecurityDescriptorPrivate::aclFlagsToString(RdsNtSecurityDescriptor::AclFlags flags)
{
	QString str;

	if (flags & RdsNtSecurityDescriptor::Protected)
		str += "P";
	if (flags & RdsNtSecurityDescriptor::AutoInheritReq)
		str += "AR";
	if (flags & RdsNtSecurityDescriptor::AutoInherited)
		str += "AI";
	return str;
}

RdsNtSecurityDescriptor::AclFlags RdsNtSecurityDescriptorPrivate::stringToAclFlags(const QString &str)
{
	RdsNtSecurityDescriptor::AclFlags ret;

	int index = 0;
	while ((index + 1) < str.count())
	{
		QStringRef section(&str, index, 2);
		index += 2;
		if (section == "P")
		{
			ret |= RdsNtSecurityDescriptor::Protected;
			continue;
		}
		if (section == "AR")
		{
			ret |= RdsNtSecurityDescriptor::AutoInheritReq;
			continue;
		}
		if (section == "AI")
		{
			ret |= RdsNtSecurityDescriptor::AutoInherited;
			continue;
		}
	}
	return ret;
}

RdsAce::Type RdsAcePrivate::stringToType(const QString &str)
{
	if (str == "A")
		return RdsAce::Allow;

	if (str == "D")
		return RdsAce::Deny;

	if (str == "OA")
		return RdsAce::ObjectAllow;

	if (str == "OD")
		return RdsAce::ObjectDeny;

	if (str == "AU")
		return RdsAce::Audit;

	if (str == "AL")
		return RdsAce::Alarm;

	if (str == "OU")
		return RdsAce::ObjectAudit;

	if (str == "OL")
		return RdsAce::ObjectAlarm;

	if (str == "ML")
		return RdsAce::MandatoryLabel;

	return RdsAce::Unknown;
}

QString RdsAcePrivate::typeToString(const RdsAce::Type &type)
{
	if (type == RdsAce::Allow)
		return "A";
	if (type == RdsAce::Deny)
		return "D";
	if (type == RdsAce::ObjectAllow)
		return "OA";
	if (type == RdsAce::ObjectDeny)
		return "OD";
	if (type == RdsAce::Audit)
		return "AU";
	if (type == RdsAce::Alarm)
		return "AL";
	if (type == RdsAce::ObjectAudit)
		return "OU";
	if (type == RdsAce::ObjectAlarm)
		return "OL";
	if (type == RdsAce::MandatoryLabel)
		return "ML";
	return "";
}

RdsAce::Flags RdsAcePrivate::stringToFlags(const QString &str)
{
	RdsAce::Flags ret;

	int index = 0;
	while ((index + 1) < str.count())
	{
		QStringRef section(&str, index, 2);
		index += 2;
		if (section == "CI")
		{
			ret |= RdsAce::ContainerInherit;
			continue;
		}
		if (section == "OI")
		{
			ret |= RdsAce::ObjectInherit;
			continue;
		}
		if (section == "NP")
		{
			ret |= RdsAce::NoPropogate;
			continue;
		}
		if (section == "IO")
		{
			ret |= RdsAce::InheritOnly;
			continue;
		}
		if (section == "ID")
		{
			ret |= RdsAce::Inherited;
			continue;
		}
		if (section == "SA")
		{
			ret |= RdsAce::Success;
			continue;
		}
		if (section == "FA")
		{
			ret |= RdsAce::Failure;
			continue;
		}
	}
	return ret;
}

QString RdsAcePrivate::flagsToString(const RdsAce::Flags &flags)
{
	QString str;

	if (flags & RdsAce::ContainerInherit)
		str += "CI";
	if (flags & RdsAce::ObjectInherit)
		str += "OI";
	if (flags & RdsAce::NoPropogate)
		str += "NP";
	if (flags & RdsAce::InheritOnly)
		str += "IO";
	if (flags & RdsAce::Inherited)
		str += "ID";
	if (flags & RdsAce::Success)
		str += "SA";
	if (flags & RdsAce::Failure)
		str += "FA";
	return str;
}

RdsAce::AccessFlags RdsAcePrivate::stringToAccess(const QString &str)
{
	RdsAce::AccessFlags ret;

	int index = 0;
	while ((index + 1) < str.count())
	{
		QStringRef section(&str, index, 2);
		index += 2;
// 		Generic access rights
		if (section == "GA")
		{
			ret |= RdsAce::All;
			continue;
		}
		if (section == "GR")
		{
			ret |= RdsAce::Read;
			continue;
		}
		if (section == "GW")
		{
			ret |= RdsAce::Write;
			continue;
		}
		if (section == "GX")
		{
			ret |= RdsAce::Execute;
			continue;
		}

// 		Standard access rights
		if (section == "RC")
		{
			ret |= RdsAce::ReadControl;
			continue;
		}
		if (section == "SD")
		{
			ret |= RdsAce::Delete;
			continue;
		}
		if (section == "WD")
		{
			ret |= RdsAce::WriteDac;
			continue;
		}
		if (section == "WO")
		{
			ret |= RdsAce::WriteOwner;
			continue;
		}

// 		Directory service object access rights
		if (section == "RP")
		{
			ret |= RdsAce::ReadProperty;
			continue;
		}
		if (section == "WP")
		{
			ret |= RdsAce::WriteProperty;
			continue;
		}
		if (section == "CC")
		{
			ret |= RdsAce::CreateChild;
			continue;
		}
		if (section == "DC")
		{
			ret |= RdsAce::DeleteChild;
			continue;
		}
		if (section == "LC")
		{
			ret |= RdsAce::ListChildren;
			continue;
		}
		if (section == "SW")
		{
			ret |= RdsAce::SelfWrite;
			continue;
		}
		if (section == "LO")
		{
			ret |= RdsAce::ListObject;
			continue;
		}
		if (section == "DT")
		{
			ret |= RdsAce::DeleteTree;
			continue;
		}
		if (section == "CR")
		{
			ret |= RdsAce::ControlAccess;
			continue;
		}

// 		File access rights
		if (section == "FA")
		{
			ret |= RdsAce::FileAll;
			continue;
		}
		if (section == "FR")
		{
			ret |= RdsAce::ReadFile;
			continue;
		}
		if (section == "FW")
		{
			ret |= RdsAce::WriteFile;
			continue;
		}
		if (section == "FX")
		{
			ret |= RdsAce::ExecuteFile;
			continue;
		}

// 		Registry key access rights
		if (section == "KA")
		{
			ret |= RdsAce::KeyAll;
			continue;
		}
		if (section == "KR")
		{
			ret |= RdsAce::ReadKey;
			continue;
		}
		if (section == "KW")
		{
			ret |= RdsAce::WriteKey;
			continue;
		}
		if (section == "KX")
		{
			ret |= RdsAce::ExecuteKey;
			continue;
		}

// 		Mandatory label rights
		if (section == "NR")
		{
			ret |= RdsAce::MLNoWrite;
			continue;
		}
		if (section == "NW")
		{
			ret |= RdsAce::MLNoRead;
			continue;
		}
		if (section == "NX")
		{
			ret |= RdsAce::MLNoExecute;
			continue;
		}
	}
	return ret;
}

QString RdsAcePrivate::accessToString(const RdsAce::AccessFlags &access)
{
	QString str;
// 		Generic access rights
	if (access & RdsAce::All)
		str += "GA";
	if (access & RdsAce::Read)
		str += "GR";
	if (access & RdsAce::Write)
		str += "GW";
	if (access & RdsAce::Execute)
		str += "GX";

// 		Standard access rights
	if (access & RdsAce::ReadControl)
		str += "RC";
	if (access & RdsAce::Delete)
		str += "SD";
	if (access & RdsAce::WriteDac)
		str += "WD";
	if (access & RdsAce::WriteOwner)
		str += "WO";

// 		Directory service object access rights
	if (access & RdsAce::ReadProperty)
		str += "RP";
	if (access & RdsAce::WriteProperty)
		str += "WP";
	if (access & RdsAce::CreateChild)
		str += "CC";
	if (access & RdsAce::DeleteChild)
		str += "DC";
	if (access & RdsAce::ListChildren)
		str += "LC";
	if (access & RdsAce::SelfWrite)
		str += "SW";
	if (access & RdsAce::ListObject)
		str += "LO";
	if (access & RdsAce::DeleteTree)
		str += "DT";
	if (access & RdsAce::ControlAccess)
		str += "CR";
	/*
	// 		File access rights
		if (access & RdsAce::FileAll)
			str += "FA";
		if (access & RdsAce::ReadFile)
			str += "FR";
		if (access & RdsAce::WriteFile)
			str += "FW";
		if (access & RdsAce::ExecuteFile)
			str += "FX";

	// 		Registry key access rights
		if (access & RdsAce::KeyAll)
			str += "KA";
		if (access & RdsAce::ReadKey)
			str += "KR";
		if (access & RdsAce::WriteKey)
			str += "KW";
		if (access & RdsAce::ExecuteKey)
			str += "KX";

	// 		Mandatory label rights
		if (access & RdsAce::MLNoWrite)
			str += "NR";
		if (access & RdsAce::MLNoRead)
			str += "NW";
		if (access & RdsAce::MLNoExecute)
			str += "NX";
	*/
	return str;
}

QDebug operator<<(QDebug dbg, const RdsNtSecurityDescriptor& desc)
{
	dbg.nospace() << "RdsNtSecurityDescriptor(" << desc.toString() << ")";
	return dbg.space();
}

QDebug operator<<(QDebug dbg, const RdsAce& ace)
{
	dbg.nospace() << "RdsAce(" << ace.toString() << ")";
	return dbg.space();
}

QDataStream& operator<<(QDataStream& d, const RdsNtSecurityDescriptor& desc)
{
	d << desc.qxt_d().data->flags;
	d << desc.qxt_d().data->owner;
	d << desc.qxt_d().data->group;
	d << desc.qxt_d().data->daclFlags;
	d << desc.qxt_d().data->saclFlags;
	d << desc.qxt_d().data->dacl;
	d << desc.qxt_d().data->sacl;
	return d;
}

QDataStream& operator>>(QDataStream& d, RdsNtSecurityDescriptor& desc)
{
	int flags = 0;
	d >> flags;
	desc.qxt_d().data->flags = (RdsNtSecurityDescriptor::Flags)flags;
	d >> desc.qxt_d().data->owner;
	d >> desc.qxt_d().data->group;
	d >> flags;
	desc.qxt_d().data->daclFlags = (RdsNtSecurityDescriptor::AclFlags)flags;
	d >> flags;
	desc.qxt_d().data->saclFlags = (RdsNtSecurityDescriptor::AclFlags)flags;
	d >> desc.qxt_d().data->dacl;
	d >> desc.qxt_d().data->sacl;
	return d;
}

QDataStream& operator<<(QDataStream& d, const RdsAce& ace)
{
	d << ace.qxt_d().data->type;
	d << ace.qxt_d().data->flags;
	d << ace.qxt_d().data->access;
	d << ace.qxt_d().data->object;
	d << ace.qxt_d().data->inheritedObject;
	d << ace.qxt_d().data->sid;
	return d;
}

QDataStream& operator>>(QDataStream& d, RdsAce& ace)
{
	int flags = 0;
	d >> flags;
	ace.qxt_d().data->type = (RdsAce::Type)flags;
	d >> flags;
	ace.qxt_d().data->flags = (RdsAce::Flags)flags;
	d >> ace.qxt_d().data->access;
	d >> ace.qxt_d().data->object;
	d >> ace.qxt_d().data->inheritedObject;
	d >> ace.qxt_d().data->sid;
	return d;
}

QList<RdsAce> RdsNtSecurityDescriptor::sortDacl(QList<RdsAce> dacl) const
{
	return(dacl);
}

QList<RdsAce> RdsNtSecurityDescriptor::sortSacl(QList<RdsAce> sacl) const
{
	return(sacl);
}



