/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU Lesser General Public License as         *
 *  published by the Free Software Foundation; either version 2 of the     *
 *  License, or (at your option) any later version.                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  Lesser General Public License for more details.                        *
 *                                                                         *
 *  You should have received a copy of the GNU Lesser General Public       *
 *  License along with this program; if not, write to the                  *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/
#include "rdsglobal.h"
#include "rdsschedule.h"
#include "rdsschedule_p.h"
#include <AutomaticMetatypeRegistry>
#include <QDebug>

QTRPC_REGISTER_METATYPE(RdsSchedule);

RdsSchedule RdsSchedule::daily()
{
	RdsSchedule sched;
	sched.setType(Daily);
	sched.setTime(QTime(0, 0, 0));
	return sched;
}

RdsSchedule RdsSchedule::weekly()
{
	RdsSchedule sched;
	sched.setType(Weekly);
	sched.setDay(6);
	sched.setTime(QTime(0, 0, 0));
	return sched;
}

RdsSchedule RdsSchedule::monthly()
{
	RdsSchedule sched;
	sched.setType(Monthly);
	sched.setDay(1);
	sched.setTime(QTime(0, 0, 0));
	return sched;
}

RdsSchedule::RdsSchedule()
{
}

RdsSchedule::RdsSchedule(const RdsSchedule& other)
{
	operator=(other);
}

RdsSchedule::~RdsSchedule()
{
}

RdsSchedule& RdsSchedule::operator=(const RdsSchedule & other)
{
	qxt_d().data = other.qxt_d().data;
	return *this;
}

RdsSchedule::Type RdsSchedule::type() const
{
	return qxt_d().data->type;
}

void RdsSchedule::setType(Type type)
{
	qxt_d().data->type = type;
}

QTime RdsSchedule::time() const
{
	return qxt_d().data->time;
}

void RdsSchedule::setTime(const QTime& time)
{
	qxt_d().data->time = time;
}

RdsSchedule::WeekMask RdsSchedule::weekMask() const
{
	return qxt_d().data->weekMask;
}

void RdsSchedule::setWeekMask(WeekMask weekMask)
{
	qxt_d().data->weekMask = weekMask;
}

int RdsSchedule::day() const
{
	return qxt_d().data->day;
}

void RdsSchedule::setDay(int day)
{
	qxt_d().data->day = day;
}

QDateTime RdsSchedule::nextEvent() const
{
	QDateTime next = QDateTime::currentDateTime();
	Qt::DayOfWeek qtoday = static_cast<Qt::DayOfWeek>(QDate::currentDate().dayOfWeek());
	Day today = qxt_d().convertDay(qtoday);
	switch (qxt_d().data->type)
	{
		case None:
			return QDateTime();
		case Daily:
		{
			if (qxt_d().data->weekMask == 0)
			{
				qCritical() << "Daily event with no week mask value set!";
				return QDateTime();
			}

			if (qxt_d().data->weekMask.testFlag(today) && QTime::currentTime() <= qxt_d().data->time)
			{
				// Today is in the host mask, and the event hasn't happened yet!
				next.setTime(qxt_d().data->time);
				return next;
			}

			for (next = QDateTime::currentDateTime().addDays(1); next.date().dayOfWeek() != qtoday; next = next.addDays(1))
			{
				Day day = qxt_d().convertDay(static_cast<Qt::DayOfWeek>(next.date().dayOfWeek()));
				if (qxt_d().data->weekMask.testFlag(day))
				{
					next.setTime(qxt_d().data->time);
					return next;
				}
			}

			if (qxt_d().data->weekMask.testFlag(today))
			{
				// It's happening today, but already happened so we need to skip ahead a week
				// Also, if this ever happens, then someone doesn't know how to use "Weekly"... :P
				next = QDateTime::currentDateTime().addDays(7);
				next.setTime(qxt_d().data->time);
				return next;
			}

			qCritical() << "Week mask is nonzero but still could not find an available day!";
			return QDateTime();
			break;
		}
		case Weekly:
		{
			int day = qxt_d().data->day % 8;
			if (day == 0)
			{
				qCritical() << "Weekly event with invalid day value set!";
			}
			for (next; true; next = next.addDays(1))
			{
				if (next.date().dayOfWeek() == day)
				{
					next.setTime(qxt_d().data->time);
					if (next > QDateTime::currentDateTime())
						return next;
				}
			}
			break;
		}
		case Monthly:
		{
			int day = qxt_d().data->day;
			if (day < -31 || day > 31 || day == 0)
			{
				qCritical() << "Monthly event with invalid day value set!";
				return QDateTime();
			}
			next = next.addDays(1 - (next.date().day()));
			if (day < 0)
			{
				day *= -1;
				for (; true; next = next.addMonths(1))
				{
					if (day > next.date().daysInMonth())
					{
						day = next.date().daysInMonth();
					}
					int dayNumber = next.date().daysInMonth() - day;
					next = next.addDays(dayNumber);
					next.setTime(qxt_d().data->time);
					if (next > QDateTime::currentDateTime())
						return next;
					next = next.addDays(-dayNumber);
				}
			}
			// unnecessary else statement but I dig clarity and I like the indentation to be the same between positive and negative :D
			else if (day > 0)
			{
				for (; true; next = next.addMonths(1))
				{
					if (day > next.date().daysInMonth())
					{
						day = next.date().daysInMonth();
					}
					int dayNumber = day - 1;
					next = next.addDays(dayNumber);
					next.setTime(qxt_d().data->time);
					if (next > QDateTime::currentDateTime())
						return next;
					next = next.addDays(-dayNumber);
				}
			}
			break;
		}
	}
	return QDateTime();
}

RdsSchedule::Day RdsSchedulePrivate::convertDay(Qt::DayOfWeek day)
{
	switch (day)
	{
		case Qt::Sunday:
			return RdsSchedule::Sunday;
		case Qt::Monday:
			return RdsSchedule::Monday;
		case Qt::Tuesday:
			return RdsSchedule::Tuesday;
		case Qt::Wednesday:
			return RdsSchedule::Wednesday;
		case Qt::Thursday:
			return RdsSchedule::Thursday;
		case Qt::Friday:
			return RdsSchedule::Friday;
		case Qt::Saturday:
			return RdsSchedule::Saturday;
	}
	return (RdsSchedule::Day)0;
}

Qt::DayOfWeek RdsSchedulePrivate::convertDay(RdsSchedule::Day day)
{
	switch (day)
	{
		case RdsSchedule::Sunday:
			return Qt::Sunday;
		case RdsSchedule::Monday:
			return Qt::Monday;
		case RdsSchedule::Tuesday:
			return Qt::Tuesday;
		case RdsSchedule::Wednesday:
			return Qt::Wednesday;
		case RdsSchedule::Thursday:
			return Qt::Thursday;
		case RdsSchedule::Friday:
			return Qt::Friday;
		case RdsSchedule::Saturday:
			return Qt::Saturday;
	}
	return (Qt::DayOfWeek)0;
}

QDataStream& operator<<(QDataStream& d, const RdsSchedule& object)
{
	d << (int)object.qxt_d().data->type;
	d << object.qxt_d().data->time;
	d << object.qxt_d().data->weekMask;
	d << object.qxt_d().data->day;
	return d;
}

QDataStream& operator>>(QDataStream& d, RdsSchedule& object)
{
	int type = 0;
	d >> type;
	object.qxt_d().data->type = (RdsSchedule::Type)type;
	d >> object.qxt_d().data->time;
	d >> type;
	object.qxt_d().data->weekMask = (RdsSchedule::WeekMask)type;
	d >> object.qxt_d().data->day;
	return d;
}


